# 機能設計書 63-例外型

## 概要

本ドキュメントは、Juliaの組み込み例外型（`ErrorException`, `ArgumentError`, `BoundsError`, `MethodError`, `DomainError` 等）の設計と実装を記述するものである。

### 本機能の処理概要

本機能は、Julia言語で使用される各種組み込み例外型を定義する。これらの型はエラーの種類を分類し、`catch` ブロックでの型ベースのフィルタリングを可能にする。

**業務上の目的・背景**：エラーの種類を構造的に分類することで、プログラムが発生したエラーの性質に応じて適切な処理を選択できるようにする。例えば、配列の範囲外アクセス（`BoundsError`）と引数エラー（`ArgumentError`）では対処方法が異なるため、型による分類が重要である。

**機能の利用シーン**：関数の引数検証（`ArgumentError`）、配列アクセス（`BoundsError`）、メソッドディスパッチ失敗（`MethodError`）、数学関数のドメインエラー（`DomainError`）、ファイルI/O（`SystemError`）、型変換エラー（`InexactError`）など、あらゆるエラー発生箇所で利用される。

**主要な処理内容**：
1. 基底型 `Exception` から派生した各種例外型の定義
2. 各例外型のコンストラクタとフィールド定義
3. `error()` / `systemerror()` / `@assert` 等のヘルパー関数・マクロ
4. `showerror` メソッドによる型別エラー表示（No.64 と連携）

**関連システム・外部連携**：Core ランタイムで定義される基本例外型、`base/error.jl` で定義されるユーティリティ。

**権限による制御**：なし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Juliaプロンプト（julia>） | 参照画面 | エラー発生時にREPLに例外型名とメッセージが表示される |

## 機能種別

型定義 / バリデーション

## 入力仕様

### 入力パラメータ

各例外型のコンストラクタ引数は型により異なる。主要なものを以下に示す。

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| msg | `AbstractString` | Yes（ErrorException） | エラーメッセージ | - |
| msg | `AbstractString` | Yes（ArgumentError） | 引数エラーメッセージ | - |
| a | `Any` | No（BoundsError） | アクセス対象オブジェクト | - |
| i | `Any` | No（BoundsError） | アクセスインデックス | - |
| f | `Any` | Yes（MethodError） | 呼び出し対象関数 | - |
| args | `Any` | Yes（MethodError） | 引数タプル | - |
| val | `Any` | Yes（DomainError） | ドメイン外の値 | - |
| prefix | `AbstractString` | Yes（SystemError） | システムエラーの説明 | - |

### 入力データソース

Juliaランタイム、ユーザーコード

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| exception | `<: Exception` | 各種例外型のインスタンス |

### 出力先

`throw()` により例外として送出される。

## 処理フロー

### 処理シーケンス

```
1. 例外型の選択
   └─ エラーの性質に応じた適切な例外型を選択
2. 例外オブジェクトの構築
   └─ コンストラクタにエラー情報を渡してインスタンス生成
3. 例外の送出
   └─ throw() で例外を送出
4. エラー表示
   └─ showerror() により型別のフォーマットで表示（No.64 参照）
```

### フローチャート

```mermaid
flowchart TD
    A[エラー検出] --> B{エラーの種類}
    B -->|引数不正| C[ArgumentError]
    B -->|範囲外| D[BoundsError]
    B -->|メソッド不在| E[MethodError]
    B -->|ドメイン外| F[DomainError]
    B -->|システムエラー| G[SystemError]
    B -->|型エラー| H[TypeError]
    B -->|その他| I[ErrorException]
    C --> J[throw]
    D --> J
    E --> J
    F --> J
    G --> J
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-63-01 | Exception 基底型 | すべての例外型は `Exception` を継承する | ユーザー定義例外を含む |
| BR-63-02 | error 関数 | `error(msg)` は `ErrorException(msg)` を throw する | `error()` 呼び出し時 |
| BR-63-03 | @assert マクロ | 条件が false の場合 `AssertionError` を throw する | `@assert` 使用時 |
| BR-63-04 | systemerror | errno を使って `SystemError` を構築する | システムコール失敗時 |

### 計算ロジック

該当なし。

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

本機能自体がエラー処理の基盤であるため、各例外型の発生条件を一覧化する。

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ErrorException | `error()` 関数呼び出し時 | エラーメッセージを確認 |
| - | ArgumentError | 関数引数が不正な場合 | 引数を修正 |
| - | BoundsError | 配列範囲外アクセス時 | インデックスを確認 |
| - | MethodError | メソッドが見つからない場合 | 引数の型を確認 |
| - | DomainError | 数学関数のドメイン外入力時 | 入力値を確認 |
| - | TypeError | 型が期待と異なる場合 | 型を確認 |
| - | SystemError | OSシステムコール失敗時 | errno を確認 |
| - | DivideError | 整数ゼロ除算時 | 除数を確認 |
| - | StackOverflowError | スタックオーバーフロー時 | 再帰を修正 |
| - | UndefRefError | 未定義参照アクセス時 | 初期化を確認 |
| - | KeyError | Dict のキー不在時 | キーを確認 |
| - | InexactError | 不正確な型変換時 | 変換元の値を確認 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

例外型のインスタンス生成は軽量であること（フィールドの格納のみ）。

## セキュリティ考慮事項

例外メッセージに機密情報を含めないよう注意が必要。

## 備考

- ユーザーは `struct MyError <: Exception` で独自例外型を定義可能
- `@assert` はデバッグ用であり、最適化レベルによっては無効化される可能性がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | boot.jl | `base/boot.jl` | Core で定義される基本例外型: `Exception`, `ErrorException`, `BoundsError`, `TypeError`, `MethodError`, `DomainError` 等 |
| 1-2 | error.jl | `base/error.jl` | `WindowsErrorInfo` 構造体（191-194行目）: Windows API エラー情報 |

**読解のコツ**: 例外型の多くは Core（`base/boot.jl`）で定義される。`base/error.jl` ではこれらの型を使用するヘルパー関数が定義される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | error.jl | `base/error.jl` | `error` 関数（33-57行目）: `ErrorException` を構築して throw する汎用関数 |
| 2-2 | error.jl | `base/error.jl` | `systemerror` 関数（181-188行目）: errno ベースの `SystemError` 構築 |
| 2-3 | error.jl | `base/error.jl` | `@assert` マクロ（209-246行目）: 条件チェックと `AssertionError` 送出 |

**主要処理フロー**:
1. **47-57行目**: `error(s...)` は文字列引数から `ErrorException` を構築して throw
2. **187行目**: `systemerror(p, b::Bool)` は条件付きで `SystemError` を throw
3. **232-246行目**: `@assert` は条件式を評価し、false なら `AssertionError` を throw

#### Step 3: Windows 固有エラーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | error.jl | `base/error.jl` | `windowserror` 関数（195-203行目）: `GetLastError` ベースの Windows エラー処理 |

### プログラム呼び出し階層図

```
error(msg)
    └─ ErrorException(msg) → throw()

systemerror(prefix, errno)
    └─ SystemError(prefix, errno) → throw()

windowserror(prefix, code)
    └─ SystemError(prefix, 0, WindowsErrorInfo(code)) → throw()

@assert(cond, msg)
    └─ AssertionError(msg) → throw()
```

### データフロー図

```
[入力]                  [処理]                        [出力]

エラー情報 ──────▶ 例外型コンストラクタ ──────▶ 例外オブジェクト
  - メッセージ           │                           │
  - 値                  │                           ▼
  - errno               │                      throw()
  - 関数/引数           ▼                           │
                   型別フォーマット                   ▼
                   (showerror)              catch ブロック
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| boot.jl | `base/boot.jl` | ソース | Core 例外型の定義 |
| error.jl | `base/error.jl` | ソース | エラー関連ヘルパー関数・マクロ |
| errorshow.jl | `base/errorshow.jl` | ソース | 例外型別の表示実装（No.64） |
| julia.h | `src/julia.h` | ヘッダ | ランタイムレベルの例外型定義 |
