# 機能設計書 64-エラー表示

## 概要

本ドキュメントは、Juliaにおけるエラー情報のフォーマットと表示機能（`showerror` / `stacktrace` / `backtrace`）の設計と実装を記述するものである。

### 本機能の処理概要

本機能は、例外発生時のエラーメッセージのフォーマットとスタックトレースの整形表示を提供する。開発者がエラーの原因を迅速に特定できるよう、構造化されたエラー情報を出力する。

**業務上の目的・背景**：エラーが発生した際に、エラーの種類・メッセージ・発生箇所（ファイル名、行番号）・呼び出し履歴を分かりやすく表示することは、デバッグ作業の効率化に不可欠である。特にJuliaのメソッドディスパッチシステムでは `MethodError` の詳細な表示（候補メソッドの提示、類似関数の提案）が重要となる。

**機能の利用シーン**：REPLでのエラー表示、テストフレームワークでのエラー報告、ログファイルへのエラー記録、カスタム例外型の表示定義において利用される。

**主要な処理内容**：
1. `showerror(io, ex)` による例外型別のエラーメッセージフォーマット
2. `showerror(io, ex, bt)` によるエラーメッセージ+バックトレースの表示
3. `show_backtrace(io, bt)` によるスタックトレースの整形表示
4. `MethodError` の詳細表示（候補メソッド、型比較、ヒント）
5. エラーヒントシステム（`Experimental.show_error_hints`）

**関連システム・外部連携**：スタックトレースモジュール（`StackTraces`）、IRShow モジュール。

**権限による制御**：なし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Juliaプロンプト（julia>） | 主画面 | 評価エラー発生時のshowerror/stacktraceによるエラーメッセージ・スタックトレース表示 |
| 4 | シェルモード（shell>） | 参照画面 | コマンド実行失敗時のdisplay_errorによるエラーメッセージ表示 |
| 11 | フォールバックREPL | 参照画面 | display_errorによるエラーメッセージ表示 |

## 機能種別

テキスト出力 / フォーマッティング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| io | `IO` | Yes | 出力先ストリーム | - |
| ex | `Exception` | Yes | 表示対象の例外オブジェクト | - |
| bt | `Vector` | No | バックトレース | - |
| backtrace | `Bool` | No | バックトレースを表示するか | デフォルト `true` |

### 入力データソース

例外オブジェクト、バックトレース（`catch_backtrace()` の返り値）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| formatted_error | テキスト | フォーマットされたエラーメッセージとスタックトレース |

### 出力先

指定された `IO` ストリーム（通常は `stderr`）

## 処理フロー

### 処理シーケンス

```
1. 例外型の判定
   └─ 例外オブジェクトの型に対応する showerror メソッドを選択
2. エラーメッセージのフォーマット
   └─ 型別にエラーの詳細情報をテキストにフォーマット
3. エラーヒントの表示
   └─ Experimental.show_error_hints で追加ヒントを表示
4. バックトレースの整形
   └─ StackFrame 配列をファイル名・行番号・関数名で整形
5. 出力
   └─ IO ストリームにフォーマット済みテキストを書き出し
```

### フローチャート

```mermaid
flowchart TD
    A[showerror 呼び出し] --> B{例外型?}
    B -->|MethodError| C[メソッド候補の表示・型比較・ヒント]
    B -->|BoundsError| D[アクセス対象と範囲を表示]
    B -->|TypeError| E[期待型と実際型の比較表示]
    B -->|SystemError| F[errno メッセージ表示]
    B -->|その他| G[汎用 show で表示]
    C --> H[show_error_hints]
    D --> H
    E --> H
    F --> I{バックトレースあり?}
    G --> I
    H --> I
    I -->|Yes| J[show_backtrace]
    I -->|No| K[終了]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-64-01 | 型別表示 | 各例外型に固有の showerror メソッドを提供する | 対応する例外型が throw された場合 |
| BR-64-02 | フォールバック | 未定義の例外型は `show(io, ex)` にフォールバックする | showerror が未定義の例外型 |
| BR-64-03 | MethodError ヒント | MethodError では候補メソッドと同名関数を他モジュールから検索して提案する | MethodError 発生時 |
| BR-64-04 | StackFrame 整形 | C フレームはデフォルトで非表示、`JULIA_STACKTRACE_ABBREVIATED` で制御 | バックトレース表示時 |

### 計算ロジック

該当なし。

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | showerror 自体でエラーが発生した場合 | try/finally で安全に処理 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

エラー表示はエラーパスでのみ実行されるため、厳密なパフォーマンス要件はない。ただし、大量のバックトレースフレームがある場合に表示が著しく遅くならないこと。

## セキュリティ考慮事項

スタックトレースにはファイルパスや関数名が含まれるため、外部に露出する場合は注意が必要。

## 備考

- ユーザーは `Base.showerror(io::IO, err::MyError)` を定義してカスタム表示を実装可能
- `Experimental.register_error_hint` でエラーヒントを登録可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stacktraces.jl | `base/stacktraces.jl` | `StackFrame` 構造体（51-67行目）: スタックフレームの表現。func, file, line, linfo, from_c, inlined, pointer を保持 |
| 1-2 | stacktraces.jl | `base/stacktraces.jl` | `StackTrace` 型エイリアス（78行目）: `Vector{StackFrame}` |

**読解のコツ**: `StackFrame` はプラットフォーム非依存で設計されており、プロファイルデータの他マシンへの転送も考慮されている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | errorshow.jl | `base/errorshow.jl` | `showerror(io, ex)` 関数（30行目）: 汎用のフォールバック実装 |
| 2-2 | errorshow.jl | `base/errorshow.jl` | `showerror(io, ex, bt)` 関数（107-113行目）: バックトレース付き表示 |

**主要処理フロー**:
1. **30行目**: デフォルトの `showerror` は `show(io, ex)` に委譲
2. **107-113行目**: try/finally で `showerror(io, ex)` を呼び、その後 `show_backtrace(io, bt)` を呼ぶ

#### Step 3: 型別表示を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | errorshow.jl | `base/errorshow.jl` | `showerror(io, ex::BoundsError)` 関数（55-76行目）: 範囲外アクセスの詳細表示 |
| 3-2 | errorshow.jl | `base/errorshow.jl` | `showerror(io, ex::TypeError)` 関数（78-105行目）: 型エラーの詳細表示 |
| 3-3 | errorshow.jl | `base/errorshow.jl` | `showerror(io, ex::MethodError)` 関数（258行目以降）: メソッドエラーの詳細表示（最も複雑） |
| 3-4 | errorshow.jl | `base/errorshow.jl` | 各種簡易エラー表示（160-175行目）: DivideError, EOFError, ErrorException 等 |

### プログラム呼び出し階層図

```
showerror(io, ex, bt; backtrace=true)
    │
    ├─ showerror(io, ex)          # 型別ディスパッチ
    │      ├─ showerror(io, ::BoundsError)
    │      ├─ showerror(io, ::TypeError)
    │      ├─ showerror(io, ::MethodError)
    │      │      ├─ show_convert_error()
    │      │      ├─ show_signature_function()
    │      │      ├─ show_tuple_as_call()
    │      │      └─ print_with_compare()
    │      ├─ showerror(io, ::DomainError)
    │      ├─ showerror(io, ::SystemError)
    │      └─ show(io, ex)        # フォールバック
    │
    ├─ Experimental.show_error_hints(io, ex)
    │
    └─ show_backtrace(io, bt)
           └─ StackFrame の整形表示
```

### データフロー図

```
[入力]                   [処理]                        [出力]

例外オブジェクト ──▶ showerror()                 ──▶ フォーマットされた
バックトレース ───▶ show_backtrace()             ──▶ エラーメッセージ
                         │                            (IO ストリーム)
                         ▼
                   StackFrame の解析
                   ファイル名・行番号の取得
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| errorshow.jl | `base/errorshow.jl` | ソース | エラー表示の主実装 |
| stacktraces.jl | `base/stacktraces.jl` | ソース | スタックトレースのデータ型と操作 |
| error.jl | `base/error.jl` | ソース | backtrace() / catch_backtrace() の実装 |
| methodshow.jl | `base/methodshow.jl` | ソース | メソッドシグネチャの表示 |
| show.jl | `base/show.jl` | ソース | 汎用の show メソッド |
