# 機能設計書 69-C型

## 概要

本ドキュメントは、JuliaにおけるC互換型定義（`Cint` / `Clong` / `Cstring` / `Cwstring` 等）の設計と実装を記述するものである。

### 本機能の処理概要

本機能は、C言語の基本型に対応するJuliaの型エイリアスを定義する。これにより、ccall（No.66）で外部C関数を呼び出す際に、Cの型を正確に指定でき、プラットフォーム間のポータビリティを確保する。

**業務上の目的・背景**：C言語の型サイズはプラットフォーム依存である（例: `long` は Linux 64bit では 8バイトだが Windows 64bit では 4バイト）。Julia の ccall で C 関数を呼び出す際に、プラットフォームに応じた正しい型マッピングを提供することで、ポータブルな FFI コードの記述を可能にする。

**機能の利用シーン**：ccall での引数型/戻り値型の指定、C構造体のJuliaでの再現、JLLパッケージのバインディング生成、プラットフォーム依存コードの記述に利用される。

**主要な処理内容**：
1. プラットフォーム非依存の型エイリアス定義（`Cint`, `Cuint`, `Cfloat`, `Cdouble` 等）
2. プラットフォーム依存の型エイリアス定義（`Clong`, `Culong`, `Cwchar_t`, `Cchar`）
3. NUL終端文字列型（`Cstring`, `Cwstring`）の定義
4. `Cmode_t` 等の OS 固有型の定義

**関連システム・外部連携**：Cランタイムとの型互換性、ccall機構との連携。

**権限による制御**：なし。

## 関連画面

該当なし。本機能はプログラマティックに使用される型定義。

## 機能種別

型定義

## 入力仕様

### 入力パラメータ

本機能は型定義のみであり、入力パラメータはない。

### 入力データソース

コンパイル時のプラットフォーム情報（`ccall(:jl_get_UNAME, Any, ())` / `ccall(:jl_is_char_signed, ...)` 等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Cuchar | `UInt8` | C `unsigned char` |
| Cshort | `Int16` | C `signed short` |
| Cushort | `UInt16` | C `unsigned short` |
| Cint | `Int32` | C `signed int` |
| Cuint | `UInt32` | C `unsigned int` |
| Clong | `Int32` or `Int` | C `signed long`（プラットフォーム依存） |
| Culong | `UInt32` or `UInt` | C `unsigned long`（プラットフォーム依存） |
| Clonglong | `Int64` | C `signed long long` |
| Culonglong | `UInt64` | C `unsigned long long` |
| Cfloat | `Float32` | C `float` |
| Cdouble | `Float64` | C `double` |
| Cchar | `Int8` or `UInt8` | C `char`（符号はプラットフォーム依存） |
| Cwchar_t | `UInt16` or `Int32` | C `wchar_t`（プラットフォーム依存） |
| Cptrdiff_t | `Int` | C `ptrdiff_t` |
| Csize_t | `UInt` | C `size_t` |
| Cssize_t | `Int` | C `ssize_t` |
| Cintmax_t | `Int64` | C `intmax_t` |
| Cuintmax_t | `UInt64` | C `uintmax_t` |
| Cstring | primitive type | NUL終端 C 文字列ポインタ |
| Cwstring | primitive type | NUL終端ワイド文字列ポインタ |

### 出力先

Julia の型システムに定数として登録

## 処理フロー

### 処理シーケンス

```
1. プラットフォーム非依存型の定義
   └─ Cint = Int32, Cuint = UInt32 等（ctypes.jl）
2. プラットフォーム検出
   └─ ccall(:jl_get_UNAME) で OS を判定、ccall(:jl_is_char_signed) で char の符号を判定
3. プラットフォーム依存型の定義
   └─ Windows: Clong = Int32, Cwchar_t = UInt16
   └─ 他: Clong = Int, Cwchar_t = Int32
4. Cstring/Cwstring の定義
   └─ 64bit: 64bit primitive type, 32bit: 32bit primitive type
```

### フローチャート

```mermaid
flowchart TD
    A[起動時] --> B[プラットフォーム非依存型定義]
    B --> C{OS判定}
    C -->|Windows| D["Clong=Int32, Culong=UInt32, Cwchar_t=UInt16"]
    C -->|Unix/Mac| E["Clong=Int, Culong=UInt, Cwchar_t=Int32"]
    D --> F{char 符号判定}
    E --> F
    F -->|signed| G["Cchar=Int8"]
    F -->|unsigned| H["Cchar=UInt8"]
    G --> I{ポインタサイズ}
    H --> I
    I -->|64bit| J["Cstring: 64bit, Cwstring: 64bit"]
    I -->|32bit| K["Cstring: 32bit, Cwstring: 32bit"]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-69-01 | プラットフォーム一致 | C 型エイリアスは対象プラットフォームの C コンパイラの型サイズと一致すること | 全プラットフォーム |
| BR-69-02 | Windows long | Windows では `Clong` は `Int32`（LLP64 モデル） | Windows |
| BR-69-03 | Unix long | Unix/macOS では `Clong` は `Int`（LP64 モデル） | Unix/macOS |
| BR-69-04 | NUL チェック | `Cstring` / `Cwstring` は文字列に埋め込み NUL がないことを検証する | ccall 引数変換時 |

### 計算ロジック

該当なし（コンパイル時定数定義のみ）。

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ErrorException | `sizeof_mode_t` が 2, 4, 8 以外の場合 | サポート外プラットフォーム |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

型エイリアスはゼロコスト抽象化であり、ランタイムオーバーヘッドなし。

## セキュリティ考慮事項

- `Cstring` の NUL 終端チェックにより、バッファオーバーランを防止する
- 型の不一致による ccall のメモリ破壊を防ぐため、正しい C 型を使用することが重要

## 備考

- Julia には C の `_Float16` に対応するエイリアスはない（C 標準ではまだ標準化されていないため）
- `Cmode_t` は Unix 系のみで定義される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: プラットフォーム非依存型を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ctypes.jl | `base/ctypes.jl` | 全体: プラットフォーム非依存の C 型エイリアス定義。Cuchar(11行目), Cshort(19行目), Cushort(27行目), Cint(35行目), Cuint(43行目), Cptrdiff_t(51行目), Csize_t(59行目), Cssize_t(67行目), Cintmax_t(75行目), Cuintmax_t(83行目), Clonglong(91行目), Culonglong(99行目), Cfloat(107行目), Cdouble(115行目) |

**読解のコツ**: 各型定義は docstring 付きの `const` 宣言であり、Julia のプリミティブ整数型・浮動小数点型へのエイリアスである。

#### Step 2: プラットフォーム依存型を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | c.jl | `base/c.jl` | `Cchar` 定義（81-91行目）: `jl_is_char_signed` で符号を判定し `Int8` または `UInt8` に設定 |
| 2-2 | c.jl | `base/c.jl` | `Clong` / `Culong` / `Cwchar_t` 定義（94-123行目）: Windows は LLP64、Unix は LP64 モデルに従う |
| 2-3 | c.jl | `base/c.jl` | `Cmode_t` 定義（125-136行目）: Unix のみ、`sizeof_mode_t` に応じて Int16/Int32/Int64 に設定 |

**主要処理フロー**:
1. **81行目**: `ccall(:jl_is_char_signed, Ref{Bool}, ())` で char の符号を判定
2. **94行目**: `ccall(:jl_get_UNAME, Any, ())` で OS 名（`:NT` = Windows）を取得
3. **95-101行目**: Windows なら `Clong=Int32`, Unix なら `Clong=Int`

#### Step 3: Cstring/Cwstring を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | refpointer.jl | `base/refpointer.jl` | `Cstring` / `Cwstring` 定義（86-92行目）: ポインタサイズに応じた primitive type。NUL終端 C 文字列ポインタを表現 |

### プログラム呼び出し階層図

```
[コンパイル時]
    │
    ├─ ctypes.jl
    │      └─ const Cint = Int32, Cfloat = Float32, ...
    │
    ├─ c.jl
    │      ├─ ccall(:jl_is_char_signed) → Cchar 決定
    │      ├─ ccall(:jl_get_UNAME)      → Clong, Culong, Cwchar_t 決定
    │      └─ ccall(:jl_sizeof_mode_t)  → Cmode_t 決定 (Unix)
    │
    └─ refpointer.jl
           └─ primitive type Cstring/Cwstring
```

### データフロー図

```
[入力]                       [処理]                      [出力]

プラットフォーム情報 ──▶ ccall(:jl_get_UNAME)    ──▶ OS 識別子
char 符号情報 ─────▶ ccall(:jl_is_char_signed) ──▶ Bool
                         │
                         ▼
                    型エイリアス定義 ──▶ Julia グローバル定数
                    Cint, Clong, Cchar, ...
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ctypes.jl | `base/ctypes.jl` | ソース | プラットフォーム非依存の C 型エイリアス |
| c.jl | `base/c.jl` | ソース | プラットフォーム依存の C 型エイリアスと ccall 関連 |
| refpointer.jl | `base/refpointer.jl` | ソース | Cstring / Cwstring の primitive type 定義 |
| pointer.jl | `base/pointer.jl` | ソース | Ptr{T} と C_NULL の定義 |
