# 機能設計書 7-浮動小数点演算

## 概要

本ドキュメントは、Julia言語におけるIEEE 754準拠の浮動小数点演算機能に関する機能設計書である。

### 本機能の処理概要

IEEE 754規格に準拠したFloat16, Float32, Float64型の浮動小数点演算を提供する。丸め制御、特殊値判定（Inf, NaN）、精度情報取得（eps, nextfloat/prevfloat）、ビット表現操作等の包括的な浮動小数点処理機能を含む。

**業務上の目的・背景**：科学計算・数値シミュレーション・機械学習等において、実数近似としての浮動小数点演算は不可欠である。IEEE 754準拠により、異なるプラットフォーム間での計算結果の再現性を保証し、特殊値（Inf, NaN）の統一的なハンドリングを実現する。Float16は深層学習の推論、Float32は一般的なグラフィクス処理、Float64は高精度科学計算にそれぞれ適している。

**機能の利用シーン**：科学計算、統計処理、信号処理、機械学習の損失関数計算、金融計算、物理シミュレーション等で使用される。

**主要な処理内容**：
1. 基本算術演算（+, -, *, /）のIEEE 754準拠実装
2. 丸め関数（round, floor, ceil, trunc）
3. 特殊値判定（isnan, isinf, isfinite, issubnormal）
4. 精度・範囲関数（eps, nextfloat, prevfloat, floatmin, floatmax）
5. ビット表現操作（exponent, significand, sign_mask等）

**関連システム・外部連携**：Core.Intrinsics による低レベルFPU命令マッピング、LLVMの浮動小数点最適化と連携する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 浮動小数点演算は画面に直接関連しないが、REPL上での数値表示に影響する |

## 機能種別

計算処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | IEEEFloat | Yes | 第一オペランド | Float16/32/64であること |
| y | IEEEFloat | Yes（二項演算） | 第二オペランド | Float16/32/64であること |
| mode | RoundingMode | No | 丸めモード | 有効なRoundingModeであること |

### 入力データソース

関数呼び出し引数として直接渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 演算結果 | IEEEFloat | 算術演算の結果 |
| 判定結果 | Bool | 特殊値判定の結果 |
| 精度情報 | IEEEFloat | eps等の精度情報 |

### 出力先

関数の戻り値として返される。

## 処理フロー

### 処理シーケンス

```
1. 引数の型確認
   └─ 多重ディスパッチにより適切なIEEEFloat型のメソッド選択
2. 基本演算の実行
   └─ FPU命令またはCore.Intrinsics関数の呼び出し
3. 特殊値の処理
   └─ NaN伝播、Inf演算のIEEE 754準拠ハンドリング
4. 丸め処理
   └─ RoundingModeに基づく丸め（RoundNearest, RoundDown等）
```

### フローチャート

```mermaid
flowchart TD
    A[浮動小数点演算呼び出し] --> B{特殊値入力?}
    B -->|NaN| C[NaN伝播]
    B -->|Inf| D[Inf演算規則適用]
    B -->|Normal| E[FPU命令実行]
    C --> F[結果返却]
    D --> F
    E --> F
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | NaN伝播 | NaNが入力に含まれる場合、結果はNaNになる | 算術演算時 |
| BR-02 | Inf演算 | Inf - Inf = NaN, Inf / Inf = NaN, 0 * Inf = NaN | 特殊値を含む演算時 |
| BR-03 | ゼロ除算 | 1.0 / 0.0 = Inf, -1.0 / 0.0 = -Inf | float除算時 |
| BR-04 | 丸めモード | デフォルトは最近偶数丸め（RoundNearest） | round関数使用時 |
| BR-05 | 等価性 | NaN == NaN は false、NaN === NaN もビットパターンにより異なる場合がある | 比較時 |

### 計算ロジック

- ビットマスク：`sign_mask(::Type{Float64}) = 0x8000_0000_0000_0000`（float.jl 107行目）
- 指数マスク：`exponent_mask(::Type{Float64}) = 0x7ff0_0000_0000_0000`（float.jl 108行目）
- 仮数マスク：`significand_mask(::Type{Float64}) = 0x000f_ffff_ffff_ffff`（float.jl 111行目）

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| DomainError | DomainError | sqrt(-1.0)等、実数範囲外の引数 | Complex引数を使用する |
| InexactError | InexactError | NaN/Infの整数変換 | isfinite判定を事前に行う |

### リトライ仕様

浮動小数点演算は純粋関数であり、リトライは適用されない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 基本浮動小数点演算はCPUのFPU/SSE/AVX命令にマッピングされ、1〜数クロックサイクルで実行される
- Float16演算はハードウェアサポートがない場合、ソフトウェアエミュレーションとなる

## セキュリティ考慮事項

- 浮動小数点の精度制限に起因する比較誤りに注意（`x == y` の代わりに `isapprox(x, y)` の使用を推奨）

## 備考

- float.jlの3行目で `const IEEEFloat = Union{Float16, Float32, Float64}` が定義されている
- floatfuncs.jlにround, maxintfloat等の高レベル関数が定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

浮動小数点型のビット表現を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | float.jl | `base/float.jl` | **3行目**: `const IEEEFloat = Union{Float16, Float32, Float64}` |
| 1-2 | float.jl | `base/float.jl` | **107-123行目**: sign_mask, exponent_mask, significand_mask各精度の定義 |
| 1-3 | float.jl | `base/float.jl` | **127-138行目**: significand_bits, exponent_bits, exponent_bias の計算 |

**読解のコツ**: IEEE 754の浮動小数点数は「符号ビット + 指数部 + 仮数部」で構成される。float.jlではこれらのビットフィールドをマスク定数として定義し、reinterpretとビット演算で操作する。

#### Step 2: エントリーポイントを理解する

特殊定数と基本関数の定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | float.jl | `base/float.jl` | **15-41行目**: Inf16/32/64, NaN16/32/64, Inf, NaN定数の定義 |
| 2-2 | float.jl | `base/float.jl` | **100-105行目**: reinterpret関数によるビット表現⇔浮動小数点の変換 |

**主要処理フロー**:
1. **15行目**: `const Inf16 = bitcast(Float16, 0x7c00)` - Float16の正の無限大
2. **38行目**: `const Inf64 = bitcast(Float64, 0x7ff0000000000000)` - Float64の正の無限大
3. **100-105行目**: reinterpret(Unsigned/Signed, x) でビットパターンへの変換

#### Step 3: 高レベル関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | floatfuncs.jl | `base/floatfuncs.jl` | **5-13行目**: copysign, flipsign, signbit の浮動小数点版 |
| 3-2 | floatfuncs.jl | `base/floatfuncs.jl` | **19-45行目**: maxintfloat関数（正確に表現可能な最大整数） |
| 3-3 | floatfuncs.jl | `base/floatfuncs.jl` | **52-75行目**: round関数の実装（digits, sigdigits, base対応） |

### プログラム呼び出し階層図

```
浮動小数点演算
    │
    ├─ 基本演算 (+, -, *, /)
    │      └─ Core.Intrinsics (add_float, sub_float, mul_float, div_float)
    │             └─ FPU/SSE/AVX命令
    │
    ├─ ビット表現操作
    │      ├─ sign_mask / exponent_mask / significand_mask (float.jl)
    │      ├─ reinterpret (float.jl:100-105)
    │      └─ bitcast (Core.Intrinsics)
    │
    ├─ 特殊値判定
    │      ├─ isnan / isinf / isfinite
    │      └─ issubnormal / iszero
    │
    └─ 丸め・精度関数
           ├─ round / floor / ceil / trunc (floatfuncs.jl)
           ├─ eps / nextfloat / prevfloat
           └─ maxintfloat (floatfuncs.jl:19-45)
```

### データフロー図

```
[入力]                      [処理]                         [出力]

Float64値 x, y        ───▶ IEEE 754演算              ───▶ Float64結果
                             │
                             ├─ NaN伝播チェック
                             ├─ Inf演算規則
                             └─ 丸め処理

Float64値 x           ───▶ reinterpret(UInt64, x)    ───▶ ビットパターン
                      ───▶ ビットマスク適用           ───▶ 指数部/仮数部
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| float.jl | `base/float.jl` | ソース | 浮動小数点型の定数・変換・ビット操作定義 |
| floatfuncs.jl | `base/floatfuncs.jl` | ソース | round, maxintfloat等の高レベル関数 |
| boot.jl | `base/boot.jl` | ソース | Float16/32/64の型定義 |
| rounding.jl | `base/rounding.jl` | ソース | RoundingMode型の定義 |
| math.jl | `base/math.jl` | ソース | 数学関数（sqrt, exp, log等） |
