# 機能設計書 73-型リフレクション

## 概要

本ドキュメントは、Julia言語における型リフレクション機能の設計を記述する。`typeof`、`isa`、`supertype`、`fieldnames`、`fieldtypes`、`isconcretetype`、`isabstracttype` などの関数を通じて、実行時の型情報取得を提供する。

### 本機能の処理概要

型リフレクション機能は、Juliaオブジェクトの型情報をプログラム実行時に取得・検査するための一連の関数を提供する。型階層の探索、構造体フィールドの内省、型の具象性・抽象性の判定、型パラメータの取得など、メタプログラミングや動的コード生成に不可欠な基盤機能である。

**業務上の目的・背景**：Juliaの多重ディスパッチシステムは型に基づいており、実行時に型情報を取得する能力は、ジェネリックプログラミング、デバッグ、コード生成、シリアライズ、ドキュメント生成など多くの場面で必要とされる。型リフレクションは、コンパイラの型推論機構とは異なり、ユーザーレベルで型情報にアクセスするための公開APIを提供する。

**機能の利用シーン**：型に基づく条件分岐、構造体のフィールド操作のための情報取得、型パラメータに基づくジェネリックアルゴリズム、REPLでの型探索、ドキュメント生成時の型情報表示、シリアライズ時の型構造解析などで利用される。

**主要な処理内容**：
1. `typeof(x)` - オブジェクトの具象型を取得
2. `isa(x, T)` - オブジェクトが型Tのインスタンスか判定
3. `supertype(T)` - 型Tの直接の親型を取得
4. `fieldnames(T)` - 構造体のフィールド名をタプルで取得
5. `fieldtypes(T)` - 構造体のフィールド型をタプルで取得
6. `isconcretetype(T)` / `isabstracttype(T)` - 型の具象/抽象判定
7. `nfields(x)` - フィールド数を取得
8. `fieldcount(T)` - 型のフィールド数を取得
9. `sizeof(T)` - 型のメモリサイズを取得
10. `subtypes(T)` - 直接のサブタイプ一覧を取得
11. `supertypes(T)` - Anyまでの親型チェーンを取得
12. `typeintersect(T, S)` - 2つの型の交差型を計算
13. `fieldoffset(T, i)` - フィールドのメモリオフセットを取得

**関連システム・外部連携**：Juliaランタイムの型システム（src/jltypes.c）と密接に連携。多くの関数はCoreモジュールの組み込み関数として実装されている。

**権限による制御**：特別な権限制御はない。全ての型情報は公開されている。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | ヘルプモード（help?>） | 補助機能 | subtypes等のリフレクション機能を用いた型情報のドキュメント表示 |

## 機能種別

型情報取得・内省処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | Any | Yes（typeof, isa, nfields） | 対象オブジェクト | 任意のJuliaオブジェクト |
| T | Type | Yes（supertype, fieldnames等） | 対象型 | 有効なJulia型 |
| i | Int | Yes（fieldoffset） | フィールドインデックス | 1 <= i <= fieldcount(T) |

### 入力データソース

- Juliaプログラムからの直接呼び出し
- コンパイラの型推論過程での間接利用

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| type | DataType | typeof()の結果 |
| is_instance | Bool | isa()の結果 |
| parent_type | DataType | supertype()の結果 |
| field_names | Tuple{Symbol...} | fieldnames()の結果 |
| field_types | Tuple{Type...} | fieldtypes()の結果 |
| is_concrete | Bool | isconcretetype()の結果 |
| is_abstract | Bool | isabstracttype()の結果 |

### 出力先

戻り値としてJuliaプログラムに返却

## 処理フロー

### 処理シーケンス

```
1. 型情報の取得要求
   └─ typeof / isa / supertype / fieldnames 等の呼び出し
2. Core組み込み関数の実行
   └─ 多くの関数はCoreモジュールの組み込み関数
3. DataType構造体のフィールドアクセス
   └─ DataType.name, DataType.super, DataType.types 等
4. 結果の返却
   └─ 型情報を適切な形式で返す
```

### フローチャート

```mermaid
flowchart TD
    A[型リフレクション呼び出し] --> B{関数の種類}
    B -->|typeof| C[Core._typeof_で型取得]
    B -->|isa| D[Core.isa で型チェック]
    B -->|supertype| E[DataType.superフィールド参照]
    B -->|fieldnames| F[DataType.name.names参照]
    B -->|fieldtypes| G[DataType.types参照]
    B -->|isconcretetype| H[isdefined flags チェック]
    B -->|subtypes| I[型テーブル走査]
    C --> J[結果返却]
    D --> J
    E --> J
    F --> J
    G --> J
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-73-01 | typeof不変 | typeof(x)は常に具象型を返す | 全オブジェクト |
| BR-73-02 | 型階層 | supertypes(T)はTからAnyまでのチェーンを返す | 全型 |
| BR-73-03 | 具象型判定 | isconcretetypeはインスタンス化可能な型にtrueを返す | DataType |
| BR-73-04 | フィールド順序 | fieldnamesはフィールド定義順にSymbolのタプルを返す | 構造体型 |
| BR-73-05 | Union型 | isabstracttype(Union{A,B})はfalseを返す | Union型 |

### 計算ロジック

`supertypes(T)` は再帰的に `supertype` を呼び出し、`Any` に到達するまでの型チェーンをタプルとして返す。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentError | 例外 | fieldnamesを非構造体型に適用 | 適切な型を指定 |
| BoundsError | 例外 | fieldoffsetに範囲外インデックス指定 | 有効なインデックスを使用 |
| MethodError | 例外 | 適用できない型に関数を呼び出し | 型を確認して適切な関数を使用 |

### リトライ仕様

リトライ不要（純粋な計算処理）

## トランザクション仕様

該当なし

## パフォーマンス要件

- typeof、isaはコンパイラが特別に最適化する組み込み関数であり、極めて高速
- fieldnames、fieldtypesは型のメタデータ参照であり、O(1)に近い計算量
- subtypesは型テーブルの走査を伴うため、型の数に比例する計算量

## セキュリティ考慮事項

特になし。型情報は全て公開情報である。

## 備考

- 多くの関数はCoreモジュールで定義された組み込み関数であり、Juliaの型システムランタイム（src/jltypes.c）に直接アクセスする
- 型リフレクションの結果はコンパイラの型推論でも活用される
- `Base.reflection.jl` に追加のリフレクションユーティリティが定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Juliaの型システムの基盤となるDataType構造体を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | boot.jl | `base/boot.jl` | DataType / TypeName / TypeVar / UnionAll の定義。型オブジェクト自体の構造を理解 |
| 1-2 | essentials.jl | `base/essentials.jl` | 基本的な型操作ユーティリティ |

**読解のコツ**: JuliaのDataType構造体のフィールド（name, super, parameters, types, instance等）が、各型リフレクション関数の返却値の源泉である。

#### Step 2: Core組み込み関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | boot.jl | `base/boot.jl` | typeof, isa, <:（subtype）がCore.Intrinsicsとして定義されている点を確認 |

**主要処理フロー**:
1. `typeof(x)` - Core._typeof_ 組み込み関数。オブジェクトヘッダからDataTypeポインタを取得
2. `isa(x, T)` - Core.isa 組み込み関数。サブタイプ関係を判定

#### Step 3: Base層のリフレクション関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | reflection.jl | `base/reflection.jl` | code_lowered、code_typed、return_types等の高レベルリフレクションAPI |

**主要処理フロー**:
- supertypesはreflection.jlで定義され、supertypeを再帰呼び出し
- subtypesはCcallでjl_subtypesを呼び出し

### プログラム呼び出し階層図

```
typeof(x)
    └─ Core._typeof_(x)  // 組み込み関数

isa(x, T)
    └─ Core.isa(x, T)    // 組み込み関数

supertype(T::DataType)
    └─ T.super            // DataTypeフィールド参照

fieldnames(T::DataType)
    └─ T.name.names       // TypeNameフィールド参照

fieldtypes(T::DataType)
    └─ T.types             // DataTypeフィールド参照

subtypes(T::DataType)
    └─ ccall(:jl_subtypes)  // Cランタイム呼び出し

isconcretetype(T)
    └─ ccall(:jl_is_concrete_type)  // Cランタイム呼び出し
```

### データフロー図

```
[入力]               [処理]                    [出力]

オブジェクト x ──▶ typeof()        ──▶ DataType
                    Core._typeof_

型 T ──────────▶ supertype()      ──▶ DataType
                    T.super

型 T ──────────▶ fieldnames()     ──▶ Tuple{Symbol...}
                    T.name.names

型 T ──────────▶ subtypes()       ──▶ Vector{Any}
                    jl_subtypes
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| reflection.jl | `base/reflection.jl` | ソース | 型リフレクション関数の主要実装（code_typed, return_types等） |
| boot.jl | `base/boot.jl` | ソース | DataType/TypeName/TypeVar等の基盤型定義、Core組み込み関数 |
| essentials.jl | `base/essentials.jl` | ソース | 基本的な型操作ユーティリティ |
| jltypes.c | `src/jltypes.c` | ソース（C） | 型システムのCランタイム実装 |
| subtype.c | `src/subtype.c` | ソース（C） | サブタイプ判定のCランタイム実装 |
