# 機能設計書 83-型推論ユーティリティ

## 概要

本ドキュメントは、Juliaコンパイラにおける型推論ユーティリティ（tfuncs）機能の設計を記述する。tfuncs は、組み込み関数（Intrinsic Functions/Builtin Functions）の型推論ルールを定義し、抽象解釈エンジンがこれらの関数の戻り値型を推論する際に使用するテーブルとユーティリティ関数群である。

### 本機能の処理概要

tfuncs は Julia の組み込み関数に対する型推論のルールセットを提供し、抽象解釈が正確な型情報を導出できるようにする。

**業務上の目的・背景**：Julia の型推論は多重ディスパッチとパラメトリック型の組み合わせにより極めて複雑である。組み込み関数（算術演算、型判定、フィールドアクセスなど）は Julia ランタイムの基盤を構成しており、これらの型推論ルールが正確でなければ、コンパイラは適切な特殊化やインライン化を行えない。tfuncs はこれらのルールを集中的に管理し、型推論の精度とパフォーマンスの両立を実現している。

**機能の利用シーン**：抽象解釈エンジンが組み込み関数呼び出しに遭遇した際に、tfuncs テーブルから対応する型推論関数を検索して呼び出す。これは全てのJuliaコードのコンパイル時に暗黙的に使用される。

**主要な処理内容**：
1. `T_IFUNC` テーブル: IntrinsicFunction（算術・ビット演算等）の型推論ルールの登録と検索
2. `T_FFUNC_KEY/VAL` テーブル: Builtin関数（throw, getfield, setfield! 等）の型推論ルールの登録と検索
3. `instanceof_tfunc`: `typeof` の逆演算（型から実行時値の型範囲を推定）
4. 各組み込み関数の型推論関数（例: `getfield_tfunc`, `setfield!_tfunc`, `isa_tfunc` 等）
5. `@nospecs` マクロ: 型推論関数の引数に `@nospecialize` を自動付与

**関連システム・外部連携**：型推論エンジン（No.79）から直接呼び出される。効果推論（No.81）では各組み込み関数の副作用情報も参照される。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコンパイラ内部機能であり、直接的なUI画面は存在しない |

## 機能種別

計算処理（コンパイラ内部の型推論テーブル管理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| AbstractLattice | AbstractLattice | Yes | 型格子構造（型推論のコンテキスト） | - |
| 引数型リスト | Vector{Any} | Yes | 呼び出しの引数の推論型 | - |
| 関数 | IntrinsicFunction / Builtin | Yes | 型推論対象の組み込み関数 | テーブルに登録されていること |

### 入力データソース

- 抽象解釈エンジンからの呼び出し
- 型格子（TypeLattice）の判定関数群

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 推論型 | Any | 組み込み関数の戻り値の推論型（Const, Type, Union 等を含む） |
| コスト | Int | 関数呼び出しのインライン化コスト |

### 出力先

- 抽象解釈エンジンの型推論結果として使用

## 処理フロー

### 処理シーケンス

```
1. 組み込み関数呼び出しの検出
   └─ 抽象解釈が call/invoke 文を処理中に組み込み関数を検出
2. テーブル検索
   └─ IntrinsicFunction の場合: T_IFUNC[reinterpret(Int32, f) + 1] で直接アクセス
   └─ Builtin の場合: T_FFUNC_KEY を線形検索して find_tfunc で索引取得
3. 引数数チェック
   └─ (minarg, maxarg) で引数数が範囲内か確認
4. 型推論関数呼び出し
   └─ tfunc(lattice, arg1_type, arg2_type, ...) を呼び出し
5. 結果返却
   └─ 推論された戻り値型を返却
```

### フローチャート

```mermaid
flowchart TD
    A[組み込み関数呼び出し検出] --> B{関数の種類}
    B -->|IntrinsicFunction| C[T_IFUNC テーブルから取得]
    B -->|Builtin| D[T_FFUNC_KEY/VAL テーブルから検索]
    C --> E[引数数チェック]
    D --> E
    E --> F{引数数が範囲内?}
    F -->|Yes| G[型推論関数を呼び出し]
    F -->|No| H[Bottom を返却]
    G --> I[推論型を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-83-01 | テーブル登録 | add_tfunc により IntrinsicFunction はインデックス直接アクセス、Builtin はリスト末尾追加 | コンパイラ初期化時 |
| BR-83-02 | 引数数検証 | 登録された (minarg, maxarg) 範囲外の引数数は Bottom を返す | 型推論関数呼び出し前 |
| BR-83-03 | @nospecs マクロ | 型推論関数の引数には @nospecialize が付与され、型推論関数自体の過度な特殊化を防ぐ | 全 tfunc 定義 |
| BR-83-04 | instanceof_tfunc | Type{S} から S の実行時型を推定。astag フラグで有効性チェックの厳密さを制御 | typeof の逆推論 |

### 計算ロジック

IntrinsicFunction のテーブルインデックス計算:
```
idx = reinterpret(Int32, f) + 1
```

コスト値は各組み込み関数に対して手動で設定され、インライン化判定に使用される。例: `throw` のコストは 0。

## データベース操作仕様

本機能はデータベースを使用しない。

### 操作別データベース影響一覧

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentError | @nospecs マクロの不正使用 | マクロ展開時にエラー |
| - | 型推論不正確 | tfunc のルールが不完全な場合 | Any を返して安全側に倒す |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- IntrinsicFunction のテーブル検索は O(1)（配列インデックスアクセス）
- Builtin のテーブル検索は O(n)（線形検索）だが、テーブルサイズが小さいため問題ない
- 各 tfunc は高速に型を推論できるよう設計されている

## セキュリティ考慮事項

特になし。コンパイラ内部処理であり、外部入力を直接扱わない。

## 備考

- `N_IFUNC` は `reinterpret(Int32, have_fma) + 1` で計算され、IntrinsicFunction の数に依存
- `INT_INF = typemax(Int)` は可変引数の上限として使用される
- `DATATYPE_TYPES_FIELDINDEX` と `DATATYPE_NAME_FIELDINDEX` は DataType のフィールドアクセス最適化に使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tfuncs.jl | `Compiler/src/tfuncs.jl` | テーブル定数（52-58行目）: T_IFUNC, T_FFUNC_KEY, T_FFUNC_VAL, T_FFUNC_COST |
| 1-2 | tfuncs.jl | `Compiler/src/tfuncs.jl` | @nospecs マクロ（23-50行目）: 型推論関数の引数を自動的に @nospecialize |

**読解のコツ**: `T_IFUNC` は固定サイズの Vector で IntrinsicFunction の ID をインデックスとして直接アクセスする。`T_FFUNC_KEY/VAL` は動的リストで Builtin 関数をキーとして線形検索する。各エントリは `(minarg, maxarg, tfunc)` のタプル。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tfuncs.jl | `Compiler/src/tfuncs.jl` | add_tfunc 関数（79-88行目）: テーブル登録の2種類のメソッド |
| 2-2 | tfuncs.jl | `Compiler/src/tfuncs.jl` | find_tfunc 関数（60-66行目）: Builtin関数のテーブル検索 |

**主要処理フロー**:
1. **79-83行目**: IntrinsicFunction 用の add_tfunc: `T_IFUNC[idx]` にインデックス直接書き込み
2. **84-88行目**: Builtin 用の add_tfunc: `T_FFUNC_KEY/VAL/COST` にそれぞれ push!
3. **60-66行目**: find_tfunc: T_FFUNC_KEY を線形走査して一致するインデックスを返却

#### Step 3: 代表的な型推論関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tfuncs.jl | `Compiler/src/tfuncs.jl` | instanceof_tfunc 関数（98行目〜）: typeof の逆演算 |
| 3-2 | tfuncs.jl | `Compiler/src/tfuncs.jl` | throw の tfunc（90行目）: 最もシンプルな例（常に Bottom を返す） |

### プログラム呼び出し階層図

```
abstractinterpretation (抽象解釈ループ)
    |
    +-- abstract_call_builtin
    |      +-- find_tfunc (Builtin 関数のテーブル検索)
    |      +-- tfunc(lattice, argtypes...) (型推論関数呼び出し)
    |
    +-- abstract_call_known (IntrinsicFunction)
           +-- T_IFUNC[idx] (直接テーブルアクセス)
           +-- tfunc(lattice, argtypes...) (型推論関数呼び出し)
```

### データフロー図

```
[入力]                         [処理]                    [出力]

関数オブジェクト       --> テーブル検索             --> (minarg, maxarg, tfunc)
引数の推論型リスト     --> 引数数チェック           --> OK / NG
(OK の場合)           --> tfunc 呼び出し            --> 推論された戻り値型
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tfuncs.jl | `Compiler/src/tfuncs.jl` | ソース | 組み込み関数の型推論ルールテーブルと推論関数群の定義 |
| abstractinterpretation.jl | `Compiler/src/abstractinterpretation.jl` | ソース | 抽象解釈エンジン（tfuncsの呼び出し元） |
| typelattice.jl | `Compiler/src/typelattice.jl` | ソース | 型格子構造の定義（tfuncsが使用する型判定関数） |
| effects.jl | `Compiler/src/effects.jl` | ソース | 効果推論（組み込み関数の副作用情報） |
