# 機能設計書 9-複素数演算

## 概要

本ドキュメントは、Julia言語におけるComplex型の四則演算・極座標変換・共役・絶対値などの複素数演算に関する機能設計書である。

### 本機能の処理概要

Complex{T} パラメトリック型による複素数の表現と、複素数に対する四則演算、極座標変換（abs, angle）、共役（conj）、実部・虚部取得（real, imag）等の包括的な複素数演算を提供する。

**業務上の目的・背景**：信号処理（フーリエ変換）、制御工学（伝達関数）、量子力学、電気工学（インピーダンス計算）等の幅広い科学技術計算分野において複素数演算は不可欠である。Julia言語ではComplex{T}をパラメトリック型として定義し、任意の実数型をパラメータとして使用可能にすることで、Float32での省メモリ計算からBigFloatでの高精度計算まで統一的なインターフェースで対応する。

**機能の利用シーン**：FFT（高速フーリエ変換）の基礎計算、電気回路のインピーダンス解析、量子状態の計算、多項式の複素根の計算、信号の位相解析等で使用される。

**主要な処理内容**：
1. Complex{T}型の定義とコンストラクタ
2. 四則演算（+, -, *, /）
3. 実部・虚部取得（real, imag, reim）
4. 共役（conj）と絶対値（abs, abs2）
5. 極座標変換（angle）
6. 複素数判定（isreal, isfinite, isnan, isinf）
7. 型昇格規則（promote_rule）

**関連システム・外部連携**：型昇格システム（base/promotion.jl）と連携し、実数との混合演算を自動処理する。

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 複素数演算は画面に直接関連しないが、REPL上で `a + bim` 形式で表示される |

## 機能種別

計算処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| re | T<:Real | Yes | 実部 | 実数型であること |
| im | T<:Real | Yes | 虚部 | 実部と同じ型（または昇格可能な型）であること |
| z | Complex | Yes（演算時） | 複素数値 | Complex型であること |

### 入力データソース

関数呼び出し引数として直接渡される。`1 + 2im` のようなリテラル表記も使用可能。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 演算結果 | Complex{T} | 複素数演算の結果 |
| 実部 | T | real(z)の結果 |
| 虚部 | T | imag(z)の結果 |
| 絶対値 | T | abs(z)の結果 |
| 偏角 | T | angle(z)の結果（ラジアン） |
| 共役 | Complex{T} | conj(z)の結果 |

### 出力先

関数の戻り値として返される。

## 処理フロー

### 処理シーケンス

```
1. Complex{T}オブジェクトの生成
   └─ re, im の型をpromoteで統一後、struct生成
2. 四則演算の実行
   └─ 実部・虚部を個別に計算
3. 型昇格の適用
   └─ promote_ruleに基づき、Real×Complex, Complex×Complex を処理
```

### フローチャート

```mermaid
flowchart TD
    A[Complex生成] --> B{re, imの型が同一?}
    B -->|Yes| C["Complex{T}(re, im)"]
    B -->|No| D["promote(re, im)"]
    D --> C

    E[複素数演算] --> F{演算種別}
    F -->|加減| G[実部同士・虚部同士を加減]
    F -->|乗算| H["(ac-bd) + (ad+bc)im"]
    F -->|除算| I["数値的に安定な除算アルゴリズム"]
    G --> J[結果返却]
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 型パラメータ一致 | Complex{T}の実部と虚部は同じ型Tを持つ | Complex構築時 |
| BR-02 | 自動昇格 | 異なる型の実部・虚部はpromoteで統一される | Complex(x::Real, y::Real)時 |
| BR-03 | 虚数単位 | `im` は `Complex(false, true)` として定義される | 常時 |
| BR-04 | isreal判定 | isreal(z)はiszero(imag(z))と等価 | 判定時 |
| BR-05 | 実数変換 | isreal(z)がtrueの複素数のみReal型に変換可能 | convert時 |

### 計算ロジック

- 乗算: `(a+bi)(c+di) = (ac-bd) + (ad+bc)i`
- 除算: 数値的安定性のためのアルゴリズム使用
- abs: `sqrt(re^2 + im^2)`、ただしオーバーフロー対策あり
- abs2: `re^2 + im^2`（sqrtなしで高速）

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InexactError | InexactError | 虚部が非ゼロの複素数をRealに変換しようとした場合 | real()で実部のみ取得する |

### リトライ仕様

複素数演算は純粋関数であり、リトライは適用されない。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- Complex{Float64}の演算はFloat64の2〜4倍程度のコスト
- abs2はsqrtを省略するため、absより高速
- sincos(z)は個別にsin(z), cos(z)を計算するより効率的

## セキュリティ考慮事項

特になし。

## 備考

- complex.jlの13-16行目で `struct Complex{T<:Real} <: Number` が定義されている
- complex.jlの36行目で `const im = Complex(false, true)` が定義されている
- ComplexF64 = Complex{Float64}, ComplexF32 = Complex{Float32}, ComplexF16 = Complex{Float16} のエイリアスが定義されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Complex型の内部構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | complex.jl | `base/complex.jl` | **13-16行目**: `struct Complex{T<:Real} <: Number` 定義。re::TとIm::Tの2フィールド |
| 1-2 | complex.jl | `base/complex.jl` | **17-18行目**: コンストラクタ。異なる型はpromoteで統一 |
| 1-3 | complex.jl | `base/complex.jl` | **36行目**: `const im = Complex(false, true)` 虚数単位 |
| 1-4 | complex.jl | `base/complex.jl` | **38-40行目**: ComplexF64/F32/F16 エイリアス |

**読解のコツ**: Complex{T}はパラメトリック構造体であり、Tは実部・虚部の型を決定する。`1 + 2im` は `Complex(1, 2)` に等しく、実部と虚部がpromoteされて `Complex{Int}(1, 2)` となる。

#### Step 2: エントリーポイントを理解する

基本操作関数の定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | complex.jl | `base/complex.jl` | **72行目**: `real(z::Complex) = z.re` - 実部取得 |
| 2-2 | complex.jl | `base/complex.jl` | **87行目**: `imag(z::Complex) = z.im` - 虚部取得 |
| 2-3 | complex.jl | `base/complex.jl` | **102行目**: `reim(z) = (real(z), imag(z))` - 実部・虚部タプル |
| 2-4 | complex.jl | `base/complex.jl` | **147-148行目**: isreal, isinteger, isfinite判定 |

**主要処理フロー**:
1. **72行目**: `real(z::Complex) = z.re`
2. **87行目**: `imag(z::Complex) = z.im`
3. **88行目**: `real(x::Real) = x` - 実数の場合はそのまま返す
4. **89行目**: `imag(x::Real) = zero(x)` - 実数の虚部はゼロ

#### Step 3: 型昇格規則を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | complex.jl | `base/complex.jl` | **49-52行目**: promote_rule定義。Complex{T}×Real{S} → Complex{promote_type(T,S)} |

### プログラム呼び出し階層図

```
Complex{T} コンストラクタ (complex.jl:13-18)
    │
    ├─ Complex(x::Real, y::Real) → promote(x, y) → Complex{T}(x', y')
    │
    ├─ 基本操作
    │      ├─ real(z) → z.re (complex.jl:72)
    │      ├─ imag(z) → z.im (complex.jl:87)
    │      ├─ conj(z) → Complex(real(z), -imag(z))
    │      └─ abs(z) → sqrt(abs2(z))
    │
    ├─ 四則演算
    │      ├─ +(z, w) → Complex(real(z)+real(w), imag(z)+imag(w))
    │      ├─ -(z, w) → Complex(real(z)-real(w), imag(z)-imag(w))
    │      ├─ *(z, w) → Complex(re*re-im*im, re*im+im*re)
    │      └─ /(z, w) → 数値安定除算アルゴリズム
    │
    └─ 型昇格
           └─ promote_rule (complex.jl:49-52)
```

### データフロー図

```
[入力]                        [処理]                          [出力]

実部 re, 虚部 im        ───▶ promote(re, im)           ───▶ Complex{T}(re', im')

Complex z               ───▶ real(z), imag(z)          ───▶ 実部 T, 虚部 T

Complex z, w            ───▶ 四則演算                   ───▶ Complex{T} 結果

Complex z               ───▶ abs(z)                    ───▶ sqrt(re^2 + im^2)
                        ───▶ angle(z)                  ───▶ atan(im, re)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| complex.jl | `base/complex.jl` | ソース | Complex型定義、四則演算、基本操作、型昇格 |
| promotion.jl | `base/promotion.jl` | ソース | promote_type/promote_ruleの基盤 |
| math.jl | `base/math.jl` | ソース | 複素数版の数学関数（sin, cos, exp等） |
| float.jl | `base/float.jl` | ソース | 浮動小数点基本定義（isnan, isinf等） |
