# 機能設計書 90-メモリ管理

## 概要

本ドキュメントは、Juliaのメモリ管理機能の設計を記述する。配列（Array）とGenericMemoryのメモリ割り当て、ライトバリア、ファイナライザなどの低レベルメモリ管理を提供する。

### 本機能の処理概要

メモリ管理は、Julia オブジェクト（特に配列とメモリバッファ）のメモリ割り当て・初期化・解放を担当するランタイム機能である。

**業務上の目的・背景**：Julia は数値計算・データサイエンス用途で大量の配列操作を行うため、効率的なメモリ管理が性能に直結する。Array はユーザーが最も頻繁に使用するデータ構造であり、GenericMemory はその内部バッファとして機能する。プールアロケータによる小オブジェクトの高速割当、大オブジェクトの malloc 委譲、GCとの連携によるライトバリアなど、多層的なメモリ管理を実現している。

**機能の利用シーン**：
- 配列の生成（`Array{T}(undef, dims...)`）
- `push!` / `append!` による配列の動的拡張
- 文字列・構造体等の一般オブジェクトの割当
- C 言語連携での手動メモリ管理

**主要な処理内容**：
1. `jl_alloc_genericmemory`: GenericMemory の割当（プール/malloc 自動選択）
2. `new_array` / `_new_array`: Array オブジェクトの生成
3. `jl_array_validate_dims`: 配列次元数の検証とオーバーフローチェック
4. `jl_gc_managed_malloc`: GC 管理下の malloc 割当
5. `jl_gc_track_malloced_genericmemory`: 大オブジェクトの GC 追跡登録
6. ゼロ初期化: 必要に応じた割当メモリのゼロクリア

**関連システム・外部連携**：GCランタイム（No.87）がメモリの回収を担当。Array型（No.15）とMemory型（No.16）がこの機能の上位インタフェース。コード生成（No.84）が配列操作のネイティブコードを生成。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はランタイム内部機能であり、直接的なUI画面は存在しない |

## 機能種別

計算処理（ランタイムメモリ割当処理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| atype / mtype | jl_value_t* | Yes | 配列/メモリの型（Array{T,N} / GenericMemory{kind,T}） | 有効な DataType であること |
| nel | size_t | Yes | 要素数 | オーバーフローチェック |
| ndims | uint32_t | Yes（配列） | 次元数 | Array{T,N} の N と一致 |
| dims | size_t* | Yes（配列） | 各次元のサイズ | 全次元の積がオーバーフローしないこと |

### 入力データソース

- Julia コードからの配列生成呼び出し
- コード生成器（codegen.cpp）からの直接呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| jl_array_t* | ポインタ | 生成された Array オブジェクト |
| jl_genericmemory_t* | ポインタ | 生成された GenericMemory オブジェクト |

### 出力先

- Julia ヒープ上に割り当てられ、GC 管理下に置かれる

## 処理フロー

### 処理シーケンス

```
1. 次元数検証（Array の場合）
   └─ jl_array_validate_dims で各次元の積をチェック
2. GenericMemory の割当
   └─ 要素数 × 要素サイズ = 必要バイト数を計算
   └─ Union 型の場合は追加のタグバイトを加算
   └─ オーバーフローチェック
3. 割当方式の選択
   └─ 合計サイズ ≤ GC_MAX_SZCLASS → プールアロケータ
   └─ 合計サイズ > GC_MAX_SZCLASS → jl_gc_managed_malloc
4. メモリの初期化
   └─ ゼロ初期化が必要な場合は memset
5. Array オブジェクトの構築（Array の場合）
   └─ jl_gc_alloc で Array ヘッダを割当
   └─ ref.mem に GenericMemory を設定
   └─ dimsize に各次元サイズを設定
```

### フローチャート

```mermaid
flowchart TD
    A[配列生成要求] --> B[jl_array_validate_dims]
    B --> C{次元数が有効?}
    C -->|No| D[ArgumentError をスロー]
    C -->|Yes| E[必要バイト数を計算]
    E --> F{オーバーフロー?}
    F -->|Yes| D
    F -->|No| G{nel == 0?}
    G -->|Yes| H[ゼロサイズインスタンスを返却]
    G -->|No| I{サイズ ≤ GC_MAX_SZCLASS?}
    I -->|Yes| J[プールアロケータで割当]
    I -->|No| K[jl_gc_managed_malloc で割当]
    J --> L[メタデータ設定]
    K --> M[GC追跡登録]
    M --> L
    L --> N{ゼロ初期化必要?}
    N -->|Yes| O[memset でゼロクリア]
    N -->|No| P[Array ヘッダ構築]
    O --> P
    P --> Q[完成した Array を返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-90-01 | ゼロサイズ最適化 | nel == 0 の場合は型のシングルトンインスタンスを返却し、割当を回避 | GenericMemory の nel == 0 時 |
| BR-90-02 | プール/malloc 閾値 | GC_MAX_SZCLASS（2032バイト）以下はプール割当、超過は malloc | GenericMemory 割当時 |
| BR-90-03 | Union メモリレイアウト | Union 要素型の場合、データ領域の後に1要素あたり1バイトのタグバイトが追加される | arrayelem_isunion フラグ時 |
| BR-90-04 | アライメント | プール割当のメモリは JL_SMALL_BYTE_ALIGNMENT でアラインされる | プール割当時 |
| BR-90-05 | オーバーフロー検出 | __builtin_mul_overflow を使用して積のオーバーフローを検出 | 次元計算・バイト数計算時 |
| BR-90-06 | MAXINTVAL 制限 | 配列サイズは `(((size_t)-1)>>1)` を超えてはならない | 全割当時 |

### 計算ロジック

必要バイト数の計算:
```c
nbytes = nel * elsz;
if (isunion) nbytes += nel;  // タグバイト
tot = nbytes + LLT_ALIGN(sizeof(jl_genericmemory_t), JL_SMALL_BYTE_ALIGNMENT);
pooled = (tot <= GC_MAX_SZCLASS);
```

Array ヘッダサイズ:
```c
tsz = sizeof(jl_array_t) + ndims * sizeof(size_t);
```

## データベース操作仕様

本機能はデータベースを使用しない。

### 操作別データベース影響一覧

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentError | 配列次元数が不正（オーバーフロー、MAXINTVAL超過） | jl_exceptionf で ArgumentError をスロー |
| - | ArgumentError | GenericMemory のサイズが不正 | jl_exceptionf で ArgumentError をスロー |
| - | OutOfMemoryError | メモリ割当失敗 | GCによる回収試行後、最終的に OOM abort |

### リトライ仕様

メモリ割当失敗時はGCを実行して再試行する（GCランタイム側の処理）。

## トランザクション仕様

該当なし。メモリ割当は個別のアトミック操作。

## パフォーマンス要件

- プールアロケータによる小オブジェクト割当は O(1)（アモータイズド）
- ゼロ初期化は memset で実行され、大きな配列では時間がかかる場合がある
- `jl_alloc_genericmemory_unchecked` はコード生成器から呼ばれる最小化されたパス

## セキュリティ考慮事項

- メモリの未初期化はセキュリティリスクとなりうる（`undef` 初期化の配列は未初期化メモリを露出）
- `unsafe_wrap` による外部メモリの参照は境界チェックが行われない

## 備考

- `jl_alloc_genericmemory_unchecked` は「ONLY USE FROM CODEGEN」とコメントされており、部分初期化のみ行う
- `jl_array_elsize` マクロは配列要素サイズの取得ショートカット
- `jl_array_typetagdata` はUnion配列のタグバイト領域へのポインタを返す

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | genericmemory.c | `src/genericmemory.c` | jl_genericmemory_typetagdata（20-25行目）: Union メモリのタグバイトアクセス |
| 1-2 | array.c | `src/array.c` | MAXINTVAL 定数（19行目）: サイズ上限 |
| 1-3 | array.c | `src/array.c` | jl_array_elsize マクロ（51行目）: 要素サイズ取得 |
| 1-4 | array.c | `src/array.c` | _new_array 関数（59-73行目）: Array オブジェクトの内部構造 |

**読解のコツ**: Array は GenericMemory をラップする薄いオブジェクトである。`ref.mem` がバッキングメモリ、`ref.ptr_or_offset` がデータへのポインタまたはオフセット、`dimsize[]` が各次元のサイズを保持する。Union 型の配列では `ptr_or_offset` がオフセットとして使用される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | array.c | `src/array.c` | jl_array_validate_dims（21-33行目）: 次元検証とオーバーフローチェック |
| 2-2 | array.c | `src/array.c` | new_array（75-80行目）: Array 生成の外部エントリーポイント |
| 2-3 | genericmemory.c | `src/genericmemory.c` | jl_alloc_genericmemory（75-80行目）: GenericMemory の公開API |

**主要処理フロー**:
1. **21-33行目**: jl_array_validate_dims: __builtin_mul_overflow で全次元の積を安全に計算
2. **59-73行目**: _new_array: jl_gc_alloc でヘッダを割当、mem/ptr_or_offset/dimsize を設定
3. **75-80行目**: new_array: validate → GenericMemory割当 → _new_array の順で呼び出し

#### Step 3: GenericMemory 割当の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | genericmemory.c | `src/genericmemory.c` | jl_alloc_genericmemory_unchecked（30-53行目）: コード生成器用の最小パス |
| 3-2 | genericmemory.c | `src/genericmemory.c` | _new_genericmemory_（55-73行目）: 内部のGenericMemory生成ロジック |

**主要処理フロー**:
1. **30-33行目**: 必要バイト数 + ヘッダサイズを計算
2. **34行目**: `pooled = tot <= GC_MAX_SZCLASS` でプール/malloc の判定
3. **37-38行目**: プールでない場合は `jl_gc_managed_malloc` で確保
4. **41行目**: `jl_gc_alloc` で GenericMemory ヘッダを割当
5. **47-48行目**: 非プールの場合は `jl_gc_track_malloced_genericmemory` で GC 追跡登録

#### Step 4: GC との連携を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | gc-common.c | `src/gc-common.c` | GC メトリクス（18行目）: gc_num グローバル |
| 4-2 | gc-common.c | `src/gc-common.c` | GC コールバック（29-78行目）: 割当通知コールバック |

### プログラム呼び出し階層図

```
Julia コード: Array{T}(undef, dims...)
    |
    +-- jl_alloc_array_nd (array.c)
    |      +-- jl_array_validate_dims
    |      +-- jl_alloc_genericmemory (genericmemory.c)
    |      |      +-- _new_genericmemory_
    |      |      |      +-- jl_alloc_genericmemory_unchecked
    |      |      |      |      +-- jl_gc_alloc (プール割当)
    |      |      |      |      +-- jl_gc_managed_malloc (大オブジェクト)
    |      |      |      +-- memset (ゼロ初期化)
    |      +-- _new_array
    |             +-- jl_gc_alloc (Array ヘッダ割当)
```

### データフロー図

```
[入力]                         [処理]                       [出力]

型 + 次元          --> jl_array_validate_dims       --> nel (総要素数)
nel + 要素サイズ   --> _new_genericmemory_          --> GenericMemory
GenericMemory      --> _new_array                    --> Array
Array              --> Julia ヒープ                  --> GC 管理下の Array
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| array.c | `src/array.c` | ソース | Array の生成・操作プリミティブ |
| genericmemory.c | `src/genericmemory.c` | ソース | GenericMemory の生成・操作 |
| gc-common.c | `src/gc-common.c` | ソース | GC 共通処理（メトリクス、コールバック） |
| gc-stock.c | `src/gc-stock.c` | ソース | jl_gc_alloc / jl_gc_managed_malloc 等の割当実装 |
| julia.h | `src/julia.h` | ヘッダ | jl_array_t / jl_genericmemory_t の型定義 |
| julia_internal.h | `src/julia_internal.h` | ヘッダ | 内部メモリ管理API |
