# 機能設計書 95-初期化

## 概要

本ドキュメントは、Julia言語ランタイムの初期化機能の設計を記述する。プロセス起動からJuliaコード実行可能状態に至るまでのシステム初期化、リソース確保、サブシステムのセットアップを担う機能である。

### 本機能の処理概要

**業務上の目的・背景**：Juliaランタイムは、メモリ管理、スレッディング、JITコンパイラ、シグナル処理、I/O、システムイメージのロードなど多数のサブシステムを初期化する必要がある。`jl_init_`関数はこれらの初期化を正しい順序で実行し、Julia REPLやスクリプト実行の準備を整える。

**機能の利用シーン**：`julia`コマンドの起動時、Juliaを組み込みで使用するアプリケーションからの`jl_init`呼び出し時、テスト実行環境のセットアップ時。

**主要な処理内容**：
1. プラットフォーム固有の初期化（スタック制限、ページサイズ、サニタイザー準備）
2. グローバルミューテックスの初期化
3. libuvイベントループの初期化と標準I/Oのセットアップ
4. シグナルハンドラの登録
5. タスクシステムとスレッディングインフラの初期化
6. GCの初期化
7. システムイメージの解析とJITターゲットの初期化
8. コード生成バックエンドの初期化
9. モジュール初期化順序の実行
10. 終了処理（atexit hook, cleanup）

**関連システム・外部連携**：libuv（I/Oイベントループ）、LLVMコード生成、OS固有の動的リンク機構（dlopen/LoadLibrary）。

**権限による制御**：特にない。`jl_init_`はプロセスのメインスレッドから1回のみ呼び出される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ランタイム内部機能のため関連画面なし |

## 機能種別

ランタイム初期化 / システムセットアップ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| sysimage | jl_image_buf_t | Yes | システムイメージバッファ（kind: NONE/JI/SO） | kindが有効な値 |

### 入力データソース

- コマンドラインオプション（`jl_options`構造体に格納済み）
- システムイメージファイル（.so/.ji）
- 環境変数

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 初期化済みランタイム | - | GC、スレッド、JITが動作可能な状態 |
| jl_main_module | jl_module_t* | Mainモジュール |
| jl_base_module | jl_module_t* | Baseモジュール |

### 出力先

グローバル変数としてランタイム全体に公開

## 処理フロー

### 処理シーケンス

```
1. TLSコールバック確定（jl_get_pgcstack）
2. 基盤初期化
   ├─ uv_mutex_init: シンボルテーブルロック、タスクロック、プロファイルロック
   ├─ jl_init_profile_lock: プロファイルバッファロック
   ├─ jl_init_stackwalk / rec_backtrace: スタックウォーク初期化
   └─ libsupport_init
3. セーフポイントとメモリ初期化
   ├─ jl_safepoint_init
   ├─ jl_page_size設定
   └─ init_global_mutexes
4. I/O初期化
   ├─ uv_default_loop: libuvイベントループ
   ├─ jl_init_uv
   └─ init_stdio: STDIN/STDOUT/STDERR
5. シグナルと浮動小数点初期化
   ├─ restore_fp_env
   ├─ restore_signals
   └─ jl_init_intrinsic_properties
6. スレッドとGC初期化
   ├─ jl_init_tasks / jl_init_threading / jl_init_threadinginfra
   ├─ jl_install_default_signal_handlers
   └─ jl_gc_init
7. ルートタスク初期化
   ├─ jl_init_threadtls
   ├─ jl_init_root_task
   └─ jl_init_box_caches / jl_init_common_symbols
8. コード生成とシステムイメージ（_finish_jl_init_）
   ├─ jl_init_processor_sysimg
   ├─ jl_init_codegen
   ├─ jl_restore_system_image（sysimageあり）/ jl_init_types（なし）
   ├─ jl_init_flisp / jl_init_serializer
   └─ jl_start_threads / jl_start_gc_threads
9. モジュール初期化（__init__メソッドの実行）
10. SIGINTハンドラの登録
```

### フローチャート

```mermaid
flowchart TD
    A[jl_init_開始] --> B[TLS/ロック初期化]
    B --> C[libsupport/スタックウォーク初期化]
    C --> D[セーフポイント/ページサイズ初期化]
    D --> E[libuv/stdio初期化]
    E --> F[シグナルハンドラ登録]
    F --> G[スレッド/GC初期化]
    G --> H[ルートタスク作成]
    H --> I[_finish_jl_init_]
    I --> J{sysimage.kind?}
    J -->|SO/JI| K[jl_restore_system_image]
    J -->|NONE| L[jl_init_types + boot.jl]
    K --> M[jl_init_codegen]
    L --> M
    M --> N[スレッド/GCスレッド起動]
    N --> O[モジュール__init__実行]
    O --> P[SIGINTハンドラ登録]
    P --> Q[初期化完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-95-01 | 初期化順序 | サブシステムは依存関係に基づく厳密な順序で初期化される | jl_init_ |
| BR-95-02 | sysimageなし時のフォールバック | sysimageが提供されない場合、jl_init_typesとboot.jlのロードで最小環境を構築 | sysimage.kind == NONE |
| BR-95-03 | Base不在時のスレッド制限 | jl_base_moduleがNULLの場合、スレッド数を1に制限 | 614-622行目 |
| BR-95-04 | handle_signals制御 | --handle-signals=noの場合、シグナルハンドラを登録しない | jl_options.handle_signals |
| BR-95-05 | サニタイザー対応 | ASAN/MSAN有効時はスタックサイズを64MBに拡大 | _COMPILER_ASAN/MSAN_ENABLED_ |

### 計算ロジック

特になし

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | stdio初期化失敗 | uv_tty_init/uv_pipe_initのエラー | jl_errorfで詳細メッセージ出力 |
| - | 浮動小数点環境エラー | jl_set_zero_subnormals/jl_set_default_nans失敗 | jl_errorで即座に終了 |
| - | handle-signals=noとマルチスレッド | --handle-signals=noかつスレッド>1 | jl_errorで即座に終了 |

### リトライ仕様

初期化失敗はリカバリ不可能であり、リトライはない。

## トランザクション仕様

該当なし

## パフォーマンス要件

- JL_TIMING(JULIA_INIT)で初期化全体の時間を計測可能
- システムイメージの.soファイル使用時はmmap/dlopen経由で高速ロード

## セキュリティ考慮事項

- 動的ライブラリハンドル（jl_libjulia_internal_handle等）は起動時に解決・保持される
- Windowsではntdll.dll、kernel32.dll等のシステムDLLハンドルを明示的に取得

## 備考

- `jl_atexit_hook`が終了時のクリーンアップを担当し、atexit/postoutputフック実行、GCファイナライザ実行、libuv shutdown、スレッド停止を行う
- `jl_exit`（正常終了）と`jl_raise`（シグナル終了）の2つの終了パスが存在

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | jloptions.h | `src/jloptions.h` | jl_options_t構造体（コマンドラインオプション） |
| 1-2 | julia.h | `src/julia.h` | jl_image_buf_t, jl_image_t構造体（システムイメージ） |

**読解のコツ**: `jl_options`グローバル変数にはコマンドラインで指定された全てのオプションが格納されている。初期化コードは随所でこの値を参照する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | init.c | `src/init.c` | jl_init_関数（691行目）がメインの初期化エントリーポイント |

**主要処理フロー**:
1. **698行目**: `jl_get_pgcstack()` - TLS確定
2. **701-705行目**: シンボルテーブル・タスク・プロファイルロック初期化
3. **718-729行目**: libuv/stdio初期化
4. **730-732行目**: 浮動小数点環境・シグナル復元
5. **765-773行目**: スレッド/タスク/シグナルハンドラ/GC初期化
6. **796行目**: `jl_init_root_task` - ルートタスク作成
7. **801行目**: `_finish_jl_init_`呼び出し

#### Step 3: システムイメージロードを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | init.c | `src/init.c` | _finish_jl_init_関数（576行目） |

**主要処理フロー**:
- **587行目**: `jl_init_processor_sysimg` - プロセッサ初期化とイメージ解析
- **589行目**: `jl_init_codegen` - コード生成バックエンド初期化
- **591-600行目**: システムイメージ復元 or 最小環境構築
- **623-625行目**: `jl_start_threads`/`jl_start_gc_threads` - スレッド起動
- **630-639行目**: モジュール`__init__`メソッドの実行

#### Step 4: 終了処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | init.c | `src/init.c` | jl_atexit_hook（233行目）、jl_exit（198行目）、jl_raise（207行目） |

**主要処理フロー**:
- **233-356行目**: atexit hook全体（_atexitフック実行、コンパイラ出力、GCファイナライザ、libuv shutdown）
- **198-202行目**: `jl_exit` - 正常終了パス
- **207-231行目**: `jl_raise` - シグナル終了パス

### プログラム呼び出し階層図

```
jl_init_ (init.c:691)
    +-- jl_get_pgcstack()
    +-- uv_mutex_init(&symtab_lock)
    +-- jl_init_profile_lock()
    +-- libsupport_init()
    +-- jl_safepoint_init()
    +-- init_global_mutexes()
    +-- uv_default_loop()
    +-- init_stdio()
    |       +-- init_stdio_handle("stdin/stdout/stderr")
    +-- restore_fp_env()
    +-- restore_signals()
    +-- jl_init_tasks()
    +-- jl_init_threading()
    +-- jl_init_threadinginfra()
    +-- jl_install_default_signal_handlers()
    +-- jl_gc_init()
    +-- jl_init_threadtls(0)
    +-- jl_init_root_task(ptls, stack_lo, stack_hi)
    +-- _finish_jl_init_(sysimage, ptls, ct)
            +-- jl_init_processor_sysimg()
            +-- jl_init_codegen()
            +-- jl_restore_system_image() / jl_init_types()
            +-- jl_init_flisp()
            +-- jl_init_serializer()
            +-- jl_start_threads()
            +-- jl_start_gc_threads()
            +-- jl_module_run_initializer() [each module]
            +-- jl_install_sigint_handler()
```

### データフロー図

```
[入力]                    [処理]                          [出力]

jl_options ─────────────> jl_init_ ──────────────────> 初期化済みランタイム
  (コマンドライン設定)        |
                             |
sysimage (JI/SO/NONE) ──> _finish_jl_init_ ──────────> jl_base_module
                             |                          jl_core_module
                             |                          jl_main_module
環境変数 ───────────────> 各init関数 ──────────────> サブシステム初期化
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| init.c | `src/init.c` | ソース | 初期化処理の中核実装 |
| jloptions.c | `src/jloptions.c` | ソース | コマンドラインオプション解析 |
| jloptions.h | `src/jloptions.h` | ヘッダ | jl_options_t構造体定義 |
| jlapi.c | `src/jlapi.c` | ソース | 公開API（jl_init等のラッパー） |
| julia.h | `src/julia.h` | ヘッダ | 公開型定義 |
| julia_internal.h | `src/julia_internal.h` | ヘッダ | 内部関数宣言 |
