# 通知設計書 15-LLVMトリプル検出失敗警告

## 概要

本ドキュメントは、Juliaの`BinaryPlatforms`モジュールにおいて、libLLVMから`llvm::sys::getProcessTriple()`シンボルを検出できない場合に発行される警告通知の設計仕様を記述する。

### 本通知の処理概要

本通知は、`detect_cxxstring_abi()`関数がlibLLVMライブラリ内のC++文字列ABI情報を検出するために`getProcessTriple`シンボルを探索した結果、cxx03形式もcxx11形式も見つからなかった場合に警告を発行する処理である。

**業務上の目的・背景**：JuliaはLLVMをバックエンドコンパイラとして使用しており、C++文字列ABI（cxx03 / cxx11）の情報はバイナリ互換性の判定に必要である。この情報はHostPlatformの構築に使用され、バイナリアーティファクトの選択に影響する。ABIが検出できない場合、互換性のないバイナリが選択される可能性があるため、開発者に警告を通知する。特にg++でコンパイルされた環境でのABI不一致問題を検出するために重要である。

**通知の送信タイミング**：`detect_cxxstring_abi()`が呼び出され、libLLVMを開いた後に`_ZN4llvm3sys16getProcessTripleEv`（cxx03）および`_ZN4llvm3sys16getProcessTripleB5cxx11Ev`（cxx11）の両方のシンボルが見つからなかった場合に送信される。

**通知の受信者**：Juliaプロセスを起動したユーザーおよび開発者。標準エラー出力を通じてログ表示される。

**通知内容の概要**：「Unable to find llvm::sys::getProcessTriple() in libLLVM!」というメッセージが出力される。

**期待されるアクション**：ユーザーは、JuliaのビルドがLLVMと正しくリンクされているか確認する。libLLVMのバージョンやビルド構成が正しいか、またはclangでビルドされた環境では本警告が問題にならないことを理解する。

## 通知種別

ログ（Warn） - Juliaの標準ログシステム（`@warn`マクロ）を通じたコンソール警告出力

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（`@warn`マクロによる即座のログ出力） |
| 優先度 | 中（バイナリ互換性に影響する可能性） |
| リトライ | なし |

### 送信先決定ロジック

Juliaの標準ロギングフレームワーク（`CoreLogging`）を通じて、現在アクティブなロガーに送信される。デフォルトでは標準エラー出力（`stderr`）に出力される。

## 通知テンプレート

### メール通知の場合

該当なし（コンソールログ出力のみ）

### 本文テンプレート

```
Warning: Unable to find llvm::sys::getProcessTriple() in libLLVM!
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| （なし） | 固定メッセージのため変数なし | - | - |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼出し | `detect_cxxstring_abi()`の実行 | `dlsym_e`でcxx03シンボルもcxx11シンボルも見つからない | libLLVM内にgetProcessTripleシンボルが存在しない |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| cxx03シンボル検出 | `_ZN4llvm3sys16getProcessTripleEv`が見つかった場合 |
| cxx11シンボル検出 | `_ZN4llvm3sys16getProcessTripleB5cxx11Ev`が見つかった場合 |
| libstdc++未リンク | `libstdc++`がロードされていない場合（clangビルド時）、関数が早期にnothingを返し警告は発行されない |
| libLLVM未検出 | libLLVMが開けない場合はerror()が発生し、@warnではなく例外が投げられる |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[detect_cxxstring_abi呼出し] --> B{libstdc++がロード済み?}
    B -->|No| C[return nothing - 警告なし]
    B -->|Yes| D[open_libllvm]
    D --> E{libLLVMを開けた?}
    E -->|No| F[error: Unable to open libLLVM]
    E -->|Yes| G{cxx03シンボル検出?}
    G -->|Yes| H[return 'cxx03']
    G -->|No| I{cxx11シンボル検出?}
    I -->|Yes| J[return 'cxx11']
    I -->|No| K["@warn 'Unable to find getProcessTriple'"]
    K --> L[return nothing]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（データベースを使用しない）

### 更新テーブル一覧

該当なし（データベースを使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| libLLVM未検出 | libLLVM/LLVM/libLLVMSupportのいずれも開けない | `error("Unable to open libLLVM!")`で例外を投げる |
| シンボル未検出 | getProcessTripleシンボルが見つからない | `@warn`を出力し`nothing`を返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（通常はJulia起動時に1回のみ呼出される）

## セキュリティ考慮事項

本通知にはシステム上のライブラリ情報は含まれない（固定メッセージ）。セキュリティ上の考慮事項は特にない。

## 備考

- `detect_cxxstring_abi()`は`libstdc++`がロードされていない場合（clangビルド時）は早期リターンし、本警告は発行されない
- `open_libllvm`は複数のライブラリ名（`Base.libllvm_name`, `"libLLVM"`, `"LLVM"`, `"libLLVMSupport"`）を順に試行する
- `dlsym_e`は`dlsym`のエラー非投入版で、シンボルが見つからない場合に`C_NULL`を返す
- C++のname manglingにより、cxx03とcxx11のABIでシンボル名が異なる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

C++文字列ABIの概念とシンボル名の関係を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | binaryplatforms.jl | `base/binaryplatforms.jl` | `Platform`構造体のタグシステムにおける`cxxstring_abi`タグの役割 |

**読解のコツ**: C++のname manglingにより`llvm::sys::getProcessTriple()`は、cxx03 ABIでは`_ZN4llvm3sys16getProcessTripleEv`、cxx11 ABIでは`_ZN4llvm3sys16getProcessTripleB5cxx11Ev`として公開される。

#### Step 2: エントリーポイントを理解する

`detect_cxxstring_abi()`関数の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | binaryplatforms.jl | `base/binaryplatforms.jl` | `detect_cxxstring_abi()`（967-1000行目）- 前提条件チェックとlibLLVMの操作 |

**主要処理フロー**:
1. **969行目**: `filter!(x -> occursin("libstdc++", x), Libdl.dllist())` - libstdc++のロード確認
2. **970-973行目**: 空なら`nothing`を返す（clangビルド時）
3. **975-987行目**: `open_libllvm`ヘルパー関数 - 複数のライブラリ名でlibLLVMを開く
4. **989-999行目**: `do hdl`ブロック - シンボル探索と結果判定

#### Step 3: シンボル探索と警告発行を理解する

libLLVM内でのシンボル探索とABI判定ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | binaryplatforms.jl | `base/binaryplatforms.jl` | 991-998行目のif/elseif/else分岐 |

**主要処理フロー**:
- **991行目**: `Libdl.dlsym_e(hdl, "_ZN4llvm3sys16getProcessTripleEv")` - cxx03シンボル探索
- **993行目**: `Libdl.dlsym_e(hdl, "_ZN4llvm3sys16getProcessTripleB5cxx11Ev")` - cxx11シンボル探索
- **996行目**: 両方見つからない場合に`@warn`を発行

### プログラム呼び出し階層図

```
host_triplet()                          [base/binaryplatforms.jl:1010]
    |
    +-- detect_cxxstring_abi()          [base/binaryplatforms.jl:967]
            |
            +-- Libdl.dllist()          [ロード済みライブラリ一覧]
            |
            +-- open_libllvm()          [base/binaryplatforms.jl:975]
            |       |
            |       +-- Libdl.dlopen_e() [4つのライブラリ名を試行]
            |       +-- Libdl.dlclose()
            |
            +-- Libdl.dlsym_e()         [シンボル探索]
            |
            +-- @warn (条件付き)         [CoreLogging]
```

### データフロー図

```
[入力]                      [処理]                              [出力]

Libdl.dllist()         --> libstdc++フィルタ           --> ロード判定
                            [binaryplatforms.jl:969]

libLLVM候補リスト      --> open_libllvm               --> ライブラリハンドル
                            [binaryplatforms.jl:975]

ライブラリハンドル     --> dlsym_e (cxx03)             --> シンボルポインタ or C_NULL
                            [binaryplatforms.jl:991]

ライブラリハンドル     --> dlsym_e (cxx11)             --> シンボルポインタ or C_NULL
                            [binaryplatforms.jl:993]

判定結果              -->                              --> "cxx03" / "cxx11" / @warn+nothing
                            [binaryplatforms.jl:991-998]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| binaryplatforms.jl | `base/binaryplatforms.jl` | ソース | BinaryPlatformsモジュール、C++文字列ABI検出 |
| libdl.jl | `base/libdl.jl` | ソース | 動的ライブラリ操作API（dlopen, dlsym等） |
| logging.jl | `base/logging.jl` | ソース | `@warn`マクロの定義 |
