# 通知設計書 19-REPLインポート失敗警告

## 概要

本ドキュメントは、JuliaのREPLモジュールのインポートに失敗した場合に発行される警告通知（`@warn "Failed to import REPL"`）の設計仕様を記述する。

### 本通知の処理概要

本通知は、`load_REPL`関数がREPL標準ライブラリの読み込みを試行し、例外が発生した場合に警告を発行する処理である。REPLのロード失敗は、フォールバックREPL（基本的な入出力ループ）の使用につながる。

**業務上の目的・背景**：JuliaのREPL（Read-Eval-Print Loop）は、行編集、タブ補完、シンタックスハイライト、履歴管理、ヘルプモード等の高機能な対話型インターフェースを提供する。REPLモジュールのインポート失敗は、これらの機能が全て利用できなくなることを意味し、ユーザーの開発体験に大きな影響を与える。本警告は、インポート失敗をユーザーに通知し、基本フォールバックREPLへの切り替えが行われたことを示す。

**通知の送信タイミング**：`load_REPL`関数内で`require_stdlib`を呼び出してREPLパッケージを読み込む際に例外が発生した場合に送信される。通常は対話型セッション開始時（`run_main_repl`から呼出される）に呼び出される。

**通知の受信者**：Juliaの対話型セッションを開始したユーザー。標準エラー出力を通じてログ表示される。

**通知内容の概要**：「Failed to import REPL」というメッセージとともに、発生した例外とそのバックトレースが付加情報として出力される。

**期待されるアクション**：ユーザーは、Juliaのインストールが正常であるか確認する。`LOAD_PATH`の設定、stdlibディレクトリの存在、パッケージの整合性を確認する。必要に応じてJuliaを再インストールする。フォールバックREPLでは基本的なJulia操作は可能だが、行編集機能等は利用できない。

## 通知種別

ログ（Warn） - Juliaの標準ログシステム（`@warn`マクロ）を通じたコンソール警告出力

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（`@warn`マクロによる即座のログ出力） |
| 優先度 | 高（REPL機能の全面的な利用不可につながる） |
| リトライ | なし |

### 送信先決定ロジック

Juliaの標準ロギングフレームワーク（`CoreLogging`）を通じて、現在アクティブなロガーに送信される。デフォルトでは標準エラー出力（`stderr`）に出力される。

## 通知テンプレート

### メール通知の場合

該当なし（コンソールログ出力のみ）

### 本文テンプレート

```
Warning: Failed to import REPL
  exception = ({例外オブジェクト}, {バックトレース})
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| exception | 発生した例外とバックトレースのタプル | `(ex, catch_backtrace())` | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| モジュールロード | `require_stdlib(PkgId(..., "REPL"))`の実行 | 例外が発生した場合 | REPL標準ライブラリのロードに失敗 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ロード成功 | `require_stdlib`が正常に完了した場合 |
| 非対話型モード | 対話型モードでない場合は`load_REPL`自体が呼ばれない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[load_REPL呼出し] --> B[try: require_stdlib REPL]
    B -->|成功| C[return REPLモジュール]
    B -->|例外| D["@warn 'Failed to import REPL'"]
    D --> E[return nothing]
    E --> F[呼出し元でフォールバックREPLに切替]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（データベースを使用しない）

### 更新テーブル一覧

該当なし（データベースを使用しない）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| パッケージ未検出 | REPL stdlibが見つからない | 警告を出力し`nothing`を返却。フォールバックREPLが使用される |
| ロードエラー | パッケージのロード中にエラーが発生 | 警告を出力し`nothing`を返却。例外とバックトレースを表示 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（対話型セッション開始時に1回のみ呼出される）

## セキュリティ考慮事項

通知には例外メッセージとバックトレースが含まれ、ファイルパスやシステム構成情報が露出する可能性がある。ただし、これは対話型セッションのローカル出力であり、通常のセキュリティリスクは低い。

## 備考

- REPLのUUIDは`0x3fa0cd96_eef1_5676_8a61_b3b8758bbffb`
- `load_REPL`の戻り値がnothingの場合、`REPL_MODULE_REF[]`はBaseのままとなり、`run_main_repl`でフォールバックREPLが使用される
- `load_REPL`は`load_InteractiveUtils`と異なり、既存ロード済みチェックを行わない（単純なtry-catchのみ）
- フォールバックREPLでは`run_fallback_repl`関数が使用され、基本的な入力→パース→評価ループのみ提供される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

REPLモジュール参照の管理方法を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | client.jl | `base/client.jl` | `REPL_MODULE_REF`グローバル変数 - REPLモジュールの参照を保持 |

**読解のコツ**: `REPL_MODULE_REF[]`が`Base`（REPLモジュールではなくBaseモジュール）を指している場合、REPLのロードに失敗したことを意味する。`run_main_repl`内の`REPL !== Base`条件でこの判定が行われる。

#### Step 2: エントリーポイントを理解する

`load_REPL`関数の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | client.jl | `base/client.jl` | `load_REPL`（429-437行目）- シンプルなtry-catch構造 |

**主要処理フロー**:
1. **431行目**: `try` - ロード試行開始
2. **432行目**: `return Base.require_stdlib(PkgId(UUID(0x3fa0cd96_eef1_5676_8a61_b3b8758bbffb), "REPL"))` - REPLのロードと即座の返却
3. **433行目**: `catch ex` - 例外キャッチ
4. **434行目**: `@warn "Failed to import REPL" exception=(ex, catch_backtrace())` - 警告出力
5. **436行目**: `return nothing` - nothing返却

#### Step 3: 呼出し元と後続処理を理解する

REPLロード失敗後のフォールバック処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | client.jl | `base/client.jl` | `run_main_repl`（515-534行目）- REPLロード失敗時のフォールバック判定 |
| 3-2 | client.jl | `base/client.jl` | `run_fallback_repl`（442-481行目）- 基本REPLの実装 |

**主要処理フロー**:
- **517行目**: `load_InteractiveUtils()` - まずInteractiveUtilsをロード
- **519-522行目**: `REPL_MODULE_REF[]`チェックと`load_REPL()`呼出し
- **524-532行目**: `REPL !== Base`の判定でフォールバックを決定

### プログラム呼び出し階層図

```
run_main_repl(interactive, quiet, banner, history_file) [base/client.jl:515]
    |
    +-- load_InteractiveUtils()                          [base/client.jl:413]
    |
    +-- load_REPL()                                      [base/client.jl:429]
    |       |
    |       +-- require_stdlib(PkgId(..., "REPL"))       [base/loading.jl]
    |       +-- @warn (例外時)                           [CoreLogging]
    |
    +-- [REPL === Base?]
        |
        +-- run_std_repl(REPL, ...)                     [成功時]
        +-- run_fallback_repl(interactive)              [失敗時]
```

### データフロー図

```
[入力]                          [処理]                           [出力]

PkgId(UUID, "REPL")       --> require_stdlib             --> Moduleオブジェクト
                                [base/loading.jl]             or 例外

Module / nothing           --> REPL_MODULE_REF更新       --> REPL or Base
                                [client.jl:519-522]

REPL_MODULE_REF           --> Base判定                   --> std_repl or fallback
                                [client.jl:525]

例外 + backtrace          --> @warn                     --> stderr出力
                                [client.jl:434]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| client.jl | `base/client.jl` | ソース | `load_REPL`関数とREPL起動制御 |
| loading.jl | `base/loading.jl` | ソース | `require_stdlib`関数の定義 |
| REPL.jl | `stdlib/REPL/src/REPL.jl` | ソース | REPLモジュール本体 |
| logging.jl | `base/logging.jl` | ソース | `@warn`マクロの定義 |
