# 通知設計書 24-非推奨API使用警告

## 概要

本ドキュメントは、Juliaにおいて非推奨（deprecated）となったAPIが使用された際に出力される警告通知の設計を記述する。

### 本通知の処理概要

本通知は、`@deprecate` マクロまたは `depwarn` 関数を通じて、非推奨のAPI（関数、メソッド、型、バインディング）が呼び出された際に発生する警告である。

**業務上の目的・背景**：Juliaは後方互換性を維持しつつAPIを進化させるために、非推奨メカニズムを持つ。古いAPIを使用し続けるユーザーに対して、新しい推奨APIへの移行を促すことが本通知の目的である。これにより、将来のバージョンで削除されるAPIの使用を事前に検出し、スムーズなマイグレーションを支援する。

**通知の送信タイミング**：非推奨のAPIが呼び出された時点。ただし、Julia起動オプション `--depwarn` の設定により挙動が異なる。
- `--depwarn=yes`: 警告を出力
- `--depwarn=no` (デフォルト): 警告を抑止
- `--depwarn=error`: 警告の代わりに `ErrorException` をスロー

**通知の受信者**：非推奨APIを呼び出したJuliaユーザー・パッケージ開発者。

**通知内容の概要**：非推奨となったAPIの名前と、推奨される代替APIを表示する。

**期待されるアクション**：ユーザーは非推奨APIの使用箇所を特定し、推奨される新しいAPIに移行する。

## 通知種別

ログ（Warn / BelowMinLevel） -- Julia標準ロギングフレームワーク（`@logmsg` マクロ）によるコンソール出力。ログレベルは `--depwarn` オプションに依存する。

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（API呼び出し時に即時処理） |
| 優先度 | 設定依存（Warn または BelowMinLevel） |
| リトライ | 無し |

### 送信先決定ロジック

`@logmsg` を使用し、`_module` は呼び出し元のモジュール（バックトレースから特定）、`_file` と `_line` は呼び出し元のファイルと行番号が設定される。デフォルトの `ConsoleLogger` では `stderr` に出力される。

## 通知テンプレート

### メール通知の場合

該当なし（コンソールログ出力）

### 本文テンプレート

**`@deprecate` マクロ経由の場合**
```
┌ Warning: `old_function(args...)` is deprecated, use `new_function(args...)` instead.
│   caller = ip:0x...
└ @ SomeModule path/to/file.jl:123
```

**`depwarn` 関数直接呼び出しの場合**
```
┌ Warning: {custom deprecation message}
│   caller = ip:0x...
└ @ SomeModule path/to/file.jl:123
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| msg | 非推奨メッセージ | `@deprecate` マクロが自動生成、または `depwarn` の第1引数 | Yes |
| funcsym | 非推奨関数のシンボル名 | `@deprecate` マクロが自動設定、または `depwarn` の第2引数 | Yes |
| caller | 呼び出し元の情報 | `backtrace()` から `firstcaller` 関数で特定 | Yes |
| _module | 呼び出し元のモジュール | バックトレースの `linfo` から取得 | Yes |
| _file | 呼び出し元のファイル | バックトレースの `caller.file` | Yes |
| _line | 呼び出し元の行番号 | バックトレースの `caller.line` | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | `@deprecate` で定義されたメソッド呼び出し | `opts.depwarn == 1` (yes) | depwarn=yes の場合 |
| API呼び出し | `@deprecate` で定義されたメソッド呼び出し | `force=true` | depwarnの設定に関わらず強制的に警告 |
| バインディング参照 | `@deprecate_binding` で定義されたバインディングの参照 | C runtimeの `jl_deprecate_binding` によるフラグチェック | |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| `opts.depwarn == 0` (no) かつ `force != true` | デフォルト設定。ログレベルが `BelowMinLevel` となり、ConsoleLogger はこのレベルを表示しない |
| `maxlog=1` | 同一の `(frame, funcsym)` ペアに対して最初の1回のみ出力 |
| `funcsym === nothing` | `funcsym` が nothing の場合は `maxlog` が nothing となり毎回出力（ただし重複チェックなし） |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[非推奨API呼び出し] --> B["depwarn(msg, funcsym)"]
    B --> C["@invokelatest _depwarn(msg, funcsym, force)"]
    C --> D{"opts.depwarn == 2?"}
    D -->|Yes| E["throw(ErrorException(msg))"]
    D -->|No| F{"force || opts.depwarn == 1?"}
    F -->|Yes| G["deplevel = Warn"]
    F -->|No| H["deplevel = BelowMinLevel"]
    G --> I["backtrace()"]
    H --> I
    I --> J["firstcaller(bt, funcsym)"]
    J --> K["caller情報取得"]
    K --> L["@logmsg deplevel msg ..."]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ErrorException | `opts.depwarn == 2` (error) | 例外がスローされる。呼び出し元で捕捉が必要 |
| ログ出力失敗 | ロガーの出力先が利用不可 | Juliaランタイムのデフォルトエラー処理に委ねる |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 同一API呼び出しあたり | 最大1回（`maxlog=1` 設定による。`funcsym` が nothing の場合は制限なし） |
| _id | `(frame, funcsym)` -- フレームアドレスと関数シンボルの組み合わせで一意 |

### 配信時間帯

制限なし

## セキュリティ考慮事項

呼び出し元のファイルパスとモジュール名がログに含まれるが、これはデバッグのための標準的な情報であり、機密性は低い。`@invokelatest` を使用して、バックエッジの追加を防止し、invalidation 耐性を高めている。

## 備考

- Julia 1.5以降、`--depwarn` のデフォルト値は `no` に変更された。テスト実行時（`Pkg.test()`）では自動的に `yes` が設定される
- `@deprecate` マクロは非推奨メソッドを自動的に定義し、`depwarn` 呼び出しと新APIへの委譲を行う
- `@deprecate_binding` はモジュールバインディングの非推奨化に使用される
- `@deprecate_stdlib` は標準ライブラリに移動した機能の非推奨化に使用される
- `firstcaller` 関数はバックトレースを走査して、非推奨関数の呼び出し元を特定する
- `@nospecializeinfer` が `depwarn` に付与されており、型推論時の特殊化を抑制している

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | deprecated.jl | `base/deprecated.jl` | 22-32行目: `__next_removal_version` と `__internal_changes_list` の定義 |
| 1-2 | deprecated.jl | `base/deprecated.jl` | 126-230行目: `@deprecate` マクロの定義とドキュメント |

**読解のコツ**: `@deprecate` は2つの形式を持つ。`@deprecate old(args) new(args)` は呼び出しパターンを含む形式、`@deprecate old new` はシンボルのみの形式。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | deprecated.jl | `base/deprecated.jl` | 252-258行目: `depwarn` 関数。外部API |
| 2-2 | deprecated.jl | `base/deprecated.jl` | 259-288行目: `_depwarn` 関数。内部実装 |

#### Step 3: 通知発生箇所を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | deprecated.jl | `base/deprecated.jl` | 259-288行目: `_depwarn` 関数。`@logmsg` による通知出力 |
| 3-2 | deprecated.jl | `base/deprecated.jl` | 261-264行目: `JLOptions` からの `depwarn` フラグ取得と条件分岐 |
| 3-3 | deprecated.jl | `base/deprecated.jl` | 266-287行目: `@logmsg` 呼び出し。`_module`, `_file`, `_line`, `_id`, `_group`, `maxlog` の設定 |
| 3-4 | deprecated.jl | `base/deprecated.jl` | 290-316行目: `firstcaller` 関数。バックトレースからの呼び出し元特定 |

**主要処理フロー**:
- **252行目**: `depwarn(msg, funcsym)` が呼ばれる
- **257行目**: `@invokelatest _depwarn(msg, funcsym, force)` で内部実装に委譲
- **261行目**: `JLOptions()` からdepwarnフラグを取得
- **262-263行目**: `depwarn == 2` なら `ErrorException` をスロー
- **265行目**: `force || depwarn == 1` なら `Warn`、そうでなければ `BelowMinLevel`
- **270-271行目**: `backtrace()` からの呼び出し元特定
- **266-287行目**: `@logmsg` でログ出力

### プログラム呼び出し階層図

```
非推奨API呼び出し
    │
    └─ @deprecate で生成されたメソッド [deprecated.jl:211-217]
           │
           └─ depwarn(msg, funcsym) [deprecated.jl:252]
                  │
                  └─ @invokelatest _depwarn(msg, funcsym, force) [deprecated.jl:257]
                         │
                         ├─ JLOptions() [deprecated.jl:261]
                         │
                         ├─ backtrace() [deprecated.jl:270]
                         │
                         ├─ firstcaller(bt, funcsym) [deprecated.jl:292]
                         │      └─ StackTraces.lookup(ip) [deprecated.jl:296]
                         │
                         └─ @logmsg deplevel msg ... [deprecated.jl:266]
```

### データフロー図

```
[入力]                         [処理]                         [出力]

非推奨API呼び出し ──────────▶ @deprecate 生成メソッド
  引数 ──────────────────────▶ │
                              ├─ depwarn(msg, funcsym)
                              │
JLOptions.depwarn ───────────▶ ├─ depwarnレベル判定 ──────────▶ ErrorException (depwarn=2)
                              │                               BelowMinLevel (depwarn=0)
                              │                               Warn (depwarn=1 or force)
backtrace() ─────────────────▶ ├─ firstcaller() ────────────▶ caller情報
                              │
msg, funcsym, caller ────────▶ └─ @logmsg ───────────────────▶ stderr (via ConsoleLogger)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| deprecated.jl | `base/deprecated.jl` | ソース | 通知の発生元。`@deprecate`マクロ、`depwarn`関数、`firstcaller`関数 |
| logging.jl | `base/logging/logging.jl` | ソース | `@logmsg` マクロの実装 |
| stacktraces.jl | `base/stacktraces.jl` | ソース | `StackTraces.lookup` 関数（呼び出し元特定に使用） |
| options.jl | `src/options.h` / Julia runtime | ソース | `JLOptions` 構造体（`depwarn` フラグを含む） |
