# 通知設計書 63-Git資格情報不明属性警告

## 概要

本ドキュメントは、Julia の LibGit2 モジュールにおいて Git 資格情報の読み取り時に未知の属性が検出された場合に出力される警告通知の設計を記述する。

### 本通知の処理概要

Git 資格情報ヘルパーからのレスポンスやファイルから `GitCredential` を `read!` で読み取る際に、既知の属性（protocol, host, path, username, password, url）以外のキーが検出された場合に警告を出力する通知である。

**業務上の目的・背景**：Git の資格情報フォーマット（git-credential IOFMT）は、キーバリュー形式で定義されている。LibGit2 の Julia 実装は `protocol`、`host`、`path`、`username`、`password`、`url` の6つの属性をサポートしている。これ以外の属性が出現した場合、資格情報ヘルパーの設定ミスや互換性の問題が考えられる。この警告により開発者は未知の属性の存在を認識し、資格情報設定の見直しを行うことができる。

**通知の送信タイミング**：`Base.read!(io::IO, cred::GitCredential)` 関数内で、IO ストリームから読み取ったキーが `GIT_CRED_ATTRIBUTES` 定数に含まれず、かつ空白文字のみでもない場合に発生する。

**通知の受信者**：LibGit2 を使用して Git 操作を行う Julia プログラムの開発者。Julia のロギングフレームワークを通じて配信される。

**通知内容の概要**：`"Unknown git credential attribute found: $(repr(key))"` というメッセージで、検出された未知の属性名が repr 形式で含まれる。

**期待されるアクション**：開発者は Git 資格情報ヘルパーの設定を確認し、不正な属性が送信されていないか検証することが期待される。

## 通知種別

ログ（Warn） - Julia 標準ロギングフレームワーク `@warn` マクロによるログ出力

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（IO 読み取り中に即座にログ出力） |
| 優先度 | 中（Warn レベル） |
| リトライ | 無 |

### 送信先決定ロジック

Julia の標準ロギングフレームワークにより、現在アクティブなロガーにメッセージが配信される。デフォルトでは `stderr` に出力される。

## 通知テンプレート

### メール通知の場合

該当なし（ログ出力のみ）

### 本文テンプレート

```
Warning: Unknown git credential attribute found: "attribute_name"
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| key | 未知の属性名（repr 形式） | IO ストリームからの読み取り結果 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | `Base.read!(io, cred::GitCredential)` 呼び出し | 読み取ったキーが `GIT_CRED_ATTRIBUTES` に含まれず、かつ空白のみでもない場合 | 資格情報ストリームに未知の属性が含まれている |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| キーが既知の属性の場合 | `protocol`, `host`, `path`, `username`, `password`, `url` のいずれか |
| キーが空白文字のみの場合 | 空行やパディングの場合は無視される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[read! 呼び出し] --> B[IO からキーバリュー読み取り]
    B --> C{key == "url"?}
    C -->|Yes| D[URL をパースして GitCredential にコピー]
    C -->|No| E{key in GIT_CRED_ATTRIBUTES?}
    E -->|Yes| F[cred のフィールドに設定]
    E -->|No| G{all isspace key?}
    G -->|Yes| H[無視して次の行へ]
    G -->|No| I["@warn Unknown git credential attribute found"]
    D --> J[次の行へ]
    F --> J
    H --> J
    I --> J
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 未知の属性検出 | IO ストリームに未知のキーが含まれる場合 | `@warn` を出力し、該当属性を無視して処理を続行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

未知の属性名がログに出力されるが、属性値は出力されない。属性名に機密情報が含まれる可能性は低いが、`repr` 形式で出力されるため文字列のエスケープは適切に処理される。

## 備考

- `GIT_CRED_ATTRIBUTES` は `("protocol", "host", "path", "username", "password", "url")` として定義されている（gitcredential.jl 行3）。
- git-credential のフォーマット仕様: https://git-scm.com/docs/git-credential#IOFMT

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | gitcredential.jl | `stdlib/LibGit2/src/gitcredential.jl` | `GitCredential` 構造体（行11-27）と `GIT_CRED_ATTRIBUTES` 定数（行3） |

**読解のコツ**: `GitCredential` は mutable struct で、各フィールドは `Union{String, Nothing}` 型。`password` のみ `Union{Base.SecretBuffer, Nothing}` 型であり、セキュリティ上の配慮がなされている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | gitcredential.jl | `stdlib/LibGit2/src/gitcredential.jl` | `Base.read!(io, cred)` 関数（行109-139）。**本通知の発生箇所（行134）** |

**主要処理フロー**:
1. **行111**: IO から EOF まで行ごとに読み取り
2. **行112**: `readuntil(io, '=')` でキーを取得
3. **行113-121**: password の場合は SecretBuffer で読み取り、それ以外は通常文字列
4. **行123-128**: `url` キーの場合は URL パースして GitCredential にコピー
5. **行129-131**: 既知の属性の場合はフィールドに設定
6. **行132-135**: 未知の属性の場合 - **行134 で `@warn` 出力**

#### Step 3: 資格情報の利用フローを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | gitcredential.jl | `stdlib/LibGit2/src/gitcredential.jl` | `run!` 関数（行184-198）。資格情報ヘルパーとの通信で `read!` を使用 |
| 3-2 | gitcredential.jl | `stdlib/LibGit2/src/gitcredential.jl` | `fill!` 関数（行209）。資格情報ヘルパーの `get` 操作 |

### プログラム呼び出し階層図

```
credentials_callback(...)                        [callbacks.jl:270]
    |
    +-- authenticate_userpass(...)                [callbacks.jl:183]
            |
            +-- GitCredential(cfg, url)          [gitcredential.jl:29]
                    |
                    +-- fill!(cfg, cred)          [gitcredential.jl:141]
                            |
                            +-- fill!(helper, cred) [gitcredential.jl:209]
                                    |
                                    +-- run!(helper, "get", cred) [gitcredential.jl:184]
                                            |
                                            +-- Base.read!(p, cred) [gitcredential.jl:109]
                                                    |
                                                    +-- @warn "Unknown git credential attribute found" [行134]
```

### データフロー図

```
[入力]                        [処理]                              [出力]

IO ストリーム       ------>  readuntil(io, '=')         ------>  key: String
（ヘルパーレスポンス）          |
                             +-- key in GIT_CRED_ATTRIBUTES?
                             |       |
                             |       +-- Yes: cred.field = value
                             |       +-- No:  @warn          ------>  警告 (stderr)
                             |
                             +-- GitCredential 更新       ------>  cred (更新済)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| gitcredential.jl | `stdlib/LibGit2/src/gitcredential.jl` | ソース | GitCredential の定義と read!/write。本通知の発生元（行134） |
| callbacks.jl | `stdlib/LibGit2/src/callbacks.jl` | ソース | 資格情報コールバック。read! の呼び出し元 |
| types.jl | `stdlib/LibGit2/src/types.jl` | ソース | CredentialPayload 等の型定義 |
