# 通知設計書 72-Bool引数インデックス非推奨警告

## 概要

本ドキュメントは、配列操作関数（`deleteat!`、`splice!`）に`Bool`型の値をインデックスとして渡した場合に出力される非推奨警告の通知設計を定義する。

### 本通知の処理概要

`BitVector`および`Vector`の操作関数において、`Bool`値がインデックスとして使用された際に非推奨であることを警告し、適切な`Int`型インデックスの使用を促す通知機能である。

**業務上の目的・背景**：Juliaでは`Bool`は`Integer`のサブタイプであるため、`true`が`1`、`false`が`0`として暗黙にインデックスに使用されうる。しかし`Bool`をインデックスとして使用することは意図しないバグの原因となりやすく、型安全性の観点から非推奨とされた。将来のバージョンでは`Bool`をインデックスとして渡すとエラーとなる予定であり、開発者に早期の修正を促すために本警告が設けられている。

**通知の送信タイミング**：`deleteat!(::BitVector, ::Bool)`、`splice!(::BitVector, ::Bool)`、`deleteat!(::Vector, ::Bool)`、`splice!(::BitVector, r::Bool, ...)`のいずれかが呼び出された時点で`depwarn()`が発火する。

**通知の受信者**：`Bool`型をインデックスとして配列操作関数に渡したJuliaコードの開発者。

**通知内容の概要**：「passing Bool as an index is deprecated」というメッセージが表示され、`Bool`値のインデックス使用が非推奨であることが通知される。

**期待されるアクション**：開発者は`Bool`型の値を明示的に`Int`に変換してからインデックスとして使用するか、ロジックを見直してインデックスに`Bool`が渡されないようにすること。

## 通知種別

ログ出力（`depwarn`経由、CoreLogging.Warn レベル / `_group=:depwarn`）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（関数呼び出し時に即座に評価） |
| 優先度 | 中（`--depwarn`フラグに応じて動的に変更） |
| リトライ | なし |

### 送信先決定ロジック

`depwarn()`関数を通じてJuliaの標準ログシステムに出力される。`--depwarn`オプションの設定に従う。同一コールサイトからの重複出力は`maxlog=1`により抑制される。

## 通知テンプレート

### メール通知の場合

該当なし（ログ出力のみ）

### 本文テンプレート

```
passing Bool as an index is deprecated
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | - |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| msg | 固定メッセージ「passing Bool as an index is deprecated」 | ハードコード | Yes |
| funcsym | 呼び出し元関数名（`:deleteat!`または`:splice!`） | 関数内でハードコード | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | `deleteat!(::BitVector, i::Integer)` where `i isa Bool` | `--depwarn=yes` | bitarray.jl:941 |
| API呼び出し | `splice!(::BitVector, i::Integer)` where `i isa Bool` | `--depwarn=yes` | bitarray.jl:1064 |
| API呼び出し | `splice!(::BitVector, r::Union{...}, ins)` where `r isa Bool` | `--depwarn=yes` | bitarray.jl:1077 |
| API呼び出し | `deleteat!(::Vector, i::Integer)` where `i isa Bool` | `--depwarn=yes` | array.jl:1869 |
| API呼び出し | `deleteat!(::Vector, r::AbstractUnitRange)` where `first(r) isa Bool` | `--depwarn=yes` | array.jl:1879 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| `--depwarn=no`（デフォルト） | depwarnのデフォルト動作により抑制 |
| `i`が`Bool`でない場合 | `i isa Bool`チェックにより通知をスキップ |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[deleteat!/splice!呼び出し] --> B{引数 i isa Bool?}
    B -->|Yes| C[depwarn発火]
    B -->|No| D[通常処理続行]
    C --> E[Int変換して処理続行]
    D --> E
    E --> F[配列操作実行]
    F --> G[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | ログ出力のみのためDB参照なし |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | ログ出力のみのためDB更新なし |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ErrorException | `--depwarn=error`設定時にBoolインデックスを使用 | depwarnがエラーとして送出される |
| BoundsError | 変換後のIntインデックスが範囲外の場合 | depwarnとは無関係に通常のバウンドチェックで送出 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（depwarnのmaxlog=1により自然に制限） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（関数呼び出し時に即座に通知）

## セキュリティ考慮事項

- 特段のセキュリティリスクはない。通知メッセージに機密情報は含まれない

## 備考

- この非推奨警告は将来のJuliaバージョンで`Bool`インデックスが完全に禁止された際に削除される予定
- `i isa Bool`チェックは型推論による最適化で、`Bool`でない場合はゼロコストで通過する
- `BitVector`と`Vector`の両方で同じメッセージが使用される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

`BitVector`と`Vector`は異なるデータ構造だが、どちらも同じパターンでBoolインデックスの非推奨チェックを行っている。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | bitarray.jl | `base/bitarray.jl` | `BitVector`のデータ構造と操作関数の概要 |
| 1-2 | array.jl | `base/array.jl` | `Vector`のデータ構造と操作関数の概要 |

**読解のコツ**: Juliaの型階層で`Bool <: Integer`であることが、この問題の根本原因。`isa`チェックで型を検査している。

#### Step 2: エントリーポイントを理解する

各関数内でのBoolチェックパターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | bitarray.jl | `base/bitarray.jl` | `deleteat!(::BitVector, ::Integer)` 940-947行目 |
| 2-2 | bitarray.jl | `base/bitarray.jl` | `splice!(::BitVector, ::Integer)` 1060-1072行目 |
| 2-3 | bitarray.jl | `base/bitarray.jl` | `splice!(::BitVector, r, ins)` 1076-1079行目 |
| 2-4 | array.jl | `base/array.jl` | `deleteat!(::Vector, ::Integer)` 1868-1872行目 |
| 2-5 | array.jl | `base/array.jl` | `deleteat!(::Vector, ::AbstractUnitRange)` 1874-1883行目 |

**主要処理フロー**:
1. **bitarray.jl:941**: `i isa Bool && depwarn("passing Bool as an index is deprecated", :deleteat!)`
2. **bitarray.jl:942**: `i = Int(i)` でBoolをIntに変換して処理続行

#### Step 3: depwarnメカニズムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | deprecated.jl | `base/deprecated.jl` | `depwarn()`関数（252行目）：非推奨警告の出力メカニズム |

**主要処理フロー**:
- **252行目**: `depwarn(msg, funcsym)`がログシステム経由で警告を出力

### プログラム呼び出し階層図

```
deleteat!(::BitVector, i::Integer) (bitarray.jl:940)
    |
    +-- i isa Bool チェック (bitarray.jl:941)
    |      +-- depwarn("passing Bool as an index is deprecated", :deleteat!)
    |             +-- _depwarn() (deprecated.jl:259)
    |                    +-- @logmsg (CoreLogging)
    |
    +-- _deleteat!(B, Int(i)) (bitarray.jl:946)

splice!(::BitVector, i::Integer) (bitarray.jl:1060)
    |
    +-- i isa Bool チェック (bitarray.jl:1064)
    |      +-- depwarn(...)
    |
    +-- _deleteat!(B, Int(i)) (bitarray.jl:1070)

deleteat!(::Vector, i::Integer) (array.jl:1868)
    |
    +-- i isa Bool チェック (array.jl:1869)
    |      +-- depwarn(...)
    |
    +-- _deleteat!(a, i, 1) (array.jl:1870)
```

### データフロー図

```
[入力]                     [処理]                          [出力]

Bool インデックス ──────> isa Bool チェック ──────> depwarn() ──> ログ出力
                              |
                              +──> Int変換 ──> 配列操作実行 ──> 変更された配列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| bitarray.jl | `base/bitarray.jl` | ソース | `BitVector`の`deleteat!`、`splice!`でのBoolチェック（941, 1064, 1077行目） |
| array.jl | `base/array.jl` | ソース | `Vector`の`deleteat!`でのBoolチェック（1869, 1879行目） |
| deprecated.jl | `base/deprecated.jl` | ソース | `depwarn()`関数の定義（252行目） |
