# 非機能要件定義書

## 概要

本ドキュメントは、Julia言語処理系（julia-master）プロジェクトの非機能要件を、ソースコードの解析結果に基づいて定義するものである。Juliaはプログラミング言語ランタイムであり、一般的なWebアプリケーションとは異なるが、性能、可用性、セキュリティ、拡張性、保守性、運用・監視の各観点から、コードに実装されている非機能的な特性を整理する。

## 性能要件

### レスポンスタイム

| 処理種別 | 目標値 | 備考 |
| --- | --- | --- |
| タイマー精度 | 1ミリ秒以上 | `Timer`および`sleep`関数の最小精度。libuv内部で1ms早くタイムアウトする傾向があるため、+1msの補正が組み込まれている（`base/asyncevent.jl`） |
| `timedwait`ポーリング間隔 | 最小1ミリ秒（`pollint >= 1e-3`） | `timedwait`関数のポーリング間隔下限値。デフォルトは0.1秒（`base/asyncevent.jl`） |
| 分散ワーカー接続タイムアウト | デフォルト60.0秒 | `JULIA_WORKER_TIMEOUT`環境変数で設定可能。ワーカープロセスがマスタープロセスとの接続を待つ最大時間（`doc/src/manual/environment-variables.md`） |
| セーフポイント到達タイムアウト | 設定可能（1秒〜INT16_MAX秒） | `--timeout-for-safepoint-straggler`オプションで指定。セーフポイントに到達しないスレッドのバックトレースをダンプする閾値（`src/jloptions.c`） |
| `wait_with_timeout`最小タイムアウト | 0.001秒（1ミリ秒） | 実験的API `Base.Experimental.wait_with_timeout`の最小タイムアウト値（`base/experimental.jl`） |

### スループット

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| スレッド数 | 1〜INT16_MAX（デフォルトはautoまたは1） | `-t`/`--threads`オプションで設定。`auto`はCPUアフィニティに基づきスレッド数を自動推定。デフォルトプールとインタラクティブプールの2プール構成（`src/jloptions.c`） |
| GCスレッド数 | `--gcthreads=N[,M]`で設定可能 | マーク用N本、コンカレントスイープ用M本（0または1）。デフォルトはワーカースレッド数と同数（`src/jloptions.c`, `src/gc-stock.c`） |
| 並列プロセス数 | `-p`/`--procs`で設定可能（1〜INT16_MAX） | `auto`の場合はCPUスレッド数に基づく（`src/jloptions.c`） |
| プリコンパイル並列タスク数 | `JULIA_NUM_PRECOMPILE_TASKS`環境変数で設定可能 | パッケージプリコンパイル時の並列処理タスク数（`doc/src/manual/environment-variables.md`） |
| プリコンパイルキャッシュ上限 | デフォルト10インスタンス | `JULIA_MAX_NUM_PRECOMPILE_FILES`環境変数で単一パッケージあたりの最大キャッシュファイル数を指定（`doc/src/manual/environment-variables.md`） |

### メモリ管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| ヒープサイズヒント | `--heap-size-hint`で設定可能 | B/K/M/G/T/% 単位で指定。GCが指定値以上のメモリ使用時に強制実行される（`src/jloptions.c`） |
| ハードヒープ上限 | `--hard-heap-limit`で設定可能 | 超過時にプロセスをアボートする絶対上限値（`src/jloptions.c`） |
| ヒープ目標増分 | `--heap-target-increment`で設定可能 | 連続GC間のヒープ目標増分の上限（`src/jloptions.c`） |
| 32bitシステムメモリ上限 | 1.5 GiB | 32bitシステムでのメモリ上界値（`src/gc-stock.c`） |
| 64bitシステムメモリ上限 | 2 PiB（ソフトリミット） | GCヒューリスティクスの目標上限。超過は許容される（`src/gc-stock.c`） |
| デフォルトGC収集間隔 | 64bit: 約45MiB / 32bit: 約12MiB | 初期GC収集トリガー閾値（`src/gc-stock.c`） |

## 可用性要件

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 稼働率 | 明示的な定義なし | Juliaはプログラミング言語ランタイムであり、サービスとしての稼働率要件はコード上に定義されていない |
| 計画停止時間 | 該当なし | スタンドアロンランタイムのため該当しない |
| 障害復旧時間（RTO） | 該当なし | サービスではなくランタイムのため該当しない |
| 目標復旧時点（RPO） | 該当なし | サービスではなくランタイムのため該当しない |
| リトライ機構 | `retry`関数（ExponentialBackOff付き） | デフォルト: n=1, first_delay=0.05秒, max_delay=10.0秒, factor=5.0, jitter=0.1。カスタムチェック関数によるリトライ制御が可能（`base/error.jl`） |
| 拡張機能ロードリトライ | `retry_load_extensions()`関数 | 失敗した拡張機能のロードを手動で再試行可能。自動リトライは行わない（`base/loading.jl`） |
| シグナルハンドリング | `--handle-signals={yes\|no}` | デフォルトはyes。Juliaのデフォルトシグナルハンドラの有効/無効を制御（`src/jloptions.c`） |
| GC同期機構 | セーフポイントベース | 全スレッドがセーフステートに到達するまでGCを待機。`uv_mutex`/`uv_cond`による同期（`src/safepoint.c`, `src/gc-stock.c`） |

## セキュリティ要件

### ネットワークセキュリティ

| 項目 | 設定 | 備考 |
| --- | --- | --- |
| ホスト検証 | `JULIA_NO_VERIFY_HOSTS` / `JULIA_SSL_NO_VERIFY_HOSTS` / `JULIA_SSH_NO_VERIFY_HOSTS` / `JULIA_ALWAYS_VERIFY_HOSTS` | トランスポートレイヤごとにホスト検証を制御（`doc/src/manual/environment-variables.md`） |
| SSL証明書ルート | `JULIA_SSL_CA_ROOTS_PATH`環境変数 | 証明書機関ルートのファイル/ディレクトリを指定（`doc/src/manual/environment-variables.md`） |
| パッケージサーバ | `JULIA_PKG_SERVER`環境変数 | デフォルトは`https://pkg.julialang.org`。空文字列で直接ホストアクセスに切替可能（`doc/src/manual/environment-variables.md`） |
| パッケージハッシュ検証 | `JULIA_PKG_IGNORE_HASHES`環境変数 | デフォルトは検証有効。`1`に設定するとハッシュ検証を無効化（`doc/src/manual/environment-variables.md`） |

### コード実行セキュリティ

| 項目 | 設定 | 備考 |
| --- | --- | --- |
| 境界チェック | `--check-bounds={yes\|no\|auto}` | デフォルトはauto。`@inbounds`宣言の尊重を制御（`src/jloptions.c`） |
| 実験的機能の利用 | `--experimental`フラグで有効化が必要 | trim等の実験的機能は明示的に有効化しない限り使用不可（`src/jloptions.c`） |
| コードコンパイル制御 | `--compile={yes\|no\|all\|min}` | JITコンパイラの有効/無効を制御（`src/jloptions.c`） |

### サニタイザ対応

| 項目 | 設定 | 備考 |
| --- | --- | --- |
| MemorySanitizer | `--target-sanitize=memory` | 生成コードへのMSan計装（`src/jloptions.c`） |
| ThreadSanitizer | `--target-sanitize=thread` | 生成コードへのTSan計装（`src/jloptions.c`） |
| AddressSanitizer | `--target-sanitize=address` | 生成コードへのASan計装（`src/jloptions.c`） |

## 拡張性要件

### スレッドプールアーキテクチャ

Juliaのスレッドモデルは2つのスレッドプールで構成される。

- **default プール**: 計算処理用のスレッドプール。`-t N`で指定。
- **interactive プール**: インタラクティブ処理用のスレッドプール。`-t N,M`のMで指定。デフォルトは1（出力生成時は0）。

スレッド数は`JULIA_NUM_THREADS`環境変数または`-t`/`--threads`コマンドラインオプションで起動時に設定する。`auto`指定でOSのCPUアフィニティに基づき自動推定される（`src/jloptions.c`, `base/threadingconstructs.jl`）。

### 分散処理

- `-p`/`--procs`オプションによるマルチプロセス構成
- `--machine-file`によるクラスタノード指定
- `--bind-to`によるバインドアドレス指定
- ワーカー認証用Cookieの設定（`--worker`オプション）

### モジュールシステム

- **プリコンパイル**: `--compiled-modules={yes|no|existing|strict}`で制御。プリコンパイル済みモジュールの利用ポリシーを指定。
- **パッケージイメージ**: `--pkgimages={yes|no|existing}`でネイティブコードキャッシュの利用を制御。
- **拡張機能**: トリガーベースの遅延ロード機構。依存パッケージのロード時に自動的に拡張機能を読み込む（`base/loading.jl`）。
- **インクリメンタルコンパイル**: `--output-incremental`で増分コンパイル出力をサポート。

### CPU最適化

- `--cpu-target`によるターゲットCPUアーキテクチャ指定
- `JULIA_CPU_TARGET`環境変数によるプリコンパイル時のCPU機能指定
- `clone_all`、`base(N)`、`opt_size`、`min_size`等の特殊フラグによるマルチバージョニング対応
- 最適化レベル: `-O {0|1|2|3}`で制御（デフォルトはO2）
- 最小最適化レベル: `--min-optlevel`で下限を設定可能

## 保守性要件

### デバッグ支援

| 項目 | 設定 | 備考 |
| --- | --- | --- |
| デバッグ情報レベル | `-g {0\|1\|2}` | デフォルトはリリースビルドで1、デバッグビルドで2（`src/jloptions.c`） |
| トレースコンパイル | `--trace-compile={stderr\|name}` | コンパイルされたメソッドのプリコンパイル文を出力（`src/jloptions.c`） |
| コンパイル時間計測 | `--trace-compile-timing` | trace-compile有効時に各コンパイル所要時間をms単位で表示（`src/jloptions.c`） |
| eval追跡 | `--trace-eval={loc\|full\|no}` | eval実行式の表示（`src/jloptions.c`） |
| コードカバレッジ | `--code-coverage[={none\|user\|all}]` | ソースコード行の実行回数を計測（`src/jloptions.c`） |
| メモリ割当追跡 | `--track-allocation[={none\|user\|all}]` | ソースコード行ごとの割当バイト数を計測（`src/jloptions.c`） |
| 非推奨警告 | `--depwarn={yes\|no\|error}` | デフォルトはno。構文およびメソッドの非推奨警告を制御（`src/jloptions.c`） |
| オーバーライド警告 | `--warn-overwrite={yes\|no}` | デフォルトはno。メソッド上書き警告を制御（`src/jloptions.c`） |
| スコープ警告 | `--warn-scope={yes\|no}` | デフォルトはyes。曖昧なトップレベルスコープの警告を制御（`src/jloptions.c`） |
| バグレポート | `--bug-report=KIND` | BugReporting.jlを利用したバグレポートセッション起動（`src/jloptions.c`） |

### ロギング

Juliaは構造化ロギングフレームワーク（`CoreLogging`モジュール）を提供する。

- **ログレベル**: `@debug`, `@info`, `@warn`, `@error`, `@logmsg` マクロ
- **ロガーインターフェース**: `AbstractLogger`抽象型による拡張可能なロガーアーキテクチャ
- **フィルタリング**: `shouldlog`関数によるモジュール/グループ/IDベースのフィルタリング
- **グローバルロガー**: `global_logger`/`with_logger`による動的なロガー切替
- **例外安全**: デフォルトでログ生成中の例外をキャッチし、プログラムクラッシュを防止（`catch_exceptions(logger) = true`）
- **デバッグログ**: `JULIA_DEBUG`環境変数でファイル/モジュール単位でデバッグロギングを有効化

（`base/logging/logging.jl`）

### コード管理

- **システムイメージ圧縮**: `--compress-sysimage={yes|no}`でsys/pkgimageヒープの圧縮を制御
- **メタデータストリップ**: `--strip-metadata`でdocstringおよびソースロケーション情報を除去
- **IRストリップ**: `--strip-ir`でコンパイル済み関数のIR（中間表現）を除去
- **トリミング**: `--trim={no|safe|unsafe|unsafe-warn}`で到達可能コードのみのsysimageビルドをサポート（実験的機能）

## 運用・監視要件

### タスクメトリクス

- `--task-metrics={yes|no}`: タスクごとの時間データ収集の有効/無効（デフォルトはno）
- `Base.Experimental.task_metrics(bool)`: プログラム内からのメトリクス収集制御
- `Base.Experimental.task_running_time_ns(t)`: タスクのCPU実行時間（ナノ秒）
- `Base.Experimental.task_wall_time_ns(t)`: タスクのウォール時間（ナノ秒）

（`base/experimental.jl`, `base/options.jl`）

### GC監視

- `jl_full_sweep_reasons`: フルGCの理由ごとの発生回数を記録するテーブル（`src/gc-stock.c`）
- `gc_heap_stats`: システム全体のヒープ統計（`src/gc-stock.c`）
- `--gc-sweep-always-full`: GCを常にフルスイープモードで実行するデバッグオプション
- ヒープスナップショット: `JULIA_PROFILE_PEEK_HEAP_SNAPSHOT`環境変数で実行中のヒープスナップショット収集を有効化

### プロファイリング

- **JITプロファイリング**: `ENABLE_JITPROFILING`環境変数でIntel VTune / OProfile / Perfとの連携
- **GDBリスナー**: `ENABLE_GDBLISTENER`環境変数でGDB登録を有効化
- **タイミングサブシステム**: `JULIA_TIMING_SUBSYSTEMS`環境変数で個別ゾーン（GC、INFERENCE等）の有効/無効を制御
- **イメージコンパイルタイミング**: `JULIA_IMAGE_TIMINGS`環境変数で詳細なタイミング情報の出力を制御

### CI連携

- `JULIA_CI`環境変数: CIシステムであることをパッケージサーバに通知
- CIデバッグ環境変数の自動検出: CIデバッグモード時に`--trace-eval`と`JULIA_TEST_VERBOSE=true`を自動有効化

### スレッドスリープ制御

- `JULIA_THREAD_SLEEP_THRESHOLD`環境変数: スピニングスレッドがスリープするまでのナノ秒数を指定。"infinite"でスリープ無効化。

### GCデバッガ連携

- `JULIA_GC_WAIT_FOR_DEBUGGER`環境変数: GCのクリティカルエラー時にアボートせずデバッガのアタッチを待機

## 備考

- 本プロジェクトはプログラミング言語ランタイムであり、一般的なWebアプリケーションやサービスにおけるレートリミット、キャッシュTTL、HTTP接続プーリング等の概念は直接的には適用されない。
- キャッシュ機構としては、プリコンパイルキャッシュ（`--compiled-modules`）およびパッケージイメージキャッシュ（`--pkgimages`）が存在し、コンパイル済みネイティブコードの再利用による起動時間短縮に寄与する。
- レートリミットに相当する機構は実装されていないが、`ExponentialBackOff`によるリトライ遅延がバックプレッシャー的な役割を果たす。
- タイムアウト設定は、分散ワーカー接続（60秒）、セーフポイントストラグラー（設定可能）、タイマーAPI（ユーザ設定）の3箇所に存在する。
- ロギングは`CoreLogging`モジュールによる4段階のログレベル（Debug/Info/Warn/Error）と、拡張可能なロガーアーキテクチャで実現されている。
- 監視機能は主にGC統計、タスクメトリクス、プロファイリングの3軸で構成されている。
