# API設計書

## 概要

本ドキュメントは、Kubernetes API Serverが提供するRESTful APIの設計書である。Kubernetes API Serverは、クラスタ内の全リソース（Pod、Service、Deployment等）のCRUD操作を統一的に提供するコントロールプレーンの中核コンポーネントである。APIは大きく「Core API（/api/v1）」と「Named API Groups（/apis/{group}/{version}）」に分類され、全537パス・1055オペレーションで構成される。

## 共通仕様

### ベースURL

```
https://<api-server-host>:6443
```

### 認証方式

Bearer Token認証（BearerToken）。リクエストヘッダーの `Authorization` フィールドにトークンを付与する。Kubernetes はこのほかにクライアント証明書認証、OpenID Connect Token、ServiceAccount Tokenなど複数の認証方式をサポートする。

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Authorization | ○ | `Bearer <token>` 形式の認証トークン |
| Content-Type | ○（書き込み系） | `application/json`, `application/yaml`, `application/vnd.kubernetes.protobuf`, `application/cbor` 等 |
| Accept | - | `application/json`, `application/yaml`, `application/vnd.kubernetes.protobuf`, `application/cbor`, `application/json;stream=watch`（Watch時）等 |

### 共通クエリパラメータ（リスト系操作）

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| watch | boolean | - | trueの場合、Watchストリームを返す |
| labelSelector | string | - | ラベルセレクタによるフィルタリング |
| fieldSelector | string | - | フィールドセレクタによるフィルタリング |
| limit | integer | - | 返却するリソース数の上限 |
| continue | string | - | ページネーションの継続トークン |
| resourceVersion | string | - | 特定のリソースバージョン以降の結果を取得 |
| resourceVersionMatch | string | - | resourceVersionの一致条件（Exact/NotOlderThan） |
| timeoutSeconds | integer | - | リクエストのタイムアウト秒数 |
| sendInitialEvents | boolean | - | Watch時に初期イベントを送信するか |

### 共通クエリパラメータ（書き込み系操作）

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| dryRun | string | - | `All` を指定するとドライラン実行（実際の変更なし） |
| fieldValidation | string | - | フィールドバリデーションの厳格度（Strict/Warn/Ignore） |
| fieldManager | string | - | Server-Side Applyでのフィールド管理者名 |
| pretty | string | - | 出力を整形するか |

### 共通エラーレスポンス

| ステータスコード | 説明 |
| --- | --- |
| 400 | Bad Request - リクエストパラメータ不正 |
| 401 | Unauthorized - 認証エラー（全エンドポイント共通） |
| 403 | Forbidden - RBAC権限エラー |
| 404 | Not Found - リソース未検出 |
| 405 | Method Not Allowed - 許可されていないHTTPメソッド |
| 409 | Conflict - リソースバージョン競合 |
| 422 | Unprocessable Entity - バリデーションエラー |
| 429 | Too Many Requests - APIプライオリティ/フェアネスによるレート制限 |
| 500 | Internal Server Error - サーバー内部エラー |
| 503 | Service Unavailable - サーバー利用不可 |

共通エラーレスポンスは `metav1.Status` オブジェクトで返却される:

```json
{
  "kind": "Status",
  "apiVersion": "v1",
  "metadata": {},
  "status": "Failure",
  "message": "pods \"example\" not found",
  "reason": "NotFound",
  "details": {
    "name": "example",
    "kind": "pods"
  },
  "code": 404
}
```

## API一覧

### ディスカバリ・メタデータ

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| ディスカバリ | /api/ | GET | Core APIのバージョン一覧取得 |
| ディスカバリ | /api/v1/ | GET | Core API v1のリソース一覧取得 |
| ディスカバリ | /apis/ | GET | 全APIグループの一覧取得 |
| バージョン | /version/ | GET | APIサーバーのバージョン情報取得 |
| OIDC | /.well-known/openid-configuration/ | GET | OIDC設定情報取得 |
| OIDC | /openid/v1/jwks/ | GET | OIDC JSON Web Key Set取得 |
| ログ | /logs/{logpath} | GET | サーバーログ取得 |

### Core API v1（/api/v1）

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Namespace | /api/v1/namespaces | GET, POST | Namespace一覧取得・作成 |
| Namespace | /api/v1/namespaces/{name} | GET, PUT, DELETE, PATCH | Namespace個別操作 |
| Namespace | /api/v1/namespaces/{name}/finalize | PUT | Namespaceファイナライズ |
| Namespace | /api/v1/namespaces/{name}/status | GET, PUT, PATCH | Namespaceステータス操作 |
| Pod | /api/v1/namespaces/{namespace}/pods | GET, POST, DELETE | Pod一覧取得・作成・一括削除 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name} | GET, PUT, DELETE, PATCH | Pod個別操作 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/log | GET | Podログ取得 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/exec | GET, POST | Podコンテナでのコマンド実行 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/attach | GET, POST | Podコンテナへのアタッチ |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/portforward | GET, POST | Podポートフォワーディング |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/proxy/{path} | GET, POST, PUT, DELETE, PATCH | Podへのプロキシ |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/status | GET, PUT, PATCH | Podステータス操作 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/binding | POST | Pod バインディング作成 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/eviction | POST | Pod Eviction作成 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/ephemeralcontainers | GET, PUT, PATCH | エフェメラルコンテナ操作 |
| Pod | /api/v1/namespaces/{namespace}/pods/{name}/resize | GET, PUT, PATCH | Podリサイズ操作 |
| Pod | /api/v1/pods | GET | 全NamespaceのPod一覧取得 |
| Service | /api/v1/namespaces/{namespace}/services | GET, POST, DELETE | Service一覧取得・作成・一括削除 |
| Service | /api/v1/namespaces/{namespace}/services/{name} | GET, PUT, DELETE, PATCH | Service個別操作 |
| Service | /api/v1/namespaces/{namespace}/services/{name}/status | GET, PUT, PATCH | Serviceステータス操作 |
| Service | /api/v1/namespaces/{namespace}/services/{name}/proxy/{path} | GET, POST, PUT, DELETE, PATCH | Serviceへのプロキシ |
| Service | /api/v1/services | GET | 全NamespaceのService一覧取得 |
| ConfigMap | /api/v1/namespaces/{namespace}/configmaps | GET, POST, DELETE | ConfigMap一覧取得・作成・一括削除 |
| ConfigMap | /api/v1/namespaces/{namespace}/configmaps/{name} | GET, PUT, DELETE, PATCH | ConfigMap個別操作 |
| ConfigMap | /api/v1/configmaps | GET | 全NamespaceのConfigMap一覧取得 |
| Secret | /api/v1/namespaces/{namespace}/secrets | GET, POST, DELETE | Secret一覧取得・作成・一括削除 |
| Secret | /api/v1/namespaces/{namespace}/secrets/{name} | GET, PUT, DELETE, PATCH | Secret個別操作 |
| Secret | /api/v1/secrets | GET | 全NamespaceのSecret一覧取得 |
| Node | /api/v1/nodes | GET, POST, DELETE | Node一覧取得・作成・一括削除 |
| Node | /api/v1/nodes/{name} | GET, PUT, DELETE, PATCH | Node個別操作 |
| Node | /api/v1/nodes/{name}/status | GET, PUT, PATCH | Nodeステータス操作 |
| Node | /api/v1/nodes/{name}/proxy/{path} | GET, POST, PUT, DELETE, PATCH | Nodeへのプロキシ |
| PersistentVolume | /api/v1/persistentvolumes | GET, POST, DELETE | PV一覧取得・作成・一括削除 |
| PersistentVolume | /api/v1/persistentvolumes/{name} | GET, PUT, DELETE, PATCH | PV個別操作 |
| PersistentVolume | /api/v1/persistentvolumes/{name}/status | GET, PUT, PATCH | PVステータス操作 |
| PersistentVolumeClaim | /api/v1/namespaces/{namespace}/persistentvolumeclaims | GET, POST, DELETE | PVC一覧取得・作成・一括削除 |
| PersistentVolumeClaim | /api/v1/namespaces/{namespace}/persistentvolumeclaims/{name} | GET, PUT, DELETE, PATCH | PVC個別操作 |
| PersistentVolumeClaim | /api/v1/namespaces/{namespace}/persistentvolumeclaims/{name}/status | GET, PUT, PATCH | PVCステータス操作 |
| ServiceAccount | /api/v1/namespaces/{namespace}/serviceaccounts | GET, POST, DELETE | ServiceAccount一覧取得・作成・一括削除 |
| ServiceAccount | /api/v1/namespaces/{namespace}/serviceaccounts/{name} | GET, PUT, DELETE, PATCH | ServiceAccount個別操作 |
| ServiceAccount | /api/v1/namespaces/{namespace}/serviceaccounts/{name}/token | POST | ServiceAccountトークン作成 |
| Endpoint | /api/v1/namespaces/{namespace}/endpoints | GET, POST, DELETE | Endpoints一覧取得・作成・一括削除 |
| Endpoint | /api/v1/namespaces/{namespace}/endpoints/{name} | GET, PUT, DELETE, PATCH | Endpoints個別操作 |
| Event | /api/v1/namespaces/{namespace}/events | GET, POST, DELETE | Event一覧取得・作成・一括削除 |
| Event | /api/v1/namespaces/{namespace}/events/{name} | GET, PUT, DELETE, PATCH | Event個別操作 |
| LimitRange | /api/v1/namespaces/{namespace}/limitranges | GET, POST, DELETE | LimitRange一覧取得・作成・一括削除 |
| LimitRange | /api/v1/namespaces/{namespace}/limitranges/{name} | GET, PUT, DELETE, PATCH | LimitRange個別操作 |
| ResourceQuota | /api/v1/namespaces/{namespace}/resourcequotas | GET, POST, DELETE | ResourceQuota一覧取得・作成・一括削除 |
| ResourceQuota | /api/v1/namespaces/{namespace}/resourcequotas/{name} | GET, PUT, DELETE, PATCH | ResourceQuota個別操作 |
| ResourceQuota | /api/v1/namespaces/{namespace}/resourcequotas/{name}/status | GET, PUT, PATCH | ResourceQuotaステータス操作 |
| ReplicationController | /api/v1/namespaces/{namespace}/replicationcontrollers | GET, POST, DELETE | ReplicationController一覧取得・作成・一括削除 |
| ReplicationController | /api/v1/namespaces/{namespace}/replicationcontrollers/{name} | GET, PUT, DELETE, PATCH | ReplicationController個別操作 |
| ReplicationController | /api/v1/namespaces/{namespace}/replicationcontrollers/{name}/scale | GET, PUT, PATCH | ReplicationControllerスケール操作 |
| ReplicationController | /api/v1/namespaces/{namespace}/replicationcontrollers/{name}/status | GET, PUT, PATCH | ReplicationControllerステータス操作 |
| PodTemplate | /api/v1/namespaces/{namespace}/podtemplates | GET, POST, DELETE | PodTemplate一覧取得・作成・一括削除 |
| PodTemplate | /api/v1/namespaces/{namespace}/podtemplates/{name} | GET, PUT, DELETE, PATCH | PodTemplate個別操作 |
| ComponentStatus | /api/v1/componentstatuses | GET | ComponentStatus一覧取得 |
| ComponentStatus | /api/v1/componentstatuses/{name} | GET | ComponentStatus個別取得 |
| Binding | /api/v1/namespaces/{namespace}/bindings | POST | Binding作成 |

### apps/v1

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Deployment | /apis/apps/v1/namespaces/{namespace}/deployments | GET, POST, DELETE | Deployment一覧取得・作成・一括削除 |
| Deployment | /apis/apps/v1/namespaces/{namespace}/deployments/{name} | GET, PUT, DELETE, PATCH | Deployment個別操作 |
| Deployment | /apis/apps/v1/namespaces/{namespace}/deployments/{name}/scale | GET, PUT, PATCH | Deploymentスケール操作 |
| Deployment | /apis/apps/v1/namespaces/{namespace}/deployments/{name}/status | GET, PUT, PATCH | Deploymentステータス操作 |
| Deployment | /apis/apps/v1/deployments | GET | 全NamespaceのDeployment一覧取得 |
| StatefulSet | /apis/apps/v1/namespaces/{namespace}/statefulsets | GET, POST, DELETE | StatefulSet一覧取得・作成・一括削除 |
| StatefulSet | /apis/apps/v1/namespaces/{namespace}/statefulsets/{name} | GET, PUT, DELETE, PATCH | StatefulSet個別操作 |
| StatefulSet | /apis/apps/v1/namespaces/{namespace}/statefulsets/{name}/scale | GET, PUT, PATCH | StatefulSetスケール操作 |
| StatefulSet | /apis/apps/v1/namespaces/{namespace}/statefulsets/{name}/status | GET, PUT, PATCH | StatefulSetステータス操作 |
| DaemonSet | /apis/apps/v1/namespaces/{namespace}/daemonsets | GET, POST, DELETE | DaemonSet一覧取得・作成・一括削除 |
| DaemonSet | /apis/apps/v1/namespaces/{namespace}/daemonsets/{name} | GET, PUT, DELETE, PATCH | DaemonSet個別操作 |
| DaemonSet | /apis/apps/v1/namespaces/{namespace}/daemonsets/{name}/status | GET, PUT, PATCH | DaemonSetステータス操作 |
| ReplicaSet | /apis/apps/v1/namespaces/{namespace}/replicasets | GET, POST, DELETE | ReplicaSet一覧取得・作成・一括削除 |
| ReplicaSet | /apis/apps/v1/namespaces/{namespace}/replicasets/{name} | GET, PUT, DELETE, PATCH | ReplicaSet個別操作 |
| ReplicaSet | /apis/apps/v1/namespaces/{namespace}/replicasets/{name}/scale | GET, PUT, PATCH | ReplicaSetスケール操作 |
| ReplicaSet | /apis/apps/v1/namespaces/{namespace}/replicasets/{name}/status | GET, PUT, PATCH | ReplicaSetステータス操作 |
| ControllerRevision | /apis/apps/v1/namespaces/{namespace}/controllerrevisions | GET, POST, DELETE | ControllerRevision一覧取得・作成・一括削除 |
| ControllerRevision | /apis/apps/v1/namespaces/{namespace}/controllerrevisions/{name} | GET, PUT, DELETE, PATCH | ControllerRevision個別操作 |

### batch/v1

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Job | /apis/batch/v1/namespaces/{namespace}/jobs | GET, POST, DELETE | Job一覧取得・作成・一括削除 |
| Job | /apis/batch/v1/namespaces/{namespace}/jobs/{name} | GET, PUT, DELETE, PATCH | Job個別操作 |
| Job | /apis/batch/v1/namespaces/{namespace}/jobs/{name}/status | GET, PUT, PATCH | Jobステータス操作 |
| CronJob | /apis/batch/v1/namespaces/{namespace}/cronjobs | GET, POST, DELETE | CronJob一覧取得・作成・一括削除 |
| CronJob | /apis/batch/v1/namespaces/{namespace}/cronjobs/{name} | GET, PUT, DELETE, PATCH | CronJob個別操作 |
| CronJob | /apis/batch/v1/namespaces/{namespace}/cronjobs/{name}/status | GET, PUT, PATCH | CronJobステータス操作 |

### networking.k8s.io/v1

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Ingress | /apis/networking.k8s.io/v1/namespaces/{namespace}/ingresses | GET, POST, DELETE | Ingress一覧取得・作成・一括削除 |
| Ingress | /apis/networking.k8s.io/v1/namespaces/{namespace}/ingresses/{name} | GET, PUT, DELETE, PATCH | Ingress個別操作 |
| Ingress | /apis/networking.k8s.io/v1/namespaces/{namespace}/ingresses/{name}/status | GET, PUT, PATCH | Ingressステータス操作 |
| IngressClass | /apis/networking.k8s.io/v1/ingressclasses | GET, POST, DELETE | IngressClass一覧取得・作成・一括削除 |
| IngressClass | /apis/networking.k8s.io/v1/ingressclasses/{name} | GET, PUT, DELETE, PATCH | IngressClass個別操作 |
| NetworkPolicy | /apis/networking.k8s.io/v1/namespaces/{namespace}/networkpolicies | GET, POST, DELETE | NetworkPolicy一覧取得・作成・一括削除 |
| NetworkPolicy | /apis/networking.k8s.io/v1/namespaces/{namespace}/networkpolicies/{name} | GET, PUT, DELETE, PATCH | NetworkPolicy個別操作 |

### autoscaling/v2

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| HPA | /apis/autoscaling/v2/namespaces/{namespace}/horizontalpodautoscalers | GET, POST, DELETE | HPA一覧取得・作成・一括削除 |
| HPA | /apis/autoscaling/v2/namespaces/{namespace}/horizontalpodautoscalers/{name} | GET, PUT, DELETE, PATCH | HPA個別操作 |
| HPA | /apis/autoscaling/v2/namespaces/{namespace}/horizontalpodautoscalers/{name}/status | GET, PUT, PATCH | HPAステータス操作 |

### rbac.authorization.k8s.io/v1

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| ClusterRole | /apis/rbac.authorization.k8s.io/v1/clusterroles | GET, POST, DELETE | ClusterRole一覧取得・作成・一括削除 |
| ClusterRole | /apis/rbac.authorization.k8s.io/v1/clusterroles/{name} | GET, PUT, DELETE, PATCH | ClusterRole個別操作 |
| ClusterRoleBinding | /apis/rbac.authorization.k8s.io/v1/clusterrolebindings | GET, POST, DELETE | ClusterRoleBinding一覧取得・作成・一括削除 |
| ClusterRoleBinding | /apis/rbac.authorization.k8s.io/v1/clusterrolebindings/{name} | GET, PUT, DELETE, PATCH | ClusterRoleBinding個別操作 |
| Role | /apis/rbac.authorization.k8s.io/v1/namespaces/{namespace}/roles | GET, POST, DELETE | Role一覧取得・作成・一括削除 |
| Role | /apis/rbac.authorization.k8s.io/v1/namespaces/{namespace}/roles/{name} | GET, PUT, DELETE, PATCH | Role個別操作 |
| RoleBinding | /apis/rbac.authorization.k8s.io/v1/namespaces/{namespace}/rolebindings | GET, POST, DELETE | RoleBinding一覧取得・作成・一括削除 |
| RoleBinding | /apis/rbac.authorization.k8s.io/v1/namespaces/{namespace}/rolebindings/{name} | GET, PUT, DELETE, PATCH | RoleBinding個別操作 |

### storage.k8s.io/v1

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| StorageClass | /apis/storage.k8s.io/v1/storageclasses | GET, POST, DELETE | StorageClass一覧取得・作成・一括削除 |
| StorageClass | /apis/storage.k8s.io/v1/storageclasses/{name} | GET, PUT, DELETE, PATCH | StorageClass個別操作 |
| VolumeAttachment | /apis/storage.k8s.io/v1/volumeattachments | GET, POST, DELETE | VolumeAttachment一覧取得・作成・一括削除 |
| VolumeAttachment | /apis/storage.k8s.io/v1/volumeattachments/{name} | GET, PUT, DELETE, PATCH | VolumeAttachment個別操作 |
| CSIDriver | /apis/storage.k8s.io/v1/csidrivers | GET, POST, DELETE | CSIDriver一覧取得・作成・一括削除 |
| CSIDriver | /apis/storage.k8s.io/v1/csidrivers/{name} | GET, PUT, DELETE, PATCH | CSIDriver個別操作 |
| CSINode | /apis/storage.k8s.io/v1/csinodes | GET, POST, DELETE | CSINode一覧取得・作成・一括削除 |
| CSINode | /apis/storage.k8s.io/v1/csinodes/{name} | GET, PUT, DELETE, PATCH | CSINode個別操作 |

### その他のAPIグループ

| APIグループ | バージョン | 主要リソース |
| --- | --- | --- |
| admissionregistration.k8s.io | v1, v1alpha1, v1beta1 | ValidatingWebhookConfiguration, MutatingWebhookConfiguration, ValidatingAdmissionPolicy, MutatingAdmissionPolicy |
| apiextensions.k8s.io | v1 | CustomResourceDefinition |
| apiregistration.k8s.io | v1 | APIService |
| authentication.k8s.io | v1 | TokenReview, SelfSubjectReview |
| authorization.k8s.io | v1 | SubjectAccessReview, SelfSubjectAccessReview, LocalSubjectAccessReview |
| certificates.k8s.io | v1, v1alpha1, v1beta1 | CertificateSigningRequest, ClusterTrustBundle |
| coordination.k8s.io | v1, v1alpha2, v1beta1 | Lease, LeaseCandidate |
| discovery.k8s.io | v1 | EndpointSlice |
| events.k8s.io | v1 | Event |
| flowcontrol.apiserver.k8s.io | v1 | FlowSchema, PriorityLevelConfiguration |
| internal.apiserver.k8s.io | v1alpha1 | StorageVersion |
| node.k8s.io | v1 | RuntimeClass |
| policy | v1 | PodDisruptionBudget |
| resource.k8s.io | v1, v1alpha3, v1beta1, v1beta2 | DeviceClass, ResourceClaim, ResourceSlice |
| scheduling.k8s.io | v1, v1alpha1 | PriorityClass |
| storagemigration.k8s.io | v1beta1 | StorageVersionMigration |

## 各APIエンドポイント定義

### ディスカバリ

#### 1. APIバージョン一覧取得

Core APIで利用可能なバージョン一覧を返却する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/` |
| 認証 | 必要 |
| 権限 | 不要（公開エンドポイント） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "APIVersions",
  "versions": [
    "v1"
  ],
  "serverAddressByClientCIDRs": [
    {
      "clientCIDR": "0.0.0.0/0",
      "serverAddress": "10.0.0.1:6443"
    }
  ]
}
```

#### 2. APIリソース一覧取得

指定バージョンで利用可能なリソースタイプの一覧を返却する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/` |
| 認証 | 必要 |
| 権限 | 不要（公開エンドポイント） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "APIResourceList",
  "groupVersion": "v1",
  "resources": [
    {
      "name": "pods",
      "singularName": "pod",
      "namespaced": true,
      "kind": "Pod",
      "verbs": ["create", "delete", "deletecollection", "get", "list", "patch", "update", "watch"]
    }
  ]
}
```

#### 3. APIグループ一覧取得

全APIグループの一覧を返却する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /apis/` |
| 認証 | 必要 |
| 権限 | 不要（公開エンドポイント） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "APIGroupList",
  "apiVersion": "v1",
  "groups": [
    {
      "name": "apps",
      "versions": [
        {
          "groupVersion": "apps/v1",
          "version": "v1"
        }
      ],
      "preferredVersion": {
        "groupVersion": "apps/v1",
        "version": "v1"
      }
    }
  ]
}
```

#### 4. バージョン情報取得

APIサーバーのビルドバージョン情報を返却する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /version/` |
| 認証 | 必要 |
| 権限 | 不要（公開エンドポイント） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "major": "1",
  "minor": "32",
  "gitVersion": "v1.32.0",
  "gitCommit": "abc123",
  "gitTreeState": "clean",
  "buildDate": "2025-01-01T00:00:00Z",
  "goVersion": "go1.23.0",
  "compiler": "gc",
  "platform": "linux/amd64"
}
```

### Pod

#### 5. Pod一覧取得

指定Namespace内のPod一覧を取得する。watchパラメータにより変更監視ストリームも取得可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces/{namespace}/pods` |
| 認証 | 必要 |
| 権限 | pods リソースの list 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |

**クエリパラメータ**

共通リスト系クエリパラメータを参照。

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "PodList",
  "apiVersion": "v1",
  "metadata": {
    "resourceVersion": "12345"
  },
  "items": [
    {
      "metadata": {
        "name": "example-pod",
        "namespace": "default",
        "uid": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
        "resourceVersion": "12340",
        "creationTimestamp": "2025-01-01T00:00:00Z",
        "labels": {
          "app": "example"
        }
      },
      "spec": {
        "containers": [
          {
            "name": "main",
            "image": "nginx:latest",
            "ports": [
              {
                "containerPort": 80
              }
            ]
          }
        ]
      },
      "status": {
        "phase": "Running"
      }
    }
  ]
}
```

#### 6. Pod作成

指定Namespace内に新しいPodを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/v1/namespaces/{namespace}/pods` |
| 認証 | 必要 |
| 権限 | pods リソースの create 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| dryRun | string | - | - | `All` でドライラン実行 |
| fieldValidation | string | - | - | バリデーション厳格度（Strict/Warn/Ignore） |

**リクエストボディ**

```json
{
  "apiVersion": "v1",
  "kind": "Pod",
  "metadata": {
    "name": "example-pod",
    "labels": {
      "app": "example"
    }
  },
  "spec": {
    "containers": [
      {
        "name": "main",
        "image": "nginx:latest",
        "ports": [
          {
            "containerPort": 80
          }
        ]
      }
    ]
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| apiVersion | string | ○ | APIバージョン（"v1"） |
| kind | string | ○ | リソース種別（"Pod"） |
| metadata.name | string | ○ | Pod名 |
| metadata.namespace | string | - | Namespace（パスパラメータで指定） |
| metadata.labels | map[string]string | - | ラベル |
| spec.containers | array | ○ | コンテナ定義の配列 |
| spec.containers[].name | string | ○ | コンテナ名 |
| spec.containers[].image | string | ○ | コンテナイメージ |

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created` / `202 Accepted`

Podオブジェクト（リクエストボディと同等の構造にstatusフィールドが追加された形式）が返却される。

**レスポンス（エラー時）**

ステータスコード: `422 Unprocessable Entity`

```json
{
  "kind": "Status",
  "apiVersion": "v1",
  "metadata": {},
  "status": "Failure",
  "message": "Pod \"example-pod\" is invalid: spec.containers: Required value",
  "reason": "Invalid",
  "details": {
    "name": "example-pod",
    "kind": "Pod",
    "causes": [
      {
        "reason": "FieldValueRequired",
        "message": "Required value",
        "field": "spec.containers"
      }
    ]
  },
  "code": 422
}
```

#### 7. Pod個別取得

指定されたPodの詳細情報を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces/{namespace}/pods/{name}` |
| 認証 | 必要 |
| 権限 | pods リソースの get 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |
| name | string | ○ | Pod名 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

Podオブジェクト（Pod一覧取得のitems要素と同一構造）。

#### 8. Pod更新

指定されたPodを全置換で更新する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /api/v1/namespaces/{namespace}/pods/{name}` |
| 認証 | 必要 |
| 権限 | pods リソースの update 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |
| name | string | ○ | Pod名 |

**リクエストボディ**

更新後のPodオブジェクト全体。metadata.resourceVersionが必須で、楽観的ロックに使用される。

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created`

#### 9. Pod部分更新

指定されたPodを部分的に更新する（Strategic Merge Patch / JSON Merge Patch / JSON Patch）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PATCH /api/v1/namespaces/{namespace}/pods/{name}` |
| 認証 | 必要 |
| 権限 | pods リソースの patch 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |
| name | string | ○ | Pod名 |

**リクエストボディ**

Content-Typeに応じたパッチ形式:
- `application/strategic-merge-patch+json`: Strategic Merge Patch
- `application/merge-patch+json`: JSON Merge Patch
- `application/json-patch+json`: JSON Patch
- `application/apply-patch+yaml`: Server-Side Apply

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created`

#### 10. Pod削除

指定されたPodを削除する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /api/v1/namespaces/{namespace}/pods/{name}` |
| 認証 | 必要 |
| 権限 | pods リソースの delete 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |
| name | string | ○ | Pod名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| dryRun | string | - | - | `All` でドライラン実行 |
| gracePeriodSeconds | integer | - | - | 猶予期間（秒）。0で即時削除 |
| propagationPolicy | string | - | - | 削除伝播ポリシー（Orphan/Background/Foreground） |

**レスポンス（成功時）**

ステータスコード: `200 OK` / `202 Accepted`

#### 11. Podログ取得

指定されたPodのコンテナログを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces/{namespace}/pods/{name}/log` |
| 認証 | 必要 |
| 権限 | pods/log サブリソースの get 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |
| name | string | ○ | Pod名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| container | string | - | - | コンテナ名（マルチコンテナPodの場合に指定） |
| follow | boolean | - | false | ログストリームをフォローするか |
| tailLines | integer | - | - | 末尾から取得する行数 |
| sinceSeconds | integer | - | - | 指定秒数以内のログを取得 |
| timestamps | boolean | - | false | タイムスタンプを付与するか |
| previous | boolean | - | false | 前回のコンテナインスタンスのログを取得 |
| limitBytes | integer | - | - | 取得するログのバイト数上限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

プレーンテキスト形式でログ内容が返却される。

#### 12. Podコマンド実行（exec）

Podのコンテナ内でコマンドを実行する（WebSocketアップグレード）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/v1/namespaces/{namespace}/pods/{name}/exec` |
| 認証 | 必要 |
| 権限 | pods/exec サブリソースの create 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |
| name | string | ○ | Pod名 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| command | string | ○ | - | 実行するコマンド（複数指定可） |
| container | string | - | - | 対象コンテナ名 |
| stdin | boolean | - | false | 標準入力を有効にするか |
| stdout | boolean | - | true | 標準出力を有効にするか |
| stderr | boolean | - | true | 標準エラー出力を有効にするか |
| tty | boolean | - | false | TTYを割り当てるか |

**レスポンス（成功時）**

ステータスコード: `200 OK` - SPDY/WebSocketプロトコルへアップグレード。

### Namespace

#### 13. Namespace一覧取得

クラスタ内のNamespace一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces` |
| 認証 | 必要 |
| 権限 | namespaces リソースの list 権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "NamespaceList",
  "apiVersion": "v1",
  "metadata": {
    "resourceVersion": "12345"
  },
  "items": [
    {
      "metadata": {
        "name": "default",
        "uid": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
        "resourceVersion": "1"
      },
      "spec": {
        "finalizers": ["kubernetes"]
      },
      "status": {
        "phase": "Active"
      }
    }
  ]
}
```

#### 14. Namespace作成

新しいNamespaceを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/v1/namespaces` |
| 認証 | 必要 |
| 権限 | namespaces リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "v1",
  "kind": "Namespace",
  "metadata": {
    "name": "my-namespace",
    "labels": {
      "env": "production"
    }
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created` / `202 Accepted`

### Service

#### 15. Service一覧取得

指定Namespace内のService一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces/{namespace}/services` |
| 認証 | 必要 |
| 権限 | services リソースの list 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "ServiceList",
  "apiVersion": "v1",
  "metadata": {
    "resourceVersion": "12345"
  },
  "items": [
    {
      "metadata": {
        "name": "my-service",
        "namespace": "default"
      },
      "spec": {
        "selector": {
          "app": "example"
        },
        "ports": [
          {
            "protocol": "TCP",
            "port": 80,
            "targetPort": 8080
          }
        ],
        "type": "ClusterIP",
        "clusterIP": "10.96.0.100"
      }
    }
  ]
}
```

#### 16. Service作成

指定Namespace内に新しいServiceを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/v1/namespaces/{namespace}/services` |
| 認証 | 必要 |
| 権限 | services リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "v1",
  "kind": "Service",
  "metadata": {
    "name": "my-service"
  },
  "spec": {
    "selector": {
      "app": "example"
    },
    "ports": [
      {
        "protocol": "TCP",
        "port": 80,
        "targetPort": 8080
      }
    ],
    "type": "ClusterIP"
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created` / `202 Accepted`

### Deployment

#### 17. Deployment一覧取得

指定Namespace内のDeployment一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /apis/apps/v1/namespaces/{namespace}/deployments` |
| 認証 | 必要 |
| 権限 | deployments リソースの list 権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| namespace | string | ○ | 対象のNamespace名 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "DeploymentList",
  "apiVersion": "apps/v1",
  "metadata": {
    "resourceVersion": "12345"
  },
  "items": [
    {
      "metadata": {
        "name": "my-deployment",
        "namespace": "default"
      },
      "spec": {
        "replicas": 3,
        "selector": {
          "matchLabels": {
            "app": "example"
          }
        },
        "template": {
          "metadata": {
            "labels": {
              "app": "example"
            }
          },
          "spec": {
            "containers": [
              {
                "name": "main",
                "image": "nginx:latest"
              }
            ]
          }
        }
      },
      "status": {
        "replicas": 3,
        "readyReplicas": 3,
        "availableReplicas": 3
      }
    }
  ]
}
```

#### 18. Deployment作成

指定Namespace内に新しいDeploymentを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /apis/apps/v1/namespaces/{namespace}/deployments` |
| 認証 | 必要 |
| 権限 | deployments リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "apps/v1",
  "kind": "Deployment",
  "metadata": {
    "name": "my-deployment"
  },
  "spec": {
    "replicas": 3,
    "selector": {
      "matchLabels": {
        "app": "example"
      }
    },
    "template": {
      "metadata": {
        "labels": {
          "app": "example"
        }
      },
      "spec": {
        "containers": [
          {
            "name": "main",
            "image": "nginx:latest",
            "ports": [
              {
                "containerPort": 80
              }
            ]
          }
        ]
      }
    }
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created` / `202 Accepted`

#### 19. Deploymentスケール操作

Deploymentのレプリカ数を取得・変更する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET/PUT/PATCH /apis/apps/v1/namespaces/{namespace}/deployments/{name}/scale` |
| 認証 | 必要 |
| 権限 | deployments/scale サブリソースの get/update/patch 権限 |

**レスポンス・リクエスト**

```json
{
  "kind": "Scale",
  "apiVersion": "autoscaling/v1",
  "metadata": {
    "name": "my-deployment",
    "namespace": "default"
  },
  "spec": {
    "replicas": 5
  },
  "status": {
    "replicas": 3,
    "selector": "app=example"
  }
}
```

### ConfigMap

#### 20. ConfigMap一覧取得

指定Namespace内のConfigMap一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces/{namespace}/configmaps` |
| 認証 | 必要 |
| 権限 | configmaps リソースの list 権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "ConfigMapList",
  "apiVersion": "v1",
  "items": [
    {
      "metadata": {
        "name": "my-config",
        "namespace": "default"
      },
      "data": {
        "key1": "value1",
        "key2": "value2"
      }
    }
  ]
}
```

#### 21. ConfigMap作成

指定Namespace内に新しいConfigMapを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/v1/namespaces/{namespace}/configmaps` |
| 認証 | 必要 |
| 権限 | configmaps リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "v1",
  "kind": "ConfigMap",
  "metadata": {
    "name": "my-config"
  },
  "data": {
    "key1": "value1",
    "config.yaml": "server:\n  port: 8080\n"
  },
  "binaryData": {
    "binary-key": "base64encodeddata"
  }
}
```

### Secret

#### 22. Secret一覧取得

指定Namespace内のSecret一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/namespaces/{namespace}/secrets` |
| 認証 | 必要 |
| 権限 | secrets リソースの list 権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "SecretList",
  "apiVersion": "v1",
  "items": [
    {
      "metadata": {
        "name": "my-secret",
        "namespace": "default"
      },
      "type": "Opaque",
      "data": {
        "username": "YWRtaW4=",
        "password": "cGFzc3dvcmQ="
      }
    }
  ]
}
```

#### 23. Secret作成

指定Namespace内に新しいSecretを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /api/v1/namespaces/{namespace}/secrets` |
| 認証 | 必要 |
| 権限 | secrets リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "v1",
  "kind": "Secret",
  "metadata": {
    "name": "my-secret"
  },
  "type": "Opaque",
  "data": {
    "username": "YWRtaW4=",
    "password": "cGFzc3dvcmQ="
  }
}
```

### Node

#### 24. Node一覧取得

クラスタ内のNode一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /api/v1/nodes` |
| 認証 | 必要 |
| 権限 | nodes リソースの list 権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "kind": "NodeList",
  "apiVersion": "v1",
  "items": [
    {
      "metadata": {
        "name": "worker-node-1",
        "labels": {
          "kubernetes.io/hostname": "worker-node-1",
          "kubernetes.io/os": "linux"
        }
      },
      "spec": {},
      "status": {
        "conditions": [
          {
            "type": "Ready",
            "status": "True"
          }
        ],
        "addresses": [
          {
            "type": "InternalIP",
            "address": "192.168.1.10"
          }
        ],
        "capacity": {
          "cpu": "4",
          "memory": "8Gi"
        },
        "allocatable": {
          "cpu": "3800m",
          "memory": "7Gi"
        }
      }
    }
  ]
}
```

### Job / CronJob

#### 25. Job作成

指定Namespace内に新しいJobを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /apis/batch/v1/namespaces/{namespace}/jobs` |
| 認証 | 必要 |
| 権限 | jobs リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "batch/v1",
  "kind": "Job",
  "metadata": {
    "name": "my-job"
  },
  "spec": {
    "template": {
      "spec": {
        "containers": [
          {
            "name": "worker",
            "image": "busybox",
            "command": ["echo", "Hello World"]
          }
        ],
        "restartPolicy": "Never"
      }
    },
    "backoffLimit": 4
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created` / `202 Accepted`

#### 26. CronJob作成

指定Namespace内に新しいCronJobを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /apis/batch/v1/namespaces/{namespace}/cronjobs` |
| 認証 | 必要 |
| 権限 | cronjobs リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "batch/v1",
  "kind": "CronJob",
  "metadata": {
    "name": "my-cronjob"
  },
  "spec": {
    "schedule": "*/5 * * * *",
    "jobTemplate": {
      "spec": {
        "template": {
          "spec": {
            "containers": [
              {
                "name": "worker",
                "image": "busybox",
                "command": ["date"]
              }
            ],
            "restartPolicy": "OnFailure"
          }
        }
      }
    }
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created` / `202 Accepted`

### 認証・認可

#### 27. TokenReview

トークンの有効性を検証する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /apis/authentication.k8s.io/v1/tokenreviews` |
| 認証 | 必要 |
| 権限 | tokenreviews リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "authentication.k8s.io/v1",
  "kind": "TokenReview",
  "spec": {
    "token": "eyJhbGciOiJSUzI1NiIsInR5cCI6IkpXVCJ9..."
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created`

```json
{
  "apiVersion": "authentication.k8s.io/v1",
  "kind": "TokenReview",
  "status": {
    "authenticated": true,
    "user": {
      "username": "system:serviceaccount:default:my-sa",
      "uid": "abc123",
      "groups": ["system:serviceaccounts", "system:serviceaccounts:default"]
    }
  }
}
```

#### 28. SubjectAccessReview

指定のサブジェクトが特定のアクションを実行できるか判定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /apis/authorization.k8s.io/v1/subjectaccessreviews` |
| 認証 | 必要 |
| 権限 | subjectaccessreviews リソースの create 権限 |

**リクエストボディ**

```json
{
  "apiVersion": "authorization.k8s.io/v1",
  "kind": "SubjectAccessReview",
  "spec": {
    "resourceAttributes": {
      "namespace": "default",
      "verb": "get",
      "resource": "pods"
    },
    "user": "jane"
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK` / `201 Created`

```json
{
  "apiVersion": "authorization.k8s.io/v1",
  "kind": "SubjectAccessReview",
  "status": {
    "allowed": true,
    "reason": "RBAC: allowed by ClusterRoleBinding \"cluster-admin\""
  }
}
```

### CustomResourceDefinition

#### 29. CustomResourceDefinition一覧取得

クラスタに登録されたCRD一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /apis/apiextensions.k8s.io/v1/customresourcedefinitions` |
| 認証 | 必要 |
| 権限 | customresourcedefinitions リソースの list 権限 |

**レスポンス（成功時）**

ステータスコード: `200 OK`

CustomResourceDefinitionListオブジェクト。

## 備考

### APIの規約

- 全てのKubernetes APIリソースはRESTfulパターンに従い、標準的なCRUD操作（create, get, list, update, patch, delete, deletecollection, watch）を統一的に提供する。
- リソースには必ず `apiVersion`, `kind`, `metadata` フィールドが含まれる。
- `metadata.resourceVersion` は楽観的並行性制御（Optimistic Concurrency Control）に使用される。
- Watch操作はHTTP Long Pollingにより、リソースの変更イベントをストリームとして受け取る。
- サブリソース（status, scale, log, exec等）は親リソースとは独立してアクセス制御される。

### コンテンツネゴシエーション

APIサーバーは以下のレスポンス形式をサポートする:
- `application/json` - JSON形式（デフォルト）
- `application/yaml` - YAML形式
- `application/vnd.kubernetes.protobuf` - Protocol Buffers形式
- `application/cbor` - CBOR形式
- `application/json;stream=watch` - Watch用JSONストリーム
- `application/vnd.kubernetes.protobuf;stream=watch` - Watch用Protocol Buffersストリーム
- `application/cbor-seq` - Watch用CBORシーケンス

### API Groupバージョニング

- `v1` (GA): 安定版。後方互換性が保証される。
- `v1beta1`/`v1beta2`: ベータ版。機能は完成しているが、マイナーバージョンで変更の可能性がある。
- `v1alpha1`/`v1alpha2`/`v1alpha3`: アルファ版。実験的機能。デフォルト無効で、将来削除される可能性がある。

### 参照情報

- OpenAPI仕様: `api/openapi-spec/swagger.json`（全537パス、1055オペレーション）
- ソースコード: `pkg/registry/` 配下にリソースごとのAPI実装
- ルーティング: `pkg/routes/` 配下にサーバーレベルのルート定義
- APIサーバーエントリポイント: `cmd/kube-apiserver/`
