# Kubernetes (K8s)

## 概要

Kubernetesは、複数のホストにまたがるコンテナ化されたアプリケーションの展開、スケーリング、運用管理を自動化するためのオープンソースのコンテナオーケストレーションプラットフォームである。Cloud Native Computing Foundation（CNCF）がホストするプロジェクトであり、Googleが社内で長年運用してきた大規模ワークロード管理システム「Borg」の経験を基盤として設計されている。

## 主な機能

| 機能カテゴリ | 説明 |
| --- | --- |
| コンテナオーケストレーション | 複数ホスト上のコンテナ化アプリケーションのデプロイ、メンテナンス、スケーリングを自動管理 |
| API Server (kube-apiserver) | Kubernetes APIを公開するコントロールプレーンのフロントエンド。RESTful APIを通じてクラスタ操作を提供 |
| スケジューリング (kube-scheduler) | Pod をノードに割り当てるスケジューリングロジックを提供 |
| コントローラ管理 (kube-controller-manager) | レプリケーション、エンドポイント、名前空間などのコントローラプロセスを実行 |
| ノード管理 (kubelet) | 各ノード上で動作し、Pod とコンテナの実行を管理するエージェント |
| ネットワークプロキシ (kube-proxy) | 各ノード上で動作し、Kubernetes Serviceの抽象化を実現するネットワークプロキシ |
| CLI ツール (kubectl) | Kubernetes クラスタを操作するためのコマンドラインインターフェース |
| クラスタブートストラップ (kubeadm) | Kubernetes クラスタのブートストラップを行うためのツール |
| Cloud Controller Manager | クラウドプロバイダ固有のコントローラロジックを管理 |
| 動的リソース割当 (DRA) | GPU等の動的リソースをPodに割り当てるための拡張機能 |
| Pod Security Admission | Pod のセキュリティ標準への準拠を強制するアドミッションコントローラ |
| CSI連携 | Container Storage Interface を通じた外部ストレージプロバイダとの統合 |
| CRI連携 | Container Runtime Interface を通じたコンテナランタイムとの統合 |

## 技術スタック

### コア

- 言語: Go 1.25
- モジュール: `k8s.io/kubernetes`
- データストア: etcd v3（`go.etcd.io/etcd/client/v3 v3.6.7`）
- 通信: gRPC v1.78.0 / Protocol Buffers
- API仕様: OpenAPI（`github.com/google/gnostic-models`）

### 主要ライブラリ

- CLI: `github.com/spf13/cobra v1.10.0`, `github.com/spf13/pflag v1.0.9`
- テスト: `github.com/onsi/ginkgo/v2 v2.27.4`, `github.com/onsi/gomega v1.39.0`, `github.com/stretchr/testify v1.11.1`
- メトリクス: `github.com/prometheus/client_golang v1.23.2`
- トレーシング: `go.opentelemetry.io/otel v1.39.0`
- ロギング: `go.uber.org/zap v1.27.1`, `github.com/go-logr/logr v1.4.3`
- ポリシーエンジン: `github.com/google/cel-go v0.26.0`（Common Expression Language）
- REST API: `github.com/emicklei/go-restful/v3 v3.13.0`
- コンテナ監視: `github.com/google/cadvisor v0.56.0`
- 認証: `github.com/coreos/go-oidc`, `golang.org/x/oauth2 v0.34.0`

## ディレクトリ構成

```
kubernetes-master/
├── api/                        # API定義（OpenAPI仕様、APIルール、ディスカバリ情報）
├── build/                      # ビルド用スクリプト・Docker設定
├── cluster/                    # クラスタ構築・管理スクリプト（GCE等）
├── cmd/                        # メインバイナリのエントリポイント
│   ├── kube-apiserver/         # API Serverバイナリ
│   ├── kube-controller-manager/# コントローラマネージャバイナリ
│   ├── kube-scheduler/         # スケジューラバイナリ
│   ├── kubelet/                # Kubeletバイナリ
│   ├── kube-proxy/             # Proxyバイナリ
│   ├── kubectl/                # CLIツールバイナリ
│   ├── kubeadm/                # クラスタブートストラップツール
│   ├── cloud-controller-manager/ # クラウドコントローラマネージャ
│   └── ...                     # その他ユーティリティ（gendocs, genmanなど）
├── docs/                       # ドキュメント
├── hack/                       # 開発補助スクリプト（ビルド、リント、コード生成等）
├── logo/                       # ロゴ画像
├── pkg/                        # 共有ライブラリパッケージ
│   ├── api/                    # 内部APIユーティリティ
│   ├── apis/                   # 内部APIスキーマ定義
│   ├── controller/             # コントローラ共通ロジック
│   ├── controlplane/           # コントロールプレーンロジック
│   ├── kubectl/                # kubectlの内部ロジック
│   ├── kubelet/                # kubeletの内部ロジック
│   ├── scheduler/              # スケジューラの内部ロジック
│   ├── registry/               # APIレジストリ実装
│   ├── volume/                 # ボリュームプラグイン
│   └── ...                     # その他（auth, proxy, quota等）
├── plugin/                     # プラグイン
│   └── pkg/                    # プラグインパッケージ
├── staging/                    # 外部公開用パッケージ（k8s.io/*として公開）
│   └── src/k8s.io/
│       ├── api/                # Kubernetes APIの型定義
│       ├── apimachinery/       # APIマシナリ（スキーマ、シリアライゼーション等）
│       ├── apiserver/          # 汎用APIサーバーライブラリ
│       ├── client-go/          # Goクライアントライブラリ
│       ├── kubectl/            # kubectl共有ライブラリ
│       ├── kubelet/            # kubelet APIと型定義
│       └── ...                 # その他30以上のサブプロジェクト
├── test/                       # テストスイート
│   ├── e2e/                    # E2Eテスト
│   ├── e2e_node/               # ノードE2Eテスト
│   ├── integration/            # 結合テスト
│   ├── conformance/            # 適合性テスト
│   └── ...                     # その他（fuzz, typecheck等）
├── third_party/                # サードパーティコード
├── vendor/                     # 依存パッケージ（vendoring）
├── go.mod                      # Goモジュール定義
├── go.sum                      # 依存パッケージチェックサム
├── go.work                     # Goワークスペース定義
├── Makefile                    # ビルドターゲット定義
└── LICENSE                     # Apache License 2.0
```

## セットアップ

### 必要条件

- Go 1.25.0 以上
- Dockerまたはコンテナランタイム（Docker経由でビルドする場合）
- GNU Make
- etcd（ローカルクラスタ実行時）
- bash

### Go環境でのビルド

```bash
git clone https://github.com/kubernetes/kubernetes
cd kubernetes
make
```

ビルド出力は `_output/bin` ディレクトリに配置される。特定のコンポーネントのみビルドする場合は `WHAT` 変数を指定する。

```bash
make WHAT=cmd/kubectl
make WHAT=cmd/kube-apiserver
```

### Docker環境でのビルド

```bash
git clone https://github.com/kubernetes/kubernetes
cd kubernetes
make quick-release
```

### etcdのインストール

```bash
hack/install-etcd.sh
```

### ローカルクラスタの起動

```bash
hack/local-up-cluster.sh
```

## 開発

### ビルド

```bash
# 全コンポーネントのビルド
make

# 特定コンポーネントのビルド
make WHAT=cmd/kubectl

# デバッグビルド（最適化無効）
make DBG=1

# クロスコンパイル
hack/build-cross.sh
```

### テスト

```bash
# 単体テストの実行
make test

# 特定パッケージのテスト
make test WHAT=./pkg/scheduler/...

# 結合テスト
make test-integration

# E2Eテスト
hack/ginkgo-e2e.sh

# ノードE2Eテスト
make test-e2e-node
```

### リント・コード品質

```bash
# コード検証
make verify

# 依存関係のリント
hack/lint-dependencies.sh
```

### コード生成

```bash
# ドキュメント生成
hack/generate-docs.sh

# Protobufバインディング更新
hack/update-generated-protobuf-dockerized.sh
```

### Makefileの主要ターゲット

| ターゲット | 説明 |
| --- | --- |
| `make` / `make all` | コードのビルド |
| `make test` | 単体テストの実行 |
| `make test-integration` | 結合テストの実行 |
| `make verify` | コード検証 |
| `make clean` | ビルド成果物のクリーンアップ |
| `make quick-release` | Dockerを使用したリリースビルド |
