# テスト方針書

## 概要

本ドキュメントは、Kubernetes（k8s.io/kubernetes）プロジェクトにおけるテスト方針を定義する。Kubernetesはコンテナオーケストレーションプラットフォームであり、Go言語（Go 1.25.0）で実装されている。本プロジェクトでは、単体テスト、結合テスト、E2E（End-to-End）テスト、適合性（Conformance）テスト、ファズテスト、ノードE2Eテストなど、複数のテストレベルを設け、品質を多層的に担保している。

## テスト戦略

### テストレベル

| レベル | 目的 | 担当 |
| --- | --- | --- |
| 単体テスト | 個々のパッケージ・関数の正確性を検証する。`pkg/`、`cmd/`、`staging/` 配下の `*_test.go` ファイルによって実施される | 各コンポーネントの開発者（SIG単位） |
| 結合テスト | 複数コンポーネント間の連携動作を検証する。`test/integration/` 配下に配置され、etcdなどの依存サービスを起動して実行する | 各SIGの開発者・テスト担当者 |
| E2Eテスト | 実際のKubernetesクラスタ上でのシステム全体の動作を検証する。`test/e2e/` 配下にGinkgo/Gomegaフレームワークで実装されている | SIG-Testing、各SIG |
| ノードE2Eテスト | ノードレベルのコンポーネント（kubelet等）の動作をリモートまたはローカルで検証する。`test/e2e_node/` に配置 | SIG-Node |
| 適合性テスト（Conformance） | Kubernetes APIの仕様準拠を検証する。E2Eテストのサブセットとして実施され、`test/conformance/` で管理される | SIG-Architecture |
| kubeadm E2Eテスト | kubeadmによるクラスタブートストラップの検証。`test/e2e_kubeadm/` に配置 | SIG-Cluster-Lifecycle |
| DRA E2Eテスト | Dynamic Resource Allocation機能のE2Eテスト。`test/e2e_dra/` に配置 | SIG-Node |
| ファズテスト | 入力データのランダム化による堅牢性テスト。`test/fuzz/` 配下にCBOR、JSON、YAML形式のファジングテストを配置 | 各SIG |

### テスト種別

| 種別 | 概要 | 実施タイミング |
| --- | --- | --- |
| 機能テスト | 各コンポーネントのAPI仕様・ビジネスロジックの正確性を検証 | PR提出時、マージ前 |
| 性能テスト | ベンチマークテスト（`hack/benchmark-go.sh`）によるパフォーマンス計測。`test/integration/benchmark/` にも性能関連テストを配置 | 定期実行、リリース前 |
| セキュリティテスト | 認証・認可・RBAC等のセキュリティ機能の検証。`test/integration/auth/` および `test/e2e/auth/` に配置 | PR提出時、マージ前 |
| 静的解析・検証 | `hack/verify-*.sh`（49種類）によるコード品質チェック（gofmt、golangci-lint、型チェック、boilerplate、codegen整合性等） | PR提出時（presubmit） |
| 適合性テスト | Kubernetes API仕様への準拠検証。`hack/update-conformance-yaml.sh` で適合テストリストを管理 | リリース前 |
| ファズテスト | CBOR/JSON/YAMLデコーダの堅牢性検証 | 定期実行 |

## テスト環境

| 環境 | 用途 | 構成 |
| --- | --- | --- |
| ローカル開発環境 | 単体テスト・結合テストの実行 | Go 1.25.0、etcd（`hack/install-etcd.sh` でインストール）、ローカルクラスタ（`hack/local-up-cluster.sh`） |
| CI環境（Prow） | 全テストレベルの自動実行 | Google Cloud Platform上のProwベースCI、テスト結果はARTIFACTSディレクトリに出力 |
| リモートノード環境 | ノードE2Eテスト | GCE上のリモートインスタンス（`REMOTE=true`）、containerdランタイム（デフォルトソケット: `/run/containerd/containerd.sock`） |
| クラウドプロバイダ環境 | E2Eテスト | GCE/GKE等のクラウドプロバイダ上のKubernetesクラスタ。`test/e2e/providers.go` でプロバイダ設定を管理 |

## テストツール

| ツール | 用途 | バージョン |
| --- | --- | --- |
| Go testing（`go test`） | 単体テスト・結合テストの実行エンジン | Go 1.25.0標準 |
| Ginkgo | E2Eテストフレームワーク。BDD形式でテストを記述・実行 | v2.27.4 |
| Gomega | E2Eテスト用マッチャーライブラリ。Ginkgoと組み合わせて使用 | v1.39.0 |
| google/go-cmp | テスト内での構造体比較 | v0.7.0 |
| golangci-lint | 静的解析ツール（`hack/golangci.yaml` で設定） | プロジェクト指定バージョン |
| etcd | 結合テスト用のキーバリューストア | `hack/install-etcd.sh` で管理 |
| jq | テストスクリプト内でのJSON処理 | システム依存 |
| kubectl | E2Eテスト内でのクラスタ操作 | ビルド成果物 |
| JUnit XMLレポーター | テスト結果のJUnit形式出力（`KUBE_JUNIT_REPORT_DIR`） | 標準 |

## カバレッジ目標

| 対象 | 目標値 |
| --- | --- |
| 行カバレッジ | 明示的な数値目標は設定されていない。`KUBE_COVER=y` 環境変数でカバレッジ収集を有効化し、`KUBE_COVERMODE=atomic` でアトミックモードを使用 |
| 分岐カバレッジ | 明示的な数値目標は設定されていない |
| 適合性テストカバレッジ | Kubernetes API仕様の全必須エンドポイントをカバー。`test/conformance/testdata/` でゴールデンリストを管理 |

## テストデータ

- **単体テスト**: 各テストファイル内にテストケースをテーブル駆動テスト形式で直接定義。`testdata/` ディレクトリにフィクスチャファイルを配置する場合もある
- **結合テスト**: テスト内でKubernetes APIオブジェクト（Pod、Service等）をプログラム的に生成。etcdをバックエンドとして使用
- **E2Eテスト**: `test/e2e/testing-manifests/` にテスト用マニフェスト（YAML）を配置。テスト実行時に実クラスタへデプロイ
- **ファズテスト**: `test/fuzz/` 配下にCBOR、JSON、YAML形式のファジング用コーパスを配置
- **フィクスチャ**: `test/fixtures/` にテスト用の固定データファイルを配置

## 不具合管理

- **GitHub Issues**: 不具合はGitHub Issuesで管理される。`.github/ISSUE_TEMPLATE/` にイシューテンプレートを配置
- **Flaky Test管理**: E2Eテストにおいて `[Flaky]` ラベルでフレーキーテストを管理。デフォルトではスキップされる（`SKIP` パターンに `\[Flaky\]` が含まれる）
- **テスト分類ラベル**: `[Slow]`、`[Serial]`、`[Flaky]`、`[Disruptive]` 等のGinkgoラベルでテストを分類し、実行対象を制御
- **OWNERS/レビュー**: 各ディレクトリに `OWNERS` ファイルを配置し、テストコードのレビュー担当者を明確化

## CI/CD連携

- **Prow**: KubernetesプロジェクトのCI/CDはProw（Kubernetes-nativeのCIシステム）によって管理される
- **Presubmitジョブ**: PR提出時に単体テスト、結合テスト、静的解析（`make verify`）が自動実行される
- **Postsubmitジョブ**: マージ後にE2Eテストを含む完全なテストスイートが実行される
- **Makeターゲット**: テスト実行は以下のMakeターゲットで統一的に管理される
  - `make test` / `make check`: 単体テスト実行（`hack/make-rules/test.sh`）
  - `make test-integration`: 結合テスト実行（`hack/make-rules/test-integration.sh`）。etcdの自動起動・停止を含む
  - `make test-e2e-node`: ノードE2Eテスト実行（`hack/make-rules/test-e2e-node.sh`）
  - `make verify`: 全presubmit検証の実行（`hack/make-rules/verify.sh`）
  - `make quick-verify`: 高速presubmit検証（低速テストを除外）
- **レースディテクター**: 単体テストではデフォルトで `-race` フラグが有効（`KUBE_RACE` 環境変数で制御）
- **キャッシュミューテーション検出**: `KUBE_CACHE_MUTATION_DETECTOR=true` がデフォルトで有効化され、キャッシュの不正変更を検出
- **テストタイムアウト**: 単体テスト180秒、結合テスト600秒がデフォルト（`KUBE_TIMEOUT` で変更可能）
- **並列実行**: 結合テストは `KUBE_INTEGRATION_TEST_MAX_CONCURRENCY` で並列度を制御。E2Eテストは `GINKGO_PARALLEL=y` で並列実行可能
- **テスト結果レポート**: JUnit形式のXMLレポートを `KUBE_JUNIT_REPORT_DIR` に出力。Coveralls.io連携にも対応（`KUBE_GOVERALLS_BIN`）

## 備考

- **テストの実行除外**: 単体テスト実行時（`make test`）は `third_party/`、`test/e2e`、`test/e2e_node`、`test/e2e_kubeadm`、`test/integration` が自動的に除外される
- **デバッグ支援**: E2Eテストは `E2E_TEST_DEBUG_TOOL` 環境変数でdelveまたはgdbによるデバッグ実行をサポート。`DBG=1` でデバッグ情報付きビルドが可能
- **マルチモジュール構成**: Go workspaceを使用したマルチモジュール構成であり、テスト対象パッケージの探索は `go list -m -json` によるワークスペース内全モジュールの走査で行われる
- **適合性テスト管理**: 適合性テストの追加・削除時は `hack/update-conformance-yaml.sh` でゴールデンリストを更新し、SIG-Architectureのレビューが必要
- **E2Eテストフレームワーク**: `test/e2e/framework/` に共通のテストユーティリティ（Pod操作、Node操作、デバッグ、kubectl操作等）が整備されている
