# バッチ設計書 18-update-codegen.sh

## 概要

本ドキュメントは、KubernetesプロジェクトにおけるAPIリソースのコード自動生成スクリプト（update-codegen.sh）の設計について記述する。deepcopy・conversion・defaulter・client等のボイラープレートコードを一括生成するバッチの仕様を定義する。

### 本バッチの処理概要

update-codegen.shは、Kubernetes APIリソースの型定義から各種ボイラープレートコードを自動生成する大規模なバッチスクリプトである。Go言語のコメントタグ（+k8s:タグ）を解析し、対応するコード生成ツールを実行して、プロジェクト全体の自動生成コードを更新する。

**業務上の目的・背景**：Kubernetesプロジェクトでは、APIリソースの型定義に対してdeepcopy関数、conversion関数、defaulter関数、validation関数、register関数、OpenAPIスキーマ、クライアントコード、lister、informer、apply configuration等の大量のボイラープレートコードが必要である。これらを手動で記述・維持することは非現実的であり、型定義から自動生成する仕組みが不可欠である。本スクリプトはそのコード生成パイプライン全体を統括する。

**バッチの実行タイミング**：API型定義の変更後に手動実行、またはupdate-all.sh（No.17）経由で自動実行される。

**主要な処理内容**：
1. Go環境のセットアップ（kube::golang::setup_env）とprotocのインストール
2. goimportsツールのインストール
3. +k8s:タグ付きファイルの検索と分類
4. protobufコード生成（codegen::protobuf）
5. deep-copy関数生成（codegen::deepcopy）
6. swagger型ドキュメント生成（codegen::swagger）
7. prerelease-lifecycle関数生成（codegen::prerelease）
8. defaulter関数生成（codegen::defaults）
9. validation関数生成（codegen::validation）
10. conversion関数生成（codegen::conversions）
11. register関数生成（codegen::register）
12. OpenAPIコード生成（codegen::openapi）
13. apply-configuration生成（codegen::applyconfigs）
14. クライアントコード生成（codegen::clients）
15. listerコード生成（codegen::listers）
16. informerコード生成（codegen::informers）
17. サブプロジェクトのコード生成（codegen::subprojects）
18. protobufバインディング生成（codegen::protobindings）

**前後の処理との関連**：API型定義（Batch API Types等）の変更をトリガーとして実行される。生成結果はverify-all.sh（No.16）で検証される。update-all.sh（No.17）から呼び出される。update-openapi-spec.sh（No.20）はこの生成結果を前提とする。

**影響範囲**：プロジェクト全体のzz_generated.*.goファイル、staging/src/k8s.io/client-go配下のクライアントコード、pkg/generated/openapi配下のOpenAPIコード。

## バッチ種別

コード生成

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（API型定義変更後） |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 / update-all.sh経由 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Go環境 | Goツールチェインがインストール済みであること |
| protoc | Protocol Buffers コンパイラ（PROTOC_VERSION）が利用可能、または自動インストール |
| Git | gitコマンドが利用可能であること（ファイル検索にgit ls-files使用） |
| ソースコード | Kubernetesリポジトリがクローン済みであること |

### 実行可否判定

スクリプト冒頭の`set -o errexit/nounset/pipefail`によりエラー時は即座に終了する。各コード生成ツールはGOPROXY=offでローカルモジュールからビルドされる。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| 位置引数（パターン） | 文字列 | No | 全コード生成を実行 | 実行するコード生成の名前パターン。`-?`で一覧表示 |
| DBG_CODEGEN | 環境変数 | No | 0 | デバッグログを有効化（1で有効） |
| GENERATED_FILE_PREFIX | 環境変数 | No | "zz_generated." | 生成ファイルのプレフィックス |
| UPDATE_API_KNOWN_VIOLATIONS | 環境変数 | No | 空 | trueの場合、既知のAPI違反リストを更新 |
| KUBE_VERBOSE | 環境変数 | No | 1 | ログ出力レベル |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| +k8s:タグ付きGoソースファイル | .goファイル | コメントタグによる生成指示を含む型定義ファイル |
| hack/boilerplate/boilerplate.generatego.txt | テキスト | 生成ファイルのヘッダーテンプレート |
| api/api-rules/violation_exceptions.list | テキスト | 既知のAPI違反例外リスト |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| zz_generated.deepcopy.go | .goファイル | DeepCopy/DeepCopyInto/DeepCopyObject関数 |
| zz_generated.defaults.go | .goファイル | SetDefaults_*関数 |
| zz_generated.conversion.go | .goファイル | Convert_*関数 |
| zz_generated.register.go | .goファイル | AddToScheme/Install関数 |
| zz_generated.prerelease-lifecycle.go | .goファイル | APILifecycleIntroduced/Deprecated等の関数 |
| zz_generated.validations.go | .goファイル | Validate_*関数 |
| zz_generated.openapi.go | .goファイル | OpenAPIスキーマ定義 |
| types_swagger_doc_generated.go | .goファイル | Swagger型ドキュメント |
| generated.proto / generated.pb.go | .proto/.goファイル | Protocol Buffers定義と生成コード |
| staging/src/k8s.io/client-go/ | .goファイル | クライアント・lister・informer・apply-config |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | zz_generated.{種別}.go（種別依存） |
| 出力先 | 各パッケージディレクトリ内 |
| 文字コード | UTF-8 |
| 区切り文字 | 該当なし |

## 処理フロー

### 処理シーケンス

```
1. 環境セットアップ
   ├─ kube::golang::setup_env でGo環境初期化
   ├─ goimports インストール
   └─ protoc インストール
2. タグ付きファイルの検索
   └─ git_grep で +k8s: タグを持つ全ファイルを列挙
3. 除外ディレクトリの特定
   └─ vendor, go.work, _codegenignore を除外
4. コード生成の実行（指定順序）
   ├─ codegen::protobuf（protobuf生成）
   ├─ codegen::deepcopy（DeepCopy関数生成）
   ├─ codegen::swagger（Swagger型ドキュメント生成）
   ├─ codegen::prerelease（プレリリースライフサイクル生成）
   ├─ codegen::defaults（デフォルト値設定関数生成）
   ├─ codegen::validation（バリデーション関数生成）
   ├─ codegen::conversions（型変換関数生成）
   ├─ codegen::register（スキーマ登録関数生成）
   ├─ codegen::openapi（OpenAPIスキーマ生成）
   ├─ codegen::applyconfigs（apply-configuration生成）
   ├─ codegen::clients（クライアントコード生成）
   ├─ codegen::listers（listerコード生成）
   ├─ codegen::informers（informerコード生成）
   ├─ codegen::subprojects（サブプロジェクトのコード生成）
   └─ codegen::protobindings（protobufバインディング生成）
```

### フローチャート

```mermaid
flowchart TD
    A[スクリプト開始] --> B[環境セットアップ]
    B --> C[+k8s:タグ付きファイル検索]
    C --> D{引数指定あり?}
    D -->|Yes| E[指定パターンに一致するコード生成のみ実行]
    D -->|No| F[全コード生成を順次実行]
    E --> G[コード生成実行]
    F --> G
    G --> H{全生成成功?}
    H -->|Yes| I[スクリプト終了（正常）]
    H -->|No| J[スクリプト終了（エラー）]
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | ツールビルドエラー | コード生成ツールのコンパイル失敗 | Go環境の確認、依存モジュールの確認 |
| 1 | 生成エラー | コード生成実行時のエラー | 型定義の整合性を確認 |
| 1 | protocエラー | protocが利用不可 | hack/install-protoc.shを実行 |
| 1 | API違反エラー | OpenAPI生成時のAPI違反レポート差異 | api/api-rules/README.mdを参照 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 自動リトライなし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

エラー発生時は、引数でパターンを指定して個別のコード生成を実行し、問題の特定を行う（例: `hack/update-codegen.sh deepcopy`）。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 各コード生成は既存ファイルを削除後に再生成 |
| コミットタイミング | 該当なし |
| ロールバック条件 | 該当なし（git resetで復元可能） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | プロジェクト全体の自動生成コード |
| 目標処理時間 | 環境依存（数十分〜1時間以上） |
| メモリ使用量上限 | 各生成ツールに依存 |

## 排他制御

同一環境での同時実行は禁止。生成ファイルの削除・再生成プロセスで競合が発生する。各コード生成は既存のzz_generated.*ファイルをxargs -0 rm -fで一括削除してから再生成するため、途中中断時はファイルが欠損する。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 進捗ログ | 各コード生成開始時 | "Generating {種別} code for N targets" |
| デバッグログ | DBG_CODEGEN=1時 | 対象ディレクトリ一覧、詳細な実行状況 |
| エラーログ | API違反検出時 | "API rule check failed" |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 生成成否 | 終了コード != 0 | CI/CDシステム |

## 備考

- 本スクリプトはKubernetesプロジェクトのコード生成パイプラインの中核を担う大規模スクリプト（1000行超）
- `-?`引数で利用可能なコード生成一覧を表示可能
- 各コード生成は関数として定義されており（codegen::*）、引数パターンで個別実行が可能
- protobufとswaggerの生成は後続のコード生成の依存元であり、先に実行される
- LC_ALL="C"でソート順を統一し、環境間での差異を防止
- GOPROXY=offで各生成ツールをローカルモジュールからビルドする
- git ls-filesを使用してファイル検索を行うため、gitリポジトリ内での実行が必要
- vendor/ディレクトリおよびgo.workファイルを含むディレクトリは除外される
- ソースコード: `hack/update-codegen.sh`
