# バッチ設計書 21-update-vendor.sh

## 概要

本ドキュメントは、Kubernetesプロジェクトにおけるvendorディレクトリ更新スクリプト `hack/update-vendor.sh` のバッチ設計書である。Go依存モジュールのベンダリングをバッチ処理として一括実行するスクリプトの仕様を定義する。

### 本バッチの処理概要

本バッチは、KubernetesプロジェクトのGo依存モジュール（vendor）を最新の状態に更新し、プロジェクト全体の依存関係の一貫性を保証するためのバッチ処理である。

**業務上の目的・背景**：Kubernetesプロジェクトはマルチモジュール構成であり、ルートモジュールと多数のstagingモジュール（k8s.io配下）が相互に参照し合っている。依存関係の不整合はビルド失敗やランタイムエラーの原因となるため、vendorディレクトリの一括更新処理が不可欠である。本バッチは、依存関係の追加・更新・削除後に実行され、全モジュールの整合性を保証する。

**バッチの実行タイミング**：随時実行（手動）。依存関係の変更時（`hack/pin-dependency.sh`実行後）、または定期的なメンテナンス時に開発者が手動で実行する。CI/CDパイプラインの一部としても実行される場合がある。

**主要な処理内容**：
1. Go環境のセットアップとGOPROXY設定の検証
2. stagingリポジトリのgo.modファイル初期化・go/godebugディレクティブの更新
3. ルート・stagingモジュール間のreplace/requireディレクティブの相互参照設定
4. go.workファイルの更新（全モジュールのuse登録）
5. `go work sync`によるMVS（Minimum Version Selection）の全モジュールへの伝播
6. 全モジュールの`go mod tidy`実行とディレクティブのグルーピング整形
7. 未使用replaceディレクティブの削除
8. go.modファイルへの自動生成コメントの追加
9. 内部モジュールの更新（`hack/update-internal-modules.sh`呼び出し）
10. `go work vendor`によるvendorディレクトリの再構築
11. vendorライセンスファイルの更新とOWNERSファイルの生成
12. stagingモジュールからk8s.io/kubernetesへの循環依存の検出・禁止

**前後の処理との関連**：前提として`hack/pin-dependency.sh`で個別の依存バージョンを固定した後に実行されることが多い。後続処理として`hack/verify-vendor.sh`による検証、及びvendor配下の変更をgit commitする作業が続く。`hack/update-internal-modules.sh`を内部で呼び出す。

**影響範囲**：プロジェクトルートおよび全stagingモジュールのgo.mod/go.sum、go.work/go.work.sum、vendorディレクトリ全体、vendor/LICENSES、vendor/OWNERSファイル。

## バッチ種別

依存関係管理 / コード生成

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時 |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動実行 / CI連携 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Go環境 | Goランタイムがインストールされ、PATHに含まれていること |
| jq | jqコマンドがインストールされていること（`kube::util::require-jq`で検証） |
| GOPROXY | `GOPROXY=off`に設定されていないこと（依存解決が不可能になるため） |
| ネットワーク接続 | Goモジュールプロキシへのアクセスが可能であること |
| ルートgo.mod | `go 1.x.y`ディレクティブと`godebug default=go1.x`ディレクティブが含まれていること |

### 実行可否判定

スクリプト冒頭で`GOPROXY=off`のチェックを行い、該当する場合はエラーメッセージを出力して即座に終了する（exit 1）。また、ルートgo.modにgo/godebugディレクティブが存在しない場合もエラー終了する。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| TMP_DIR | 環境変数 | No | `/tmp/update-vendor.XXXX`（mktemp生成） | 一時ファイルの格納ディレクトリ |
| LOG_FILE | 環境変数 | No | `${TMP_DIR}/update-vendor.log` | ログ出力先ファイルパス |
| GOPROXY | 環境変数 | No | システムデフォルト | Goモジュールプロキシ設定（`off`不可） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| go.mod（ルート） | Goモジュールファイル | プロジェクトルートのモジュール定義 |
| staging/src/k8s.io/*/go.mod | Goモジュールファイル | 各stagingモジュールのモジュール定義 |
| go.work | Goワークスペースファイル | マルチモジュールワークスペース定義 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| go.mod（ルート及びstaging各モジュール） | Goモジュールファイル | 更新されたモジュール定義（require/replace整形済み） |
| go.work | Goワークスペースファイル | 更新されたワークスペース定義 |
| go.work.sum | Goチェックサム | ワークスペースの依存チェックサム |
| vendor/ | ディレクトリ | ベンダリングされた依存モジュール群 |
| vendor/LICENSES | テキスト | 依存モジュールのライセンス情報 |
| vendor/OWNERS | YAML | vendorディレクトリのOWNERSファイル（dep-approvers/dep-reviewers） |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | go.mod, go.work, vendor/OWNERS, vendor/LICENSES 等 |
| 出力先 | プロジェクトルート及びstaging各ディレクトリ |
| 文字コード | UTF-8 |
| 区切り文字 | 該当なし |

## 処理フロー

### 処理シーケンス

```
1. 環境セットアップ
   └─ Go環境変数設定（GOWORK=off, GO111MODULE=on, GOFLAGS=-mod=mod, LANG=C, LC_ALL=C）
2. 前提条件チェック
   └─ GOPROXY=offチェック、jq存在確認、go/godebugディレクティブ取得
3. stagingモジュール初期化
   └─ 各staging/src/k8s.io/*/go.modの存在確認・初期化、go/godebugディレクティブ設定
4. 相互参照設定
   └─ ルート・stagingモジュール間のrequire v0.0.0 / replace（ローカルパス）設定
5. go.work更新
   └─ go work init（未存在時）、useディレクティブの刷新、go/godebug設定
6. go work sync
   └─ MVSの全モジュールへの伝播
7. go mod tidy
   └─ 全staging + ルートモジュールのtidy実行、ディレクティブグルーピング
8. 未使用replace削除
   └─ 使用されていないreplaceディレクティブの検出・削除
9. 自動生成コメント追加
   └─ go.modファイル先頭へのコメント挿入
10. 内部モジュール更新
    └─ hack/update-internal-modules.sh呼び出し
11. vendorディレクトリ再構築
    └─ go.work.sum再生成、go work vendor実行
12. ライセンス・OWNERS更新
    └─ vendor/LICENSES更新、vendor/OWNERS生成
13. 循環依存チェック
    └─ staging→k8s.io/kubernetes依存の禁止、推移的循環依存の禁止
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[Go環境セットアップ]
    B --> C{GOPROXY=off?}
    C -->|Yes| ERR1[エラー終了]
    C -->|No| D[jq存在確認]
    D --> E[go/godebugディレクティブ取得]
    E --> F{ディレクティブ存在?}
    F -->|No| ERR2[エラー終了]
    F -->|Yes| G[stagingモジュール初期化]
    G --> H[相互参照replace/require設定]
    H --> I[go.work更新]
    I --> J[go work sync]
    J --> K[全モジュール go mod tidy]
    K --> L[未使用replace削除]
    L --> M[自動生成コメント追加]
    M --> N[内部モジュール更新]
    N --> O[go work vendor実行]
    O --> P[ライセンス・OWNERS更新]
    P --> Q{循環依存チェック}
    Q -->|検出| ERR3[エラー終了]
    Q -->|なし| R[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベースを使用しない。ファイルシステム上のgo.mod/go.work/vendorディレクトリを操作対象とする。

| 処理 | 対象 | 操作種別 | 概要 |
|-----|------|---------|------|
| go.mod更新 | go.mod（ルート+staging） | UPDATE | require/replaceディレクティブの追加・削除・整形 |
| go.work更新 | go.work | UPDATE | useディレクティブの刷新 |
| vendor再構築 | vendor/ | DELETE+CREATE | vendorディレクトリの再構築 |

### テーブル別操作詳細

該当なし（データベース不使用）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| exit 1 | 環境設定エラー | GOPROXY=offが設定されている | GOPROXY設定を変更して再実行 |
| exit 1 | go.mod不正 | go 1.x.yディレクティブが存在しない | ルートgo.modにgoディレクティブを追加 |
| exit 1 | go.mod不正 | godebug default=go1.xディレクティブが存在しない | ルートgo.modにgodebugディレクティブを追加 |
| exit 1 | 循環依存検出 | staging→k8s.io/kubernetesへの依存が存在 | stagingモジュールの依存関係を修正 |
| exit 1 | 推移的循環依存 | k8s.io/kubernetes→*→k8s.io/kubernetesの依存パスが存在 | 該当依存パスを除去 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（スクリプト内にリトライロジックなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

スクリプトは`set -o errexit`によりコマンド失敗時に即座に終了する。`trap finish EXIT`により終了時にログファイルのパスが表示される。詳細なエラー内容は`${LOG_FILE}`（デフォルト: `/tmp/update-vendor.XXXX/update-vendor.log`）に記録される。障害時は当該ログファイルを確認し、原因を特定した上で再実行する。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | なし（ファイルシステム操作のためトランザクション制御なし） |
| コミットタイミング | 該当なし |
| ロールバック条件 | スクリプト途中失敗時はgit checkoutにより変更を戻す（手動） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | stagingモジュール数に依存（数十モジュール） |
| 目標処理時間 | 数分〜十数分（ネットワーク速度・モジュール数に依存） |
| メモリ使用量上限 | 特に制限なし（Go toolchainの標準的なメモリ使用量） |

## 排他制御

本スクリプトは同一プロジェクトディレクトリで同時に複数実行することを想定していない。go.mod/go.work/vendorディレクトリへの同時書き込みは不整合の原因となるため、排他的に実行する必要がある。スクリプト自体にはロック機構は実装されていない。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | スクリプト開始時 | ログファイルパスの表示 |
| 進捗ログ | 各主要ステップ実行時 | `go.mod: update staging module references`等のステータスメッセージ（`kube::log::status`経由、fd 11=stdout） |
| 詳細ログ | 全処理 | `set -x`によるコマンドトレース（`${LOG_FILE}`へ自動出力） |
| エラーログ | エラー発生時 | エラーメッセージ（fd 22=stderr経由で画面表示） |
| 終了ログ | エラー終了時 | ログファイル参照を促すメッセージ |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 終了コード | 0以外 | CI/CDパイプラインの実行結果通知 |
| 処理時間 | プロジェクト固有の設定に依存 | CIジョブタイムアウト |

## 備考

- 本スクリプトは`set -x`により全コマンドをログファイルにトレース出力する。画面出力（fd 11/22）とログ出力を分離する設計になっている。
- `go env`の出力もログに記録されるため、問題発生時のデバッグに活用できる。
- vendor/OWNERSファイルは`no_parent_owners: true`を設定し、ルートのapprovalが再帰的に適用されないようにしている。
- 参照ドキュメント: https://git.k8s.io/community/contributors/devel/sig-architecture/vendor.md
