# 帳票設計書 10-Ingress一覧

## 概要

本ドキュメントは、Kubernetes の Ingress リソース一覧をテーブル形式で標準出力に出力する帳票の設計を定義する。`kubectl get ingress` コマンド実行時に使用される。

### 本帳票の処理概要

本帳票は、Kubernetes クラスタ上の Ingress リソースの一覧情報を、テーブル形式で標準出力に表示する処理を行う。

**業務上の目的・背景**：Ingress はクラスタ外部からのHTTP/HTTPS トラフィックをクラスタ内の Service にルーティングするためのリソースである。Web アプリケーションの公開、TLS 終端、パスベースルーティング等のリバースプロキシ設定を管理する。運用者が各 Ingress のクラス、ホスト名、アドレス、ポート設定を一覧で確認し、外部公開の状態を把握するために本帳票が必要となる。

**帳票の利用シーン**：外部公開エンドポイントの一覧確認、IngressClass の確認、ホスト名ルーティング設定の確認、TLS 設定状態（80/443 ポート）の確認、LoadBalancer アドレスの確認。

**主要な出力内容**：
1. Ingress 名（Name）
2. IngressClass 名（Class）
3. ホスト名一覧（Hosts）
4. LoadBalancer アドレス（Address）
5. ポート（Ports）
6. 作成からの経過時間（Age）

**帳票の出力タイミング**：ユーザが `kubectl get ingress` コマンドを実行した際に出力される。

**帳票の利用者**：Kubernetes クラスタ管理者、ネットワークエンジニア、アプリケーション開発者。

## 帳票種別

一覧表（リソース一覧出力）

## 利用画面

| 画面No | 画面名 | URL/ルーティング | 出力操作 |
|--------|--------|-----------------|---------|
| N/A | ターミナル（CLI） | N/A | `kubectl get ingress [flags]` コマンド実行 |

## 出力形式

### 基本仕様

| 項目 | 内容 |
|-----|------|
| ファイル形式 | 標準出力（テキスト / テーブル形式） |
| 用紙サイズ | N/A |
| 向き | N/A |
| ファイル名 | N/A |
| 出力方法 | 標準出力（stdout） |
| 文字コード | UTF-8 |

### テーブル出力固有設定

| 項目 | 内容 |
|-----|------|
| カラム区切り | タブ文字（tabwriter） |
| ヘッダー表示 | デフォルト有効 |
| Wide 出力 | 追加カラムなし（全カラムが Priority=0） |

## 帳票レイアウト

### レイアウト概要

```
┌──────────────────────────────────────────────────────────────────────┐
│  NAME      CLASS    HOSTS              ADDRESS        PORTS     AGE │
├──────────────────────────────────────────────────────────────────────┤
│  web       nginx    example.com        10.0.0.1       80, 443   5d │
│  api       <none>   api.example.com    203.0.113.50   80        3d │
│  multi     nginx    a.com,b.com,c.com  10.0.0.2       80        1d │
└──────────────────────────────────────────────────────────────────────┘
```

### ヘッダー部

| No | 項目名 | 説明 | データ取得元 | 表示形式 |
|----|-------|------|-------------|---------|
| 1 | Name | Ingress の名前 | `obj.ObjectMeta.Name` | 文字列（name フォーマット） |
| 2 | Class | IngressClass 名 | `obj.Spec.IngressClassName` | 文字列、未設定時は `<none>` |
| 3 | Hosts | ルーティング対象のホスト名 | `obj.Spec.Rules` | `formatHosts` で整形（最大 3 件 + more） |
| 4 | Address | LoadBalancer の IP/ホスト名 | `obj.Status.LoadBalancer` | `ingressLoadBalancerStatusStringer` で整形 |
| 5 | Ports | ポート（80 または 80, 443） | `obj.Spec.TLS` | `formatPorts` で整形 |
| 6 | Age | 経過時間 | `obj.ObjectMeta.CreationTimestamp` | `translateTimestampSince` |

### 明細部

明細部は各 Ingress につき 1 行。Wide 出力時の追加カラムはない。

### フッター部

フッター部は存在しない。

## 出力条件

### 抽出条件

| 条件名 | 説明 | 必須 |
|-------|------|-----|
| Namespace | 対象 Namespace | No |
| All Namespaces | 全 Namespace | No |
| Label Selector | ラベルフィルタ | No |
| Field Selector | フィールドフィルタ | No |

### ソート順

| 優先度 | 項目 | 昇順/降順 |
|-------|------|---------|
| 1 | Ingress 名（デフォルト） | 昇順 |

### 改ページ条件

改ページは発生しない。

## データベース参照仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 結合条件 |
|-----------|------|---------|
| etcd: `/registry/ingresses/{namespace}/{name}` | Ingress オブジェクトの永続化ストア | N/A |

### テーブル別参照項目詳細

#### Ingress リソース（API: `networking.k8s.io/v1/ingresses`）

| 参照項目（フィールド） | 帳票項目との対応 | 取得条件 | 備考 |
|-------------------|----------------|---------|------|
| `metadata.name` | Name | N/A | Ingress の名前 |
| `metadata.creationTimestamp` | Age | N/A | 経過時間に変換 |
| `spec.ingressClassName` | Class | N/A | *string 型、nil なら "<none>" |
| `spec.rules` | Hosts | N/A | formatHosts で整形 |
| `spec.rules[].host` | Hosts（個別） | N/A | 空の場合はスキップ |
| `spec.tls` | Ports | N/A | TLS 設定の有無でポート表示が変わる |
| `status.loadBalancer.ingress` | Address | N/A | IP または Hostname |

## 計算仕様

### 計算項目一覧

| 項目名 | 計算式 | 端数処理 | 備考 |
|-------|-------|---------|------|
| Class | `obj.Spec.IngressClassName != nil ? *IngressClassName : "<none>"` | N/A | |
| Hosts | `formatHosts(obj.Spec.Rules)` | N/A | 最大 3 件、超過時は "+ N more..." |
| Address | `ingressLoadBalancerStatusStringer(obj.Status.LoadBalancer, wide)` | N/A | 非 wide 時は 16 文字で切り詰め |
| Ports | `formatPorts(obj.Spec.TLS)` | N/A | TLS あり: "80, 443"、なし: "80" |

#### Hosts 表示ロジック詳細（formatHosts 関数、行 1423-1443）

1. Rules を順に走査、Host が空でないものを最大 3 件まで収集
2. 3 件を超える場合 `"{host1},{host2},{host3} + N more..."` と表示
3. Host が 1 件もなければ `"*"` を表示

#### Address 表示ロジック詳細（ingressLoadBalancerStatusStringer、行 1468-1486）

1. `Status.LoadBalancer.Ingress` を走査
2. IP または Hostname を sets.NewString に追加（重複排除）
3. ソートしてカンマ区切りで結合
4. 非 wide 出力時は 16 文字（loadBalancerWidth）で切り詰め（末尾 "..."）

## 処理フロー

### 出力フロー

```mermaid
flowchart TD
    A[kubectl get ingress 実行] --> B[API サーバへリクエスト]
    B --> C[Ingress リスト取得]
    C --> D[printIngressList]
    D --> E[printIngress で各行生成]
    E --> F[formatHosts でホスト文字列生成]
    E --> G[ingressLoadBalancerStatusStringer でアドレス文字列生成]
    E --> H[formatPorts でポート文字列生成]
    F --> I[HumanReadablePrinter でテーブル出力]
    G --> I
    H --> I
    I --> J[標準出力]
```

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 表示メッセージ | 対処方法 |
|----------|---------|--------------|---------|
| データなし | Ingress が存在しない | `No resources found in {namespace} namespace.` | 条件見直し |
| API エラー | 接続失敗 | API エラーメッセージ | kubeconfig 確認 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定データ件数 | 数件〜数百件 |
| 目標出力時間 | 即時 |
| 同時出力数上限 | N/A |

## セキュリティ考慮事項

- Ingress 一覧取得には `ingresses` リソース（networking.k8s.io グループ）の `list` / `get` 権限が必要
- 外部公開ホスト名と IP アドレスが表示されるため、ネットワーク構成情報の露出に注意

## 備考

- Wide 出力に追加カラムはない（全カラムが Priority=0）
- Hosts は最大 3 件表示の制限がある
- Address の切り詰めは loadBalancerWidth（16 文字）で行われる
- TLS 設定の有無によりポート表示が "80" または "80, 443" に変わる

---

## コードリーディングガイド

本帳票を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | printers.go | `pkg/printers/internalversion/printers.go` | ingressColumnDefinitions（行 212-219）: Name, Class, Hosts, Address, Ports, Age の 6 カラム。全て Priority=0 |

**読解のコツ**: Ingress は Wide 追加カラムがない珍しいリソース。また、Ports は Spec.Ports ではなく Spec.TLS の有無で判定される点に注意。

#### Step 2: print 関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | printers.go | `pkg/printers/internalversion/printers.go` | `printIngress`（行 1452-1466）: IngressClassName nil チェック、formatHosts / ingressLoadBalancerStatusStringer / formatPorts の呼び出し |
| 2-2 | printers.go | `pkg/printers/internalversion/printers.go` | `formatHosts`（行 1423-1443）: Rules からホスト名を最大 3 件取得、超過時 "+ N more..." |
| 2-3 | printers.go | `pkg/printers/internalversion/printers.go` | `formatPorts`（行 1445-1450）: TLS 配列が空でなければ "80, 443"、空なら "80" |
| 2-4 | printers.go | `pkg/printers/internalversion/printers.go` | `ingressLoadBalancerStatusStringer`（行 1468-1486）: LB ステータスから IP/Hostname を取得、ソート、切り詰め |
| 2-5 | printers.go | `pkg/printers/internalversion/printers.go` | `printIngressList`（行 1488-1498）: Items を順に委譲 |

**主要処理フロー**:
- **行 1456-1459**: IngressClassName が nil なら "<none>"、そうでなければ値を使用
- **行 1460**: `formatHosts(obj.Spec.Rules)` でホスト文字列を生成
- **行 1461**: `ingressLoadBalancerStatusStringer(obj.Status.LoadBalancer, options.Wide)` でアドレスを生成
- **行 1462**: `formatPorts(obj.Spec.TLS)` でポート文字列を生成
- **行 1463**: `translateTimestampSince(obj.CreationTimestamp)` で Age を生成
- **行 1464**: Name, Class, Hosts, Address, Ports, Age をセルに追加

### プログラム呼び出し階層図

```
kubectl get ingress
    |
    +-- GetOptions.Run() [get.go]
           |
           +-- API Server Request
           |       |
           |       +-- printIngressList() [printers.go 行1488]
           |               |
           |               +-- printIngress() [printers.go 行1452]
           |                       |
           |                       +-- formatHosts() [printers.go 行1423]
           |                       +-- ingressLoadBalancerStatusStringer() [printers.go 行1468]
           |                       +-- formatPorts() [printers.go 行1445]
           |                       +-- translateTimestampSince()
           |
           +-- HumanReadablePrinter.PrintObj()
```

### データフロー図

```
[入力]                          [処理]                              [出力]

API Server (etcd)          printIngress()                     標準出力 (stdout)
  Ingress Object       --> - IngressClassName nil チェック  --> テーブル形式テキスト
  (networking.Ingress)     - formatHosts()                     NAME CLASS HOSTS ADDRESS ...
                           - ingressLoadBalancerStatus...      web  nginx example.com 10.0.0.1 ...
                           - formatPorts()
                           - TableRow 生成
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| printers.go | `pkg/printers/internalversion/printers.go` | ソース | Ingress 用カラム定義と print 関数（行 212-221, 1423-1498） |
| tablegenerator.go | `pkg/printers/tablegenerator.go` | ソース | テーブル生成基盤 |
| tableprinter.go | `staging/src/k8s.io/cli-runtime/pkg/printers/tableprinter.go` | ソース | テーブルレンダリング |
| get.go | `staging/src/k8s.io/kubectl/pkg/cmd/get/get.go` | ソース | kubectl get コマンドエントリーポイント |
