# 帳票設計書 12-StatefulSet一覧

## 概要

本ドキュメントは、Kubernetes StatefulSetリソースの一覧をテーブル形式で出力する帳票の設計書である。`kubectl get statefulsets` コマンド実行時にStatefulSetリソースの情報を人間が読みやすいテーブル形式で標準出力に表示する。

### 本帳票の処理概要

StatefulSetリソースの一覧情報を取得し、テーブル形式で出力する処理を行う。

**業務上の目的・背景**：StatefulSetはステートフルなアプリケーション（データベース、メッセージキューなど）を管理するためのワークロードリソースである。各Podに固有のID・永続ストレージが割り当てられるため、その稼働状況（Ready数/Desired数）を一覧で把握することがクラスタ運用において重要である。障害発生時のトラブルシューティングや、スケーリング状態の確認に不可欠な帳票である。

**帳票の利用シーン**：StatefulSetの稼働状態を確認する場合、Ready状態のPod数とDesired数が一致しているか監視する場合、StatefulSetのスケーリング状況を確認する場合に利用される。

**主要な出力内容**：
1. StatefulSet名
2. Ready状態（readyReplicas/desiredReplicas形式）
3. 作成からの経過時間（Age）
4. コンテナ名一覧（Wide出力時）
5. イメージ名一覧（Wide出力時）

**帳票の出力タイミング**：ユーザが `kubectl get statefulsets` または `kubectl get sts` コマンドを実行した際に出力される。

**帳票の利用者**：Kubernetesクラスタ管理者、SREエンジニア、アプリケーション開発者

## 帳票種別

一覧表（リソース一覧出力）

## 利用画面

| 画面No | 画面名 | URL/ルーティング | 出力操作 |
|--------|--------|-----------------|---------|
| - | CLIターミナル | - | `kubectl get statefulsets` コマンド実行 |

## 出力形式

### 基本仕様

| 項目 | 内容 |
|-----|------|
| ファイル形式 | テキスト（標準出力） |
| 用紙サイズ | N/A（ターミナル出力） |
| 向き | N/A |
| ファイル名 | N/A（標準出力）|
| 出力方法 | 標準出力（stdout） |
| 文字コード | UTF-8 |

## 帳票レイアウト

### レイアウト概要

タブ区切りのテーブル形式で、通常モードとWideモードの2パターンが存在する。

```
[通常モード]
NAME      READY   AGE
mysql     3/3     10d
redis     2/2     5d

[Wideモード (-o wide)]
NAME      READY   AGE   CONTAINERS   IMAGES
mysql     3/3     10d   mysql        mysql:8.0
redis     2/2     5d    redis        redis:7.0
```

### ヘッダー部

| No | 項目名 | 説明 | データ取得元 | 表示形式 |
|----|-------|------|-------------|---------|
| 1 | NAME | StatefulSet名 | `obj.Name` | 文字列 |
| 2 | READY | Ready Pod数 / Desired Pod数 | `obj.Status.ReadyReplicas` / `obj.Spec.Replicas` | "N/M"形式 |
| 3 | AGE | 作成からの経過時間 | `obj.CreationTimestamp` | 相対時間（例: 10d, 5h） |
| 4 | CONTAINERS | コンテナ名一覧（Wide時のみ） | `obj.Spec.Template.Spec.Containers` | カンマ区切り文字列 |
| 5 | IMAGES | コンテナイメージ一覧（Wide時のみ） | `obj.Spec.Template.Spec.Containers` | カンマ区切り文字列 |

### 明細部

ヘッダーと同一カラムで各StatefulSetリソースの情報を1行ずつ出力する。

### フッター部

フッター部は存在しない。

## 出力条件

### 抽出条件

| 条件名 | 説明 | 必須 |
|-------|------|-----|
| リソースタイプ | StatefulSetリソースを対象とする | Yes |
| Namespace | 指定されたNamespace内のリソースを対象とする | No（デフォルトは現在のNamespace） |
| ラベルセレクタ | `-l` オプションで指定されたラベルによるフィルタリング | No |
| フィールドセレクタ | `--field-selector` オプションで指定されたフィールドによるフィルタリング | No |

### ソート順

| 優先度 | 項目 | 昇順/降順 |
|-------|------|---------|
| 1 | APIサーバからの返却順（デフォルト） | 昇順 |

### 改ページ条件

改ページは発生しない（ターミナル出力のため）。

## データベース参照仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 結合条件 |
|-----------|------|---------|
| etcd: /registry/statefulsets/{namespace}/{name} | StatefulSetリソースの保存先 | Namespace + リソース名で参照 |

### テーブル別参照項目詳細

#### StatefulSet リソース

| 参照項目（フィールド名） | 帳票項目との対応 | 取得条件 | 備考 |
|-------------------|----------------|---------|------|
| `metadata.name` | NAME | 全件取得 | そのまま文字列出力 |
| `metadata.creationTimestamp` | AGE | 全件取得 | 相対時間に変換 |
| `spec.replicas` | READY（分母） | 全件取得 | Desired数 |
| `status.readyReplicas` | READY（分子） | 全件取得 | Ready数 |
| `spec.template.spec.containers[*].name` | CONTAINERS | Wide時のみ | layoutContainerCells関数で整形 |
| `spec.template.spec.containers[*].image` | IMAGES | Wide時のみ | layoutContainerCells関数で整形 |

## 計算仕様

### 計算項目一覧

| 項目名 | 計算式 | 端数処理 | 備考 |
|-------|-------|---------|------|
| READY | fmt.Sprintf("%d/%d", readyReplicas, desiredReplicas) | N/A | 整数表示 |
| AGE | 現在時刻 - CreationTimestamp | N/A | `translateTimestampSince`関数で変換 |
| CONTAINERS | layoutContainerCells(containers) | N/A | コンテナ名のカンマ区切り |
| IMAGES | layoutContainerCells(containers) | N/A | イメージ名のカンマ区切り |

## 処理フロー

### 出力フロー

```mermaid
flowchart TD
    A[kubectl get statefulsets 実行] --> B[コマンドパラメータ解析]
    B --> C[API Server へ LIST リクエスト]
    C --> D[StatefulSet リソース一覧取得]
    D --> E[printStatefulSet 関数で各行生成]
    E --> F{Wideモード?}
    F -->|Yes| G[コンテナ名・イメージ名を追加]
    F -->|No| H[基本カラムのみ]
    G --> I[HumanReadablePrinter でテーブル整形]
    H --> I
    I --> J[標準出力へ出力]
```

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 表示メッセージ | 対処方法 |
|----------|---------|--------------|---------|
| リソース未存在 | 指定した名前のStatefulSetが存在しない | `Error from server (NotFound): statefulsets.apps "<名前>" not found` | 正しいリソース名を指定 |
| Namespace未存在 | 指定したNamespaceが存在しない | `Error from server (NotFound): namespaces "<名前>" not found` | 正しいNamespace名を指定 |
| 権限不足 | StatefulSetリソースへのアクセス権限がない | `Error from server (Forbidden): ...` | RBACで適切な権限を付与 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定データ件数 | 数件〜数百件 |
| 目標出力時間 | 1秒以内 |
| 同時出力数上限 | N/A（CLIコマンド） |

## セキュリティ考慮事項

StatefulSetリソースの閲覧にはRBACによる `get` / `list` 権限が必要。StatefulSetはNamespaceスコープのリソースである。Podテンプレート内のコンテナイメージ名が表示されるが、機密情報ではないため特別なマスキング処理は不要。

## 備考

- `kubectl get sts` は `kubectl get statefulsets` の省略形
- Wide出力（`-o wide`）ではContainers列とImages列が追加表示される
- READYカラムのフォーマットは `readyReplicas/desiredReplicas` で、Deploymentと同様の表示形式

---

## コードリーディングガイド

本帳票を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

StatefulSetリソースのAPI型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.go | `staging/src/k8s.io/api/apps/v1/types.go` | StatefulSet, StatefulSetSpec, StatefulSetStatus構造体の定義 |

**読解のコツ**: `Spec.Replicas`はDesired数、`Status.ReadyReplicas`はReady数を表す。

#### Step 2: エントリーポイントを理解する

テーブル出力のためのカラム定義と印刷関数の登録を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | printers.go | `pkg/printers/internalversion/printers.go` | AddHandlers関数内のstatefulSetColumnDefinitions（行232-240） |

**主要処理フロー**:
1. **行232-238**: statefulSetColumnDefinitionsでカラム定義（Name, Ready, Age, Containers[Priority:1], Images[Priority:1]）
2. **行239-240**: TableHandlerにprintStatefulSetとprintStatefulSetListを登録

#### Step 3: 印刷関数を理解する

実際のテーブル行生成ロジックを読み解く。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | printers.go | `pkg/printers/internalversion/printers.go` | printStatefulSet関数（行1534-1547） |

**主要処理フロー**:
- **行1538**: `obj.Spec.Replicas`でDesired数を取得
- **行1539**: `obj.Status.ReadyReplicas`でReady数を取得
- **行1541**: `fmt.Sprintf("%d/%d", readyReplicas, desiredReplicas)` でREADY文字列を生成
- **行1542-1544**: Wideモード時、`layoutContainerCells`でコンテナ名・イメージ名を追加

### プログラム呼び出し階層図

```
kubectl get statefulsets
    |
    +-- NewCmdGet (pkg/cmd/get/get.go)
    |      +-- RunGet
    |             +-- API Server LIST /apis/apps/v1/namespaces/{ns}/statefulsets
    |
    +-- AddHandlers (pkg/printers/internalversion/printers.go:93)
    |      +-- TableHandler(statefulSetColumnDefinitions, printStatefulSet) (行239)
    |
    +-- printStatefulSet (pkg/printers/internalversion/printers.go:1534)
    |      +-- layoutContainerCells (Wideモード時)
    |      +-- translateTimestampSince
    |
    +-- HumanReadablePrinter.PrintObj (cli-runtime/pkg/printers/tableprinter.go)
           +-- tabwriter による整形出力
```

### データフロー図

```
[入力]                        [処理]                           [出力]

API Server               printStatefulSet()               標準出力(stdout)
(StatefulSet             行1534-1547                      テーブル形式
 リソース一覧)   -------> - Ready文字列生成         -------> NAME | READY | AGE
                         - Age計算                        (Wide時: + CONTAINERS | IMAGES)
                         - Container/Image取得(Wide時)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| printers.go | `pkg/printers/internalversion/printers.go` | ソース | カラム定義とprintStatefulSet関数 |
| tableprinter.go | `staging/src/k8s.io/cli-runtime/pkg/printers/tableprinter.go` | ソース | HumanReadablePrinterによるテーブル出力エンジン |
| types.go | `staging/src/k8s.io/api/apps/v1/types.go` | ソース | StatefulSet API型定義 |
| printers_test.go | `pkg/printers/internalversion/printers_test.go` | テスト | printStatefulSetのテストケース |
