# 帳票設計書 17-Secret一覧

## 概要

本ドキュメントは、Kubernetes Secretリソースの一覧をテーブル形式で出力する帳票の設計書である。`kubectl get secrets` コマンド実行時にSecretリソースの情報を人間が読みやすいテーブル形式で標準出力に表示する。

### 本帳票の処理概要

Secretリソースの一覧情報を取得し、テーブル形式で出力する処理を行う。

**業務上の目的・背景**：Secretはパスワード、OAuthトークン、TLS証明書などの機密情報を安全に保管するためのリソースである。クラスタ内のSecret管理状況を把握し、各Secretの種別（Opaque、TLS、docker-registry等）やデータ項目数を確認するために本帳票が必要である。Secretの棚卸しやセキュリティ監査において参照される重要な帳票である。

**帳票の利用シーン**：Namespace内のSecret一覧を確認する場合、Secretの種別（Type）を確認する場合、Secretに含まれるデータ項目数を確認する場合、セキュリティ監査でSecret管理状況を確認する場合に利用される。

**主要な出力内容**：
1. Secret名
2. Secretの種別（Type: Opaque、kubernetes.io/tls等）
3. データ項目数（Dataフィールドのキー数）
4. 作成からの経過時間（Age）

**帳票の出力タイミング**：ユーザが `kubectl get secrets` コマンドを実行した際に出力される。

**帳票の利用者**：Kubernetesクラスタ管理者、セキュリティエンジニア、アプリケーション開発者

## 帳票種別

一覧表（リソース一覧出力）

## 利用画面

| 画面No | 画面名 | URL/ルーティング | 出力操作 |
|--------|--------|-----------------|---------|
| - | CLIターミナル | - | `kubectl get secrets` コマンド実行 |

## 出力形式

### 基本仕様

| 項目 | 内容 |
|-----|------|
| ファイル形式 | テキスト（標準出力） |
| 用紙サイズ | N/A（ターミナル出力） |
| 向き | N/A |
| ファイル名 | N/A（標準出力）|
| 出力方法 | 標準出力（stdout） |
| 文字コード | UTF-8 |

## 帳票レイアウト

### レイアウト概要

タブ区切りのテーブル形式で出力する。

```
NAME                  TYPE                                  DATA   AGE
default-token-abc     kubernetes.io/service-account-token   3      30d
my-secret             Opaque                                2      5d
tls-cert              kubernetes.io/tls                     2      10d
```

### ヘッダー部

| No | 項目名 | 説明 | データ取得元 | 表示形式 |
|----|-------|------|-------------|---------|
| 1 | NAME | Secret名 | `obj.Name` | 文字列 |
| 2 | TYPE | Secretの種別 | `obj.Type` | 文字列（SecretType型をstring変換） |
| 3 | DATA | データ項目数 | `len(obj.Data)` | 整数 |
| 4 | AGE | 作成からの経過時間 | `obj.CreationTimestamp` | 相対時間（例: 30d, 5h） |

### 明細部

ヘッダーと同一カラムで各Secretリソースの情報を1行ずつ出力する。

### フッター部

フッター部は存在しない。

## 出力条件

### 抽出条件

| 条件名 | 説明 | 必須 |
|-------|------|-----|
| リソースタイプ | Secretリソースを対象とする | Yes |
| Namespace | 指定されたNamespace内のリソースを対象とする | No（デフォルトは現在のNamespace） |
| ラベルセレクタ | `-l` オプションで指定されたラベルによるフィルタリング | No |
| フィールドセレクタ | `--field-selector` で種別フィルタ可能（例: `type=Opaque`） | No |

### ソート順

| 優先度 | 項目 | 昇順/降順 |
|-------|------|---------|
| 1 | APIサーバからの返却順（デフォルト） | 昇順 |

### 改ページ条件

改ページは発生しない（ターミナル出力のため）。

## データベース参照仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 結合条件 |
|-----------|------|---------|
| etcd: /registry/secrets/{namespace}/{name} | Secretリソースの保存先 | Namespace + リソース名で参照 |

### テーブル別参照項目詳細

#### Secret リソース

| 参照項目（フィールド名） | 帳票項目との対応 | 取得条件 | 備考 |
|-------------------|----------------|---------|------|
| `metadata.name` | NAME | 全件取得 | そのまま文字列出力 |
| `metadata.creationTimestamp` | AGE | 全件取得 | 相対時間に変換 |
| `type` | TYPE | 全件取得 | SecretType型をstringにキャスト |
| `data` | DATA | 全件取得 | mapのキー数をlen()で取得 |

## 計算仕様

### 計算項目一覧

| 項目名 | 計算式 | 端数処理 | 備考 |
|-------|-------|---------|------|
| AGE | 現在時刻 - CreationTimestamp | N/A | `translateTimestampSince`関数で変換 |
| DATA | int64(len(obj.Data)) | N/A | Dataマップのエントリ数 |

## 処理フロー

### 出力フロー

```mermaid
flowchart TD
    A[kubectl get secrets 実行] --> B[コマンドパラメータ解析]
    B --> C[API Server へ LIST リクエスト]
    C --> D[Secret リソース一覧取得]
    D --> E[printSecret 関数で各行生成]
    E --> F[HumanReadablePrinter でテーブル整形]
    F --> G[標準出力へ出力]
```

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 表示メッセージ | 対処方法 |
|----------|---------|--------------|---------|
| リソース未存在 | 指定した名前のSecretが存在しない | `Error from server (NotFound): secrets "<名前>" not found` | 正しいリソース名を指定 |
| 権限不足 | Secretリソースへのアクセス権限がない | `Error from server (Forbidden): ...` | RBACで適切な権限を付与 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定データ件数 | 数件〜数百件 |
| 目標出力時間 | 1秒以内 |
| 同時出力数上限 | N/A（CLIコマンド） |

## セキュリティ考慮事項

Secretリソースの一覧表示ではSecret名と種別のみが表示され、実際の機密データ（data/stringDataフィールドの値）は表示されない。DATAカラムには項目数のみが表示される。ただし、Secret名からその用途が推測される可能性がある。Secretリソースの閲覧にはRBACによる `get` / `list` 権限が必要であり、通常は制限されるべきである。SecretはNamespaceスコープリソースである。

## 備考

- 一覧表示ではSecretの実際のデータ内容は表示されない（DATA列は項目数のみ）
- `-o wide` オプションを指定しても追加カラムは表示されない
- Secret名からServiceAccount用トークンなどの種別が推測できる
- SecretTypeの代表的な値: Opaque, kubernetes.io/service-account-token, kubernetes.io/tls, kubernetes.io/dockercfg

---

## コードリーディングガイド

本帳票を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

SecretリソースのAPI型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.go | `staging/src/k8s.io/api/core/v1/types.go` | Secret, SecretType の定義 |

**読解のコツ**: SecretのDataフィールドはmap[string][]byte型。TypeはSecretType型（文字列エイリアス）。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | printers.go | `pkg/printers/internalversion/printers.go` | AddHandlers関数内のsecretColumnDefinitions（行289-296） |

**主要処理フロー**:
1. **行289-294**: secretColumnDefinitionsでカラム定義（Name, Type, Data, Age）
2. **行295-296**: TableHandlerにprintSecretとprintSecretListを登録

#### Step 3: 印刷関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | printers.go | `pkg/printers/internalversion/printers.go` | printSecret関数（行1912-1918） |

**主要処理フロー**:
- **行1916**: `obj.Name`, `string(obj.Type)`, `int64(len(obj.Data))`, `translateTimestampSince(obj.CreationTimestamp)` の4値をCellsに追加

### プログラム呼び出し階層図

```
kubectl get secrets
    |
    +-- NewCmdGet (pkg/cmd/get/get.go)
    |      +-- RunGet
    |             +-- API Server LIST /api/v1/namespaces/{ns}/secrets
    |
    +-- AddHandlers (pkg/printers/internalversion/printers.go:93)
    |      +-- TableHandler(secretColumnDefinitions, printSecret) (行295)
    |
    +-- printSecret (pkg/printers/internalversion/printers.go:1912)
    |      +-- translateTimestampSince
    |
    +-- HumanReadablePrinter.PrintObj (cli-runtime/pkg/printers/tableprinter.go)
           +-- tabwriter による整形出力
```

### データフロー図

```
[入力]                        [処理]                           [出力]

API Server               printSecret()                    標準出力(stdout)
(Secret                  行1912-1918                      テーブル形式
 リソース一覧)   -------> - Type文字列変換           -------> NAME | TYPE | DATA | AGE
                         - Data項目数カウント
                         - Age計算
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| printers.go | `pkg/printers/internalversion/printers.go` | ソース | カラム定義とprintSecret関数 |
| tableprinter.go | `staging/src/k8s.io/cli-runtime/pkg/printers/tableprinter.go` | ソース | HumanReadablePrinterによるテーブル出力エンジン |
| types.go | `staging/src/k8s.io/api/core/v1/types.go` | ソース | Secret API型定義 |
| printers_test.go | `pkg/printers/internalversion/printers_test.go` | テスト | printSecretのテストケース |
