# 帳票設計書 26-RoleBinding一覧

## 概要

本ドキュメントは、Kubernetes RoleBindingリソースの一覧をテーブル形式で出力する帳票の設計書である。`kubectl get rolebindings`コマンドで使用される。

### 本帳票の処理概要

RoleBindingリソースの一覧情報を、人間が読みやすいテーブル形式で標準出力に出力する帳票である。

**業務上の目的・背景**：RoleBindingはRBAC（Role-Based Access Control）の構成要素であり、特定のNamespace内でRoleまたはClusterRoleをユーザー、グループ、ServiceAccountに紐付ける。本帳票により、Namespace内のアクセス制御設定を一覧で確認でき、権限管理やセキュリティ監査に活用される。

**帳票の利用シーン**：RBAC設定の確認時、セキュリティ監査時、権限付与の確認時、アクセス制御のトラブルシューティング時に利用される。

**主要な出力内容**：
1. RoleBinding名（Name）
2. 参照先Role（Role）
3. 経過時間（Age）
4. 紐付けユーザー一覧（Users、Wideモード）
5. 紐付けグループ一覧（Groups、Wideモード）
6. 紐付けServiceAccount一覧（ServiceAccounts、Wideモード）

**帳票の出力タイミング**：ユーザーが`kubectl get rolebindings`コマンドを実行した際にリアルタイムで出力される。

**帳票の利用者**：セキュリティエンジニア、クラスタ管理者、SRE

## 帳票種別

一覧表（テーブル形式標準出力）

## 利用画面

| 画面No | 画面名 | URL/ルーティング | 出力操作 |
|--------|--------|-----------------|---------|
| - | CLIターミナル | - | `kubectl get rolebindings` |

## 出力形式

### 基本仕様

| 項目 | 内容 |
|-----|------|
| ファイル形式 | テキスト（標準出力） |
| 用紙サイズ | N/A（ターミナル出力） |
| 向き | N/A |
| ファイル名 | N/A（標準出力） |
| 出力方法 | 標準出力（stdout） |
| 文字コード | UTF-8 |

## 帳票レイアウト

### レイアウト概要

テーブル形式でヘッダー行に続いてリソースごとに1行ずつ出力される。`-o wide`オプションで追加カラムが表示される。

```
NAME                   ROLE                   AGE
admin-binding          ClusterRole/admin      30d
```

Wideモード時：
```
NAME              ROLE                AGE   USERS          GROUPS         SERVICEACCOUNTS
admin-binding     ClusterRole/admin   30d   user1, user2   group1         ns:sa1
```

### 明細部

| No | 項目名 | 説明 | データ取得元 | 表示形式 | 列幅 |
|----|-------|------|-------------|---------|-----|
| 1 | Name | RoleBinding名 | `obj.Name` | 文字列 | 可変 |
| 2 | Role | 参照先Role | `obj.RoleRef` | "{Kind}/{Name}" | 可変 |
| 3 | Age | 作成からの経過時間 | `obj.CreationTimestamp` | 人間可読形式 | 可変 |
| 4 | Users | 紐付けユーザー（Wide） | `obj.Subjects` | カンマ区切り | 可変 |
| 5 | Groups | 紐付けグループ（Wide） | `obj.Subjects` | カンマ区切り | 可変 |
| 6 | ServiceAccounts | 紐付けSA（Wide） | `obj.Subjects` | カンマ区切り | 可変 |

## 出力条件

### 抽出条件

| 条件名 | 説明 | 必須 |
|-------|------|-----|
| Namespace | 対象Namespace（デフォルト: default） | No |
| LabelSelector | ラベルセレクタによるフィルタリング | No |
| FieldSelector | フィールドセレクタによるフィルタリング | No |

### ソート順

| 優先度 | 項目 | 昇順/降順 |
|-------|------|---------|
| 1 | Name | 昇順（APIサーバーのデフォルト） |

### 改ページ条件

該当なし（ターミナル出力のため改ページなし）

## データベース参照仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 結合条件 |
|-----------|------|---------|
| RoleBinding（API Resource） | RoleBinding情報の取得 | APIサーバーの `/apis/rbac.authorization.k8s.io/v1/rolebindings` エンドポイント |

### テーブル別参照項目詳細

#### RoleBinding

| 参照項目（カラム名） | 帳票項目との対応 | 取得条件 | 備考 |
|-------------------|----------------|---------|------|
| `metadata.name` | Name | - | - |
| `roleRef.kind` | Role（Kind部分） | - | "Role"または"ClusterRole" |
| `roleRef.name` | Role（Name部分） | - | - |
| `subjects` | Users, Groups, ServiceAccounts | Wide時のみ | `rbac.SubjectsStrings`で分類・文字列化 |
| `metadata.creationTimestamp` | Age | - | `translateTimestampSince`で変換 |

## 計算仕様

### 計算項目一覧

| 項目名 | 計算式 | 端数処理 | 備考 |
|-------|-------|---------|------|
| Role | `fmt.Sprintf("%s/%s", obj.RoleRef.Kind, obj.RoleRef.Name)` | - | L2232 |
| Users/Groups/ServiceAccounts | `rbac.SubjectsStrings(obj.Subjects)` | - | Wide時のみ。SubjectsをUser/Group/ServiceAccountに分類（L2235） |
| Age | `translateTimestampSince(obj.CreationTimestamp)` | - | 人間可読の経過時間文字列 |

## 処理フロー

### 出力フロー

```mermaid
flowchart TD
    A[kubectl get rolebindings 実行] --> B[APIサーバーへリクエスト]
    B --> C[RoleBindingList取得]
    C --> D[printRoleBindingList呼出]
    D --> E[各RoleBindingに対してprintRoleBinding呼出]
    E --> F[RoleRefからRole文字列生成]
    F --> G{Wideモード?}
    G -->|Yes| H[SubjectsStringsでUsers/Groups/SAs分離]
    G -->|No| I[TableRow生成]
    H --> I
    I --> J[テーブル形式で標準出力]
```

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 表示メッセージ | 対処方法 |
|----------|---------|--------------|---------|
| データなし | RoleBindingリソースが存在しない | `No resources found in {namespace} namespace` | Namespaceを確認 |
| API接続エラー | APIサーバーに接続できない | 接続エラーメッセージ | APIサーバーの稼働状態を確認 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定データ件数 | 数十件〜数百件 |
| 目標出力時間 | 数秒以内 |
| 同時出力数上限 | 制限なし |

## セキュリティ考慮事項

- RoleBindingの閲覧にはNamespaceレベルまたはクラスタレベルのRBAC権限（`get`、`list`）が必要
- RoleBindingの一覧はクラスタのアクセス制御設計を示す重要なセキュリティ情報であり、閲覧権限の付与には十分な注意が必要
- Wideモードではユーザー名やServiceAccount名が表示される

## 備考

- RoleBindingはrbac.authorization.k8s.io/v1 APIグループに属する
- RoleRefのKindは"Role"（Namespace内）または"ClusterRole"（クラスタ全体）のいずれか
- `-o wide`オプションでUsers, Groups, ServiceAccountsカラムが追加表示される

---

## コードリーディングガイド

本帳票を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.go | `pkg/apis/rbac/types.go` | RoleBinding, RoleRef, Subject構造体の定義 |

**読解のコツ**: `Subjects`は`[]Subject`型で、各SubjectのKindが"User"/"Group"/"ServiceAccount"のいずれかを示す。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | printers.go | `pkg/printers/internalversion/printers.go` | RoleBinding用テーブルハンドラ登録（L385-394） |

**主要処理フロー**:
1. **L385-391**: `roleBindingsColumnDefinitions`で6カラムを定義（うち3つはPriority:1でWide時のみ）
2. **L393-394**: ハンドラ登録

#### Step 3: 出力処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | printers.go | `pkg/printers/internalversion/printers.go` | `printRoleBinding`関数（L2227-2239） |

**主要処理フロー**:
- **L2232**: RoleRefを"{Kind}/{Name}"形式で文字列化
- **L2233**: Name, roleRef, AgeをCellsに追加
- **L2234-2237**: Wideモード時に`rbac.SubjectsStrings`でUsers/Groups/ServiceAccountsを分離して追加

### プログラム呼び出し階層図

```
kubectl get rolebindings
    │
    ├─ get.go: RunGet()
    │      └─ APIサーバーへREST GET /apis/rbac.authorization.k8s.io/v1/namespaces/{ns}/rolebindings
    │
    ├─ printers.go: AddHandlers() [初期化時]
    │      └─ TableHandler登録 (roleBindingsColumnDefinitions)
    │
    └─ printers.go: printRoleBindingList()
           └─ printRoleBinding() [各アイテム]
                  ├─ RoleRef文字列化
                  └─ rbac.SubjectsStrings() [Wide時]
```

### データフロー図

```
[入力]                             [処理]                           [出力]

APIサーバー ───▶ RoleBindingList取得 ───▶ 標準出力(テーブル)
  /apis/rbac../v1/       printRoleBindingList()        NAME  ROLE  AGE
  rolebindings               └─ printRoleBinding()
                                  ├─ RoleRef整形
                                  └─ SubjectsStrings() [Wide]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| printers.go | `pkg/printers/internalversion/printers.go` | ソース | RoleBindingのテーブル出力ハンドラ定義・行生成ロジック |
| printers_test.go | `pkg/printers/internalversion/printers_test.go` | テスト | RoleBinding出力のテストケース（TestPrintRoleBinding L4522） |
| types.go | `pkg/apis/rbac/types.go` | ソース | RoleBinding内部API型定義 |
| helpers.go | `pkg/apis/rbac/helpers.go` | ソース | SubjectsStringsヘルパー関数 |
| tableprinter.go | `staging/src/k8s.io/cli-runtime/pkg/printers/tableprinter.go` | ソース | テーブル形式出力の共通基盤 |
| get.go | `staging/src/k8s.io/kubectl/pkg/cmd/get/get.go` | ソース | kubectl getコマンドのエントリーポイント |
