# 帳票設計書 27-ClusterRoleBinding一覧

## 概要

本ドキュメントは、Kubernetes ClusterRoleBindingリソースの一覧をテーブル形式で出力する帳票の設計書である。`kubectl get clusterrolebindings`コマンドで使用される。

### 本帳票の処理概要

ClusterRoleBindingリソースの一覧情報を、人間が読みやすいテーブル形式で標準出力に出力する帳票である。

**業務上の目的・背景**：ClusterRoleBindingはRBAC（Role-Based Access Control）の構成要素であり、クラスタ全体のスコープでClusterRoleをユーザー、グループ、ServiceAccountに紐付ける。本帳票により、クラスタ全体のアクセス制御設定を一覧で確認でき、権限管理やセキュリティ監査に活用される。RoleBindingとは異なりNamespaceに限定されないため、クラスタ全体の権限を横断的に確認できる。

**帳票の利用シーン**：クラスタ全体のRBAC設定確認時、セキュリティ監査時、権限の棚卸し時、過剰権限の検出時に利用される。

**主要な出力内容**：
1. ClusterRoleBinding名（Name）
2. 参照先ClusterRole（Role）
3. 経過時間（Age）
4. 紐付けユーザー一覧（Users、Wideモード）
5. 紐付けグループ一覧（Groups、Wideモード）
6. 紐付けServiceAccount一覧（ServiceAccounts、Wideモード）

**帳票の出力タイミング**：ユーザーが`kubectl get clusterrolebindings`コマンドを実行した際にリアルタイムで出力される。

**帳票の利用者**：セキュリティエンジニア、クラスタ管理者、SRE、監査担当者

## 帳票種別

一覧表（テーブル形式標準出力）

## 利用画面

| 画面No | 画面名 | URL/ルーティング | 出力操作 |
|--------|--------|-----------------|---------|
| - | CLIターミナル | - | `kubectl get clusterrolebindings` |

## 出力形式

### 基本仕様

| 項目 | 内容 |
|-----|------|
| ファイル形式 | テキスト（標準出力） |
| 用紙サイズ | N/A（ターミナル出力） |
| 向き | N/A |
| ファイル名 | N/A（標準出力） |
| 出力方法 | 標準出力（stdout） |
| 文字コード | UTF-8 |

## 帳票レイアウト

### レイアウト概要

テーブル形式でヘッダー行に続いてリソースごとに1行ずつ出力される。`-o wide`オプションで追加カラムが表示される。

```
NAME                                         ROLE                               AGE
cluster-admin                                ClusterRole/cluster-admin           365d
system:node                                  ClusterRole/system:node             365d
```

Wideモード時：
```
NAME              ROLE                        AGE    USERS          GROUPS                    SERVICEACCOUNTS
cluster-admin     ClusterRole/cluster-admin    365d                  system:masters
```

### 明細部

| No | 項目名 | 説明 | データ取得元 | 表示形式 | 列幅 |
|----|-------|------|-------------|---------|-----|
| 1 | Name | ClusterRoleBinding名 | `obj.Name` | 文字列 | 可変 |
| 2 | Role | 参照先ClusterRole | `obj.RoleRef` | "{Kind}/{Name}" | 可変 |
| 3 | Age | 作成からの経過時間 | `obj.CreationTimestamp` | 人間可読形式 | 可変 |
| 4 | Users | 紐付けユーザー（Wide） | `obj.Subjects` | カンマ区切り | 可変 |
| 5 | Groups | 紐付けグループ（Wide） | `obj.Subjects` | カンマ区切り | 可変 |
| 6 | ServiceAccounts | 紐付けSA（Wide） | `obj.Subjects` | カンマ区切り | 可変 |

## 出力条件

### 抽出条件

| 条件名 | 説明 | 必須 |
|-------|------|-----|
| Namespace | ClusterRoleBindingはクラスタスコープのため不要 | No |
| LabelSelector | ラベルセレクタによるフィルタリング | No |
| FieldSelector | フィールドセレクタによるフィルタリング | No |

### ソート順

| 優先度 | 項目 | 昇順/降順 |
|-------|------|---------|
| 1 | Name | 昇順（APIサーバーのデフォルト） |

### 改ページ条件

該当なし（ターミナル出力のため改ページなし）

## データベース参照仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 結合条件 |
|-----------|------|---------|
| ClusterRoleBinding（API Resource） | ClusterRoleBinding情報の取得 | APIサーバーの `/apis/rbac.authorization.k8s.io/v1/clusterrolebindings` エンドポイント |

### テーブル別参照項目詳細

#### ClusterRoleBinding

| 参照項目（カラム名） | 帳票項目との対応 | 取得条件 | 備考 |
|-------------------|----------------|---------|------|
| `metadata.name` | Name | - | - |
| `roleRef.kind` | Role（Kind部分） | - | 通常は"ClusterRole" |
| `roleRef.name` | Role（Name部分） | - | - |
| `subjects` | Users, Groups, ServiceAccounts | Wide時のみ | `rbac.SubjectsStrings`で分類・文字列化 |
| `metadata.creationTimestamp` | Age | - | `translateTimestampSince`で変換 |

## 計算仕様

### 計算項目一覧

| 項目名 | 計算式 | 端数処理 | 備考 |
|-------|-------|---------|------|
| Role | `fmt.Sprintf("%s/%s", obj.RoleRef.Kind, obj.RoleRef.Name)` | - | L2259 |
| Users/Groups/ServiceAccounts | `rbac.SubjectsStrings(obj.Subjects)` | - | Wide時のみ。SubjectsをUser/Group/ServiceAccountに分類（L2262） |
| Age | `translateTimestampSince(obj.CreationTimestamp)` | - | 人間可読の経過時間文字列 |

## 処理フロー

### 出力フロー

```mermaid
flowchart TD
    A[kubectl get clusterrolebindings 実行] --> B[APIサーバーへリクエスト]
    B --> C[ClusterRoleBindingList取得]
    C --> D[printClusterRoleBindingList呼出]
    D --> E[各ClusterRoleBindingに対してprintClusterRoleBinding呼出]
    E --> F[RoleRefからRole文字列生成]
    F --> G{Wideモード?}
    G -->|Yes| H[SubjectsStringsでUsers/Groups/SAs分離]
    G -->|No| I[TableRow生成]
    H --> I
    I --> J[テーブル形式で標準出力]
```

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 表示メッセージ | 対処方法 |
|----------|---------|--------------|---------|
| データなし | ClusterRoleBindingリソースが存在しない | `No resources found` | クラスタ設定を確認 |
| 権限不足 | RBAC権限がない | `Forbidden` | 適切なRBAC権限を付与 |
| API接続エラー | APIサーバーに接続できない | 接続エラーメッセージ | APIサーバーの稼働状態を確認 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定データ件数 | 数十件〜数百件 |
| 目標出力時間 | 数秒以内 |
| 同時出力数上限 | 制限なし |

## セキュリティ考慮事項

- ClusterRoleBindingの閲覧にはクラスタレベルのRBAC権限（`get`、`list`）が必要
- ClusterRoleBindingの一覧はクラスタ全体のアクセス制御設計を示す最重要セキュリティ情報であり、閲覧権限の付与には最大限の注意が必要
- Wideモードではユーザー名、グループ名、ServiceAccount名が表示される

## 備考

- ClusterRoleBindingはrbac.authorization.k8s.io/v1 APIグループに属する
- クラスタスコープリソースのため、Namespace指定は不要
- RoleBinding（No.26）と同一の出力ロジック構造を持つが、スコープが異なる
- `-o wide`オプションでUsers, Groups, ServiceAccountsカラムが追加表示される

---

## コードリーディングガイド

本帳票を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.go | `pkg/apis/rbac/types.go` | ClusterRoleBinding, RoleRef, Subject構造体の定義 |

**読解のコツ**: ClusterRoleBindingはRoleBindingと同じSubjects/RoleRefフィールドを持つが、クラスタスコープである点が異なる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | printers.go | `pkg/printers/internalversion/printers.go` | ClusterRoleBinding用テーブルハンドラ登録（L396-405） |

**主要処理フロー**:
1. **L396-402**: `clusterRoleBindingsColumnDefinitions`で6カラムを定義（うち3つはPriority:1でWide時のみ）
2. **L404-405**: ハンドラ登録

#### Step 3: 出力処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | printers.go | `pkg/printers/internalversion/printers.go` | `printClusterRoleBinding`関数（L2254-2266） |

**主要処理フロー**:
- **L2259**: RoleRefを"{Kind}/{Name}"形式で文字列化
- **L2260**: Name, roleRef, AgeをCellsに追加
- **L2261-2264**: Wideモード時に`rbac.SubjectsStrings`でUsers/Groups/ServiceAccountsを分離して追加

### プログラム呼び出し階層図

```
kubectl get clusterrolebindings
    │
    ├─ get.go: RunGet()
    │      └─ APIサーバーへREST GET /apis/rbac.authorization.k8s.io/v1/clusterrolebindings
    │
    ├─ printers.go: AddHandlers() [初期化時]
    │      └─ TableHandler登録 (clusterRoleBindingsColumnDefinitions)
    │
    └─ printers.go: printClusterRoleBindingList()
           └─ printClusterRoleBinding() [各アイテム]
                  ├─ RoleRef文字列化
                  └─ rbac.SubjectsStrings() [Wide時]
```

### データフロー図

```
[入力]                                [処理]                              [出力]

APIサーバー ───▶ ClusterRoleBindingList取得 ───▶ 標準出力(テーブル)
  /apis/rbac../v1/         printClusterRoleBindingList()    NAME  ROLE  AGE
  clusterrolebindings          └─ printClusterRoleBinding()
                                    ├─ RoleRef整形
                                    └─ SubjectsStrings() [Wide]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| printers.go | `pkg/printers/internalversion/printers.go` | ソース | ClusterRoleBindingのテーブル出力ハンドラ定義・行生成ロジック |
| printers_test.go | `pkg/printers/internalversion/printers_test.go` | テスト | ClusterRoleBinding出力のテストケース（TestPrintClusterRoleBinding L4597） |
| types.go | `pkg/apis/rbac/types.go` | ソース | ClusterRoleBinding内部API型定義 |
| helpers.go | `pkg/apis/rbac/helpers.go` | ソース | SubjectsStringsヘルパー関数 |
| tableprinter.go | `staging/src/k8s.io/cli-runtime/pkg/printers/tableprinter.go` | ソース | テーブル形式出力の共通基盤 |
| get.go | `staging/src/k8s.io/kubectl/pkg/cmd/get/get.go` | ソース | kubectl getコマンドのエントリーポイント |
