# 機能設計書 136-ResourceQuota管理

## 概要

本ドキュメントは、Kubernetes における ResourceQuota 管理機能の設計を記述する。Namespace ごとのリソース使用量制限の設定と管理を行う機能である。

### 本機能の処理概要

**業務上の目的・背景**：マルチテナント環境において、各 Namespace のリソース消費量を制限し、クラスター全体のリソースの公平な配分を保証する。CPU、メモリ、ストレージ、オブジェクト数等の上限を設定することで、一つの Namespace がクラスター全体のリソースを独占することを防止する。

**機能の利用シーン**：複数のチームやプロジェクトが同一の Kubernetes クラスターを共有する場合に、各 Namespace に対してリソースの上限を設定して利用する。

**主要な処理内容**：
1. ResourceQuota オブジェクトの CRUD 操作（REST API）
2. 作成・更新時のバリデーション（リソース名の妥当性、requests <= limits の警告等）
3. Status と Spec の分離管理（Status は ResourceQuota Controller が更新）
4. requests > limits の場合の警告メッセージ生成

**関連システム・外部連携**：kube-apiserver（REST API）、etcd（永続化）、ResourceQuota Controller（使用量計算・Status更新）、Admission Controller（クォータ超過の拒否）

**権限による制御**：RBAC により ResourceQuota オブジェクトの作成・更新・削除・参照が制御される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | kubectl や API を通じて操作 |

## 機能種別

リソース管理（API リソースの CRUD）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| metadata.name | string | Yes | ResourceQuota 名 | DNS サブドメイン準拠 |
| metadata.namespace | string | Yes | 対象 Namespace | 有効な Namespace 名 |
| spec.hard | ResourceList | Yes | リソース上限の定義 | 有効なリソース名と数量 |
| spec.scopes | []ResourceQuotaScope | No | クォータの適用範囲 | 有効なスコープ値 |
| spec.scopeSelector | ScopeSelector | No | スコープセレクタ | 有効なスコープセレクタ |

### 入力データソース

- Kubernetes REST API（kubectl apply, kubectl create）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| spec.hard | ResourceList | 設定されたリソース上限 |
| status.hard | ResourceList | 現在適用されているリソース上限 |
| status.used | ResourceList | 現在のリソース使用量 |

### 出力先

- etcd（永続化）
- REST API レスポンス

## 処理フロー

### 処理シーケンス

```
1. Create リクエスト受信
   └─ PrepareForCreate: status フィールドをクリア
2. バリデーション
   └─ ValidateResourceQuota: Spec の妥当性検証
3. 警告生成
   └─ WarningsOnCreate: requests > limits の検出と警告
4. etcd への永続化
   └─ genericregistry.Store を通じて保存
5. Status 更新（ResourceQuota Controller による）
   └─ StatusStrategy: spec を変更せずに status のみ更新
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-136-01 | Namespace スコープ | ResourceQuota は Namespace スコープのリソース | 常時 |
| BR-136-02 | Status 保護 | 作成時に Status はクリアされる。更新時に Status は旧値が保持される | Create/Update 時 |
| BR-136-03 | Spec 保護（Status更新時） | Status 更新時に Spec は旧値が保持される | StatusStrategy.PrepareForUpdate 時 |
| BR-136-04 | requests <= limits 警告 | CPU, Memory, Storage, EphemeralStorage について requests > limits の場合に警告 | Create 時 |
| BR-136-05 | AllowCreateOnUpdate | false - Update で新規作成は不可 | Update 時 |
| BR-136-06 | AllowUnconditionalUpdate | true - 無条件更新を許可 | Update 時 |

### 計算ロジック

- 警告判定: `requests.{resource}` と `limits.{resource}` を比較。CPU/Memory は bare name（`cpu`, `memory`）での指定もフォールバックとして対応

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象リソース | 操作種別 | 概要 |
|-----|-------------|---------|------|
| Create | ResourceQuota | INSERT | 新規 ResourceQuota の作成 |
| Update | ResourceQuota | UPDATE | Spec の更新（Status は保持） |
| Status Update | ResourceQuota | UPDATE | Status の更新（Spec は保持） |
| Delete | ResourceQuota | DELETE | ResourceQuota の削除 |
| Get/List | ResourceQuota | SELECT | ResourceQuota の取得 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | BadRequest | バリデーションエラー | エラーメッセージと共に拒否 |
| 409 | Conflict | 同時更新の競合 | クライアントにリトライを促す |
| 422 | UnprocessableEntity | 不正なリソース名や数量 | エラーメッセージと共に拒否 |

## トランザクション仕様

etcd のトランザクション機構に依存。楽観的ロックによる更新競合検出（ResourceVersion ベース）。

## パフォーマンス要件

- ResourceQuota の CRUD 操作は通常の Kubernetes API オブジェクト操作と同等
- Status 更新は ResourceQuota Controller の sync 間隔に依存

## セキュリティ考慮事項

- RBAC により ResourceQuota の管理権限を制御
- Status のリセットフィールドにより、ユーザーが直接 Status を操作することを防止
- ショートネーム `quota` でアクセス可能

## 備考

- ショートネーム: `quota`
- DeletedObject を返す設定（ReturnDeletedObject: true）
- Status は別のエンドポイント（StatusREST）で管理

---

## コードリーディングガイド

### 推奨読解順序

#### Step 1: Strategy パターンを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | strategy.go | `pkg/registry/core/resourcequota/strategy.go` | resourcequotaStrategy 構造体（33-36行目）: ObjectTyper と NameGenerator を埋め込み |
| 1-2 | strategy.go | `pkg/registry/core/resourcequota/strategy.go` | GetResetFields()（49-57行目）: v1 API で status フィールドをリセット対象に指定 |
| 1-3 | strategy.go | `pkg/registry/core/resourcequota/strategy.go` | PrepareForCreate()（60-63行目）: 作成時に Status をクリア |
| 1-4 | strategy.go | `pkg/registry/core/resourcequota/strategy.go` | PrepareForUpdate()（66-70行目）: 更新時に Status を旧値で保持 |

#### Step 2: 警告生成ロジック

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | strategy.go | `pkg/registry/core/resourcequota/strategy.go` | knownResourceNames（79-84行目）: 警告対象のリソース名定義 |
| 2-2 | strategy.go | `pkg/registry/core/resourcequota/strategy.go` | WarningsOnCreate()（87-108行目）: requests > limits の警告生成。CPU/Memory は bare name もフォールバック |

#### Step 3: Storage レイヤー

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | storage.go | `pkg/registry/core/resourcequota/storage/storage.go` | NewREST()（41-66行目）: Store の初期化と StatusREST の生成 |
| 3-2 | storage.go | `pkg/registry/core/resourcequota/storage/storage.go` | ShortNames()（72-74行目）: "quota" ショートネーム |

### プログラム呼び出し階層図

```
REST API リクエスト
    │
    ├─ Create → genericregistry.Store
    │      ├─ Strategy.PrepareForCreate() [Status クリア]
    │      ├─ Strategy.Validate() → validation.ValidateResourceQuota()
    │      └─ Strategy.WarningsOnCreate() [requests > limits 警告]
    │
    ├─ Update → genericregistry.Store
    │      ├─ Strategy.PrepareForUpdate() [Status 保持]
    │      └─ Strategy.ValidateUpdate() → validation.ValidateResourceQuotaUpdate()
    │
    └─ Status Update → StatusREST → statusStore
           ├─ StatusStrategy.PrepareForUpdate() [Spec 保持]
           └─ StatusStrategy.ValidateUpdate() → validation.ValidateResourceQuotaStatusUpdate()
```

### データフロー図

```
[入力]                    [処理]                       [出力]

API Request         ──▶ PrepareForCreate/Update   ──▶ sanitized object
  (ResourceQuota)        (Status/Spec 保護)

sanitized object    ──▶ Validate()                ──▶ field.ErrorList
                         ValidateResourceQuota()

sanitized object    ──▶ WarningsOnCreate()         ──▶ []string (warnings)
                         requests > limits check

validated object    ──▶ etcd store                 ──▶ persisted ResourceQuota
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| strategy.go | `pkg/registry/core/resourcequota/strategy.go` | ソース | Create/Update 戦略定義 |
| storage/storage.go | `pkg/registry/core/resourcequota/storage/storage.go` | ソース | REST ストレージ実装 |
| validation.go | `pkg/apis/core/validation/validation.go` | ソース | バリデーションロジック |
