# 機能設計書 148-CRIクライアント

## 概要

本ドキュメントは、Kubernetes CRI（Container Runtime Interface）クライアントライブラリの設計について記述する。CRIクライアントはKubeletからコンテナランタイムへのgRPC通信を実装し、CRI APIの呼び出しを抽象化する。

### 本機能の処理概要

**業務上の目的・背景**：KubeletがCRI API定義（No.147）に基づいてコンテナランタイムと通信するためのクライアント実装が必要である。CRIクライアントはgRPC接続管理、タイムアウト制御、ログリダクション、OpenTelemetryトレーシングなどの横断的関心事を提供し、Kubeletのランタイム通信コードを簡潔に保つ。

**機能の利用シーン**：KubeletがPod/コンテナのライフサイクル操作、イメージ管理、統計情報取得などを行う際に使用される。

**主要な処理内容**：
1. gRPC接続管理: コンテナランタイムへのgRPC接続の確立と維持
2. RuntimeService実装: CRI RuntimeServiceの全RPCメソッドのクライアント側実装
3. ImageService実装: CRI ImageServiceの全RPCメソッドのクライアント側実装
4. タイムアウト制御: RPC呼び出しのタイムアウト管理
5. ログリダクション: 同一エラーの繰り返しログ出力の抑制
6. OpenTelemetryトレーシング: gRPC呼び出しの分散トレーシング
7. バージョンネゴシエーション: CRIバージョンの検出と互換性管理

**関連システム・外部連携**：Kubelet、containerd、CRI-O、CRI API（No.147）

**権限による制御**：CRIソケットファイルへのアクセス権限が必要（通常root権限）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | なし | 内部コンポーネント | Kubelet内部で使用されるクライアントライブラリ |

## 機能種別

ライブラリ / gRPCクライアント

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| endpoint | string | Yes | CRIランタイムソケットのアドレス | 有効なUnixソケットパス |
| timeout | time.Duration | Yes | RPCのデフォルトタイムアウト | 正の値 |
| CRIVersion | CRIVersion | No | 使用するCRIバージョン | v1 |

### 入力データソース

Kubeletの設定（--container-runtime-endpoint）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| RPC応答 | 各種Response型 | CRI RPC呼び出しの結果 |
| エラー | error | RPC失敗時のエラー情報 |

### 出力先

呼び出し元のKubeletコード。

## 処理フロー

### 処理シーケンス

```
1. CRIクライアント初期化
   └─ gRPC接続の確立（Unixソケット、バックオフ設定）
2. バージョン確認
   └─ Version RPCでCRIバージョンを確認
3. RPC呼び出し
   └─ コンテキストにタイムアウトを設定
   └─ gRPC呼び出しの実行
   └─ OpenTelemetryスパンの記録
4. エラーハンドリング
   └─ gRPCステータスコードの解釈
   └─ ログリダクション（同一エラーの抑制）
5. レスポンス返却
```

### フローチャート

```mermaid
flowchart TD
    A[Kubelet呼び出し] --> B[コンテキスト+タイムアウト設定]
    B --> C[gRPC RPC呼び出し]
    C --> D{レスポンス}
    D -->|成功| E[結果返却]
    D -->|エラー| F{gRPCステータス}
    F -->|NotFound| G[ErrContainerStatusNil等]
    F -->|DeadlineExceeded| H[ErrCommandTimedOut]
    F -->|その他| I[ログリダクション判定]
    I --> J[エラー返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-148-01 | バックオフ設定 | 最大バックオフ3秒、ベースバックオフ100ms | gRPC接続時 |
| BR-148-02 | 最小接続タイムアウト | 5秒以上の接続タイムアウトが必要 | 接続確立時 |
| BR-148-03 | ログリダクション | 同一エラーは1分間隔で報告 | エラーログ出力時 |
| BR-148-04 | CRIバージョン | CRIVersionV1（"v1"）がサポートされる | 常時 |
| BR-148-05 | Insecure接続 | ローカルUnixソケットのためTLSなし | gRPC接続時 |

### 計算ロジック

該当なし。

## データベース操作仕様

本機能はクライアントライブラリであり、データベース操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ErrContainerStatusNil | ContainerStatusの応答がnil | コンテナの存在を確認 |
| - | ErrCommandTimedOut | ExecSyncコマンドがタイムアウト | タイムアウト値を調整 |
| - | gRPC Unavailable | ランタイムに接続不可 | ランタイムの状態を確認 |

### リトライ仕様

gRPC接続はバックオフ付きの自動再接続を行う。個別RPC呼び出しのリトライはKubelet側で管理。

## トランザクション仕様

該当なし（各RPC呼び出しは独立）。

## パフォーマンス要件

Unixソケット通信により低レイテンシを実現。タイムアウト設定により長時間ブロックを防止。

## セキュリティ考慮事項

- Unixソケットのパーミッションにより通信を保護
- gRPC Insecure接続はローカル通信のみを前提としている
- OpenTelemetryトレーシングによりRPC呼び出しの監査が可能

## 備考

- CRIクライアントは`staging/src/k8s.io/cri-client`に格納
- remoteRuntimeServiceとremoteImageServiceの2つの構造体で実装される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | remote_runtime.go | `staging/src/k8s.io/cri-client/pkg/remote_runtime.go` | remoteRuntimeService構造体とgRPC接続設定 |

**読解のコツ**: remoteRuntimeServiceはruntimeapi.RuntimeServiceClientをラップし、タイムアウト管理やログリダクション機能を追加している。

**主要処理フロー**:
- **47-54行目**: remoteRuntimeService構造体 -- timeout, runtimeClient, logReduction, logger, conn
- **56-64行目**: 定数定義 -- identicalErrorDelay=1分, maxBackoffDelay=3秒, baseBackoffDelay=100ms, minConnectionTimeout=5秒
- **68-76行目**: CRIVersion型と事前定義エラー -- ErrContainerStatusNil, ErrCommandTimedOut
- **78-80行目**: CRIVersionV1定数 = "v1"

#### Step 2: イメージサービスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | remote_image.go | `staging/src/k8s.io/cri-client/pkg/remote_image.go` | remoteImageService構造体 |

#### Step 3: ユーティリティを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | utils.go | `staging/src/k8s.io/cri-client/pkg/utils.go` | 共通ユーティリティ関数 |
| 3-2 | util/ | `staging/src/k8s.io/cri-client/pkg/util/` | 追加ユーティリティ |

### プログラム呼び出し階層図

```
Kubelet
    |
    +-- cri.NewRemoteRuntimeService(endpoint, timeout)
    |       |
    |       +-- grpc.Dial(endpoint, opts...)  -- gRPC接続確立
    |       |       +-- insecure.NewCredentials()
    |       |       +-- otelgrpc (OpenTelemetryインターセプター)
    |       |       +-- backoff設定 (base=100ms, max=3s)
    |       |
    |       +-- remoteRuntimeService
    |               +-- RunPodSandbox() → runtimeClient.RunPodSandbox()
    |               +-- CreateContainer() → runtimeClient.CreateContainer()
    |               +-- ExecSync() → runtimeClient.ExecSync()
    |               +-- ... (全RuntimeService RPC)
    |
    +-- cri.NewRemoteImageService(endpoint, timeout)
            +-- remoteImageService
                    +-- PullImage() → imageClient.PullImage()
                    +-- ListImages() → imageClient.ListImages()
                    +-- ... (全ImageService RPC)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Kubelet呼び出し ──> remoteRuntimeService ──> gRPC call ──> Runtime Response
                   +- タイムアウト設定                      |
                   +- OTelトレーシング                      v
                   +- ログリダクション        ──> Kubelet（結果/エラー）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| remote_runtime.go | `staging/src/k8s.io/cri-client/pkg/remote_runtime.go` | ソース | RuntimeServiceクライアント実装 |
| remote_image.go | `staging/src/k8s.io/cri-client/pkg/remote_image.go` | ソース | ImageServiceクライアント実装 |
| utils.go | `staging/src/k8s.io/cri-client/pkg/utils.go` | ソース | 共通ユーティリティ |
| util/ | `staging/src/k8s.io/cri-client/pkg/util/` | ソース | 追加ユーティリティ |
| internal/ | `staging/src/k8s.io/cri-client/pkg/internal/` | ソース | 内部実装 |
| fake/ | `staging/src/k8s.io/cri-client/pkg/fake/` | テスト | テスト用フェイク実装 |
| remote_runtime_test.go | `staging/src/k8s.io/cri-client/pkg/remote_runtime_test.go` | テスト | RuntimeServiceテスト |
| remote_image_test.go | `staging/src/k8s.io/cri-client/pkg/remote_image_test.go` | テスト | ImageServiceテスト |
