# 機能設計書 2-認証（Authentication）

## 概要

本ドキュメントは、Kubernetes API Serverにおける認証（Authentication）機能の設計書である。クライアント証明書、トークン、OIDC等の複数方式によるリクエスト送信者の認証を行う。

### 本機能の処理概要

認証機能は、API Serverに到達した全てのHTTPリクエストに対して、リクエスト送信者のアイデンティティを特定する。複数の認証方式をチェーンとして構成し、いずれかの方式で認証が成功すればユーザー情報を返却する。

**業務上の目的・背景**：Kubernetesクラスターへのアクセスを安全に管理するために、全てのAPIリクエストの送信者を正確に特定する必要がある。認証はセキュリティの最前線であり、認可やアドミッション制御の前提条件となる。不正アクセスの防止、監査ログにおけるアクション主体の追跡、マルチテナント環境でのユーザー分離など、クラスターセキュリティの根幹を担う。

**機能の利用シーン**：API Serverへの全てのHTTPリクエスト受信時に自動的に実行される。kubectlコマンド実行時のkubeconfigに記載された認証情報の検証、サービスアカウントトークンを使用したPod内からのAPI呼び出し、外部IDプロバイダー（OIDC）を介した認証、フロントプロキシ経由でのリクエスト認証など。

**主要な処理内容**：
1. x509クライアント証明書による認証（TLSハンドシェイク時の証明書検証）
2. Bearer Tokenによる認証（ServiceAccountトークン、静的トークン）
3. OpenID Connect（OIDC）トークンによる認証
4. Webhook Token認証（外部認証サービスへの委譲）
5. リクエストヘッダー認証（フロントプロキシ用）
6. 匿名認証（上記全てで認証できなかった場合のフォールバック）
7. 認証結果のキャッシュ

**関連システム・外部連携**：外部OIDCプロバイダー、Webhook認証サービス、フロントプロキシサーバー、証明書認証局（CA）

**権限による制御**：認証機能自体には権限による制御はないが、認証結果として特定されたユーザー/グループ情報が後続の認可処理で使用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 26 | kubectl auth | API連携 | 現在の認証情報を使用してAPI Serverに接続する |
| 48 | kubeadm token | API連携 | トークンベースの認証情報を管理する |

## 機能種別

認証処理 / セキュリティ基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Authorization Header | string | No | Bearer <token> 形式のトークン | トークン形式の妥当性 |
| Client Certificate | x509.Certificate | No | TLS接続時のクライアント証明書 | CA署名の検証、有効期限 |
| X-Remote-User | string | No | フロントプロキシからのユーザー名ヘッダー | 信頼されたプロキシからのリクエストであること |
| X-Remote-Group | string | No | フロントプロキシからのグループヘッダー | 信頼されたプロキシからのリクエストであること |
| X-Remote-Extra-* | string | No | フロントプロキシからの追加情報ヘッダー | 信頼されたプロキシからのリクエストであること |

### 入力データソース

HTTPリクエストヘッダー、TLSハンドシェイク情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| User.Name | string | 認証されたユーザー名 |
| User.UID | string | ユーザーの一意識別子 |
| User.Groups | []string | ユーザーが所属するグループ一覧 |
| User.Extra | map[string][]string | 追加のユーザー情報 |
| Audiences | []string | トークンが対象とするオーディエンス |
| Authenticated | bool | 認証成功/失敗フラグ |

### 出力先

API Serverのリクエストコンテキスト（後続の認可・アドミッション制御で使用）

## 処理フロー

### 処理シーケンス

```
1. HTTPリクエスト受信
   └─ TLSハンドシェイク完了後のリクエスト
2. 認証チェーンの実行
   └─ 設定された順序で各認証方式を試行
3. フロントプロキシ認証（RequestHeader）
   └─ 信頼されたCA署名のクライアント証明書を持つプロキシからのヘッダーを検証
4. x509クライアント証明書認証
   └─ クライアント証明書のCN（CommonName）をユーザー名、O（Organization）をグループとして抽出
5. Bearer Token認証
   └─ トークンキャッシュを確認後、各トークン認証プロバイダーに委譲
6. 匿名認証（フォールバック）
   └─ 全方式で認証できなかった場合、system:anonymousユーザーとして処理
7. 認証結果のコンテキスト設定
   └─ 認証されたユーザー情報をリクエストコンテキストに格納
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[認証チェーン開始]
    B --> C{フロントプロキシ認証}
    C -->|成功| H[ユーザー情報設定]
    C -->|不該当| D{x509証明書認証}
    D -->|成功| H
    D -->|不該当| E{Bearer Token認証}
    E -->|成功| H
    E -->|失敗| F{匿名認証有効?}
    F -->|Yes| G[system:anonymous設定]
    F -->|No| Z[401 Unauthorized]
    G --> H
    H --> I[リクエストコンテキストに格納]
    I --> J[認可処理へ]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-1 | 認証チェーン順序 | フロントプロキシ→x509→トークンの順で認証を試行 | 全リクエスト |
| BR-2 | 最初の成功で終了 | チェーン内でいずれかの認証方式が成功した時点で認証完了 | 全リクエスト |
| BR-3 | トークンキャッシュ | 認証結果をTTL付きキャッシュに保存し、同一トークンの再認証を回避 | Bearer Token認証時 |
| BR-4 | ServiceAccountトークン | Namespace/ServiceAccount名からユーザー名を自動生成（system:serviceaccount:<ns>:<name>） | SA Token認証時 |
| BR-5 | 匿名アクセス制御 | AnonymousAuthConfigで匿名アクセスの有効/無効、許可条件を制御 | 全認証方式で失敗時 |

### 計算ロジック

トークンキャッシュのTTLは`DelegatingAuthenticatorConfig.CacheTTL`で設定される。キャッシュミス時はバックエンドの認証プロバイダーに問い合わせを行う。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ServiceAccountトークン検証 | etcd (secrets) | SELECT | ServiceAccountに紐づくトークンシークレットを取得 |
| TokenReview | etcd (tokenreviews) | SELECT | Webhook認証時のTokenReviewリソースを処理 |

### テーブル別操作詳細

#### etcd (ServiceAccount Tokens)

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | /registry/secrets/{namespace}/{name} | ServiceAccountトークンの検証 | バウンドトークン方式の場合はTokenRequestから発行 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 全認証方式で認証失敗（匿名認証無効時） | 有効な認証情報を提供 |
| 401 | TokenExpired | トークンの有効期限切れ | 新しいトークンを取得 |
| 401 | CertificateInvalid | クライアント証明書が無効（期限切れ、CA不一致） | 有効な証明書を再取得 |
| 500 | WebhookError | Webhook認証サービスへの通信エラー | Webhook認証サービスの状態を確認 |

### リトライ仕様

Webhook認証にはExponential Backoffによるリトライが設定可能。`WebhookRetryBackoff`パラメータで最大リトライ回数とバックオフ間隔を制御。

## トランザクション仕様

認証処理はステートレスであり、トランザクション管理は不要。トークンキャッシュはインメモリで管理される。

## パフォーマンス要件

- 認証処理のレイテンシ: キャッシュヒット時は1ms以下
- トークンキャッシュサイズ: 設定可能（デフォルトで十分な容量）
- Webhook認証のタイムアウト: TokenAccessReviewTimeoutで設定

## セキュリティ考慮事項

- クライアント証明書の検証にはCA証明書の厳格な管理が必要
- OIDCトークンの検証にはJWKSエンドポイントへの安全なアクセスが必要
- トークンキャッシュの過大なTTL設定はトークン失効後の不正アクセスリスクを増大させる
- フロントプロキシ認証は信頼されたプロキシからのリクエストのみに限定する必要がある

## 備考

Kubernetes 1.24以降、自動マウントされるServiceAccountトークンはバウンドトークン（有効期限付き、オーディエンス付き）に移行している。レガシートークン（Secret基盤）は非推奨。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | interfaces.go | `staging/src/k8s.io/apiserver/pkg/authentication/authenticator/interfaces.go` | Token, Requestインターフェースの定義。Response構造体（Audiences, User）を確認 |
| 1-2 | user/info.go | `staging/src/k8s.io/apiserver/pkg/authentication/user/` | user.Info インターフェース（Name, UID, Groups, Extra） |

**読解のコツ**: `authenticator.Request`インターフェースがHTTPリクエストから認証情報を抽出する中核。`authenticator.Token`はトークン文字列から認証する。両者は異なるレイヤーで使用される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | delegating.go | `staging/src/k8s.io/apiserver/pkg/authentication/authenticatorfactory/delegating.go` | DelegatingAuthenticatorConfig構造体がトークンレビュークライアント、キャッシュTTL、x509 CA等を保持 |

**主要処理フロー**:
1. **69行目**: `New()`関数で認証チェーンを構築
2. **75-80行目**: フロントプロキシ認証（RequestHeader）を最初に追加
3. トークン認証（Bearer Token）を追加
4. x509クライアント証明書認証を追加
5. 全認証方式をunion（OR結合）で統合

#### Step 3: 各認証方式を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | x509/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/x509/` | クライアント証明書認証。CN→ユーザー名、O→グループへのマッピング |
| 3-2 | bearertoken/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/bearertoken/` | Authorizationヘッダーからトークンを抽出 |
| 3-3 | headerrequest/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/headerrequest/` | X-Remote-User等のヘッダーからユーザー情報を抽出 |
| 3-4 | anonymous/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/anonymous/` | 匿名認証のフォールバック処理 |
| 3-5 | cache/ | `staging/src/k8s.io/apiserver/pkg/authentication/token/cache/` | トークン認証結果のキャッシュ機構 |

### プログラム呼び出し階層図

```
DelegatingAuthenticatorConfig.New()
    │
    ├─ headerrequest.NewDynamicVerifyOptionsSecure()  [フロントプロキシ認証]
    │
    ├─ x509.NewDynamic()  [クライアント証明書認証]
    │
    ├─ bearertoken.New()  [Bearer Token認証]
    │      └─ cache.New()  [トークンキャッシュ]
    │             └─ webhooktoken.New()  [Webhook Token認証]
    │
    ├─ websocket.NewProtocolAuthenticator()  [WebSocket認証]
    │
    └─ unionauth.New()  [認証チェーン統合]
           └─ group.NewAuthenticatedGroupAdder()  [グループ追加]
                  └─ anonymous.NewAuthenticator()  [匿名認証フォールバック]
```

### データフロー図

```
[入力]                    [処理]                              [出力]

HTTPリクエスト ──────▶ Authentication Chain ──────▶ authenticator.Response
├─ TLS証明書              ├─ x509 Authenticator                ├─ User.Name
├─ Authorization Header    ├─ BearerToken Authenticator         ├─ User.UID
├─ X-Remote-* Headers     ├─ HeaderRequest Authenticator       ├─ User.Groups
└─ WebSocket Protocol      ├─ Token Cache                      └─ Audiences
                           └─ Anonymous Authenticator
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| interfaces.go | `staging/src/k8s.io/apiserver/pkg/authentication/authenticator/interfaces.go` | ソース | 認証インターフェース定義 |
| delegating.go | `staging/src/k8s.io/apiserver/pkg/authentication/authenticatorfactory/delegating.go` | ソース | 委譲型認証チェーン構築 |
| x509/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/x509/` | ソース | x509クライアント証明書認証 |
| bearertoken/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/bearertoken/` | ソース | Bearer Token認証 |
| headerrequest/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/headerrequest/` | ソース | リクエストヘッダー認証 |
| anonymous/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/anonymous/` | ソース | 匿名認証 |
| union/ | `staging/src/k8s.io/apiserver/pkg/authentication/request/union/` | ソース | 認証チェーン統合 |
| cache/ | `staging/src/k8s.io/apiserver/pkg/authentication/token/cache/` | ソース | トークンキャッシュ |
| serviceaccount/ | `staging/src/k8s.io/apiserver/pkg/authentication/serviceaccount/` | ソース | ServiceAccount認証 |
