# 機能設計書 26-Certificate署名コントローラー

## 概要

本ドキュメントは、KubernetesのCertificate署名コントローラーの機能設計を記述する。CertificateSigningRequest（CSR）リソースの承認・署名を管理する抽象コントローラーフレームワークを提供する。

### 本機能の処理概要

**業務上の目的・背景**：Kubernetesクラスター内でのTLS通信を実現するため、証明書の署名プロセスを自動化する。Kubeletのクライアント証明書やサービング証明書のローテーションを自動的に処理し、運用負荷を軽減する。

**機能の利用シーン**：Kubeletが新しいクライアント証明書を要求した際にCSRを自動承認・署名する。カスタム署名者を実装する際のベースコントローラーとして使用する。kubectl certificates approve/deny操作後の署名処理。

**主要な処理内容**：
1. CertificateSigningRequestリソースの変更（作成・更新・削除）を監視する
2. プラグイン可能なhandler関数を通じてCSRの承認・署名処理を実行する
3. 既に証明書が発行済みのCSRはスキップする
4. カスタムレートリミッター（エクスポネンシャルバックオフ + バケットリミッター）を適用する

**関連システム・外部連携**：API Server（CSRの取得・更新）、CAキーペア（証明書署名用）

**権限による制御**：各署名者インスタンスは対応するsignerNameに対する署名権限を持つ。CSRの承認はRBAC Admission Controlで制御される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | kubectl certificate | API連携 | CSR承認後にCertificate署名コントローラーが証明書を署名する |

## 機能種別

データ処理 / 証明書管理（バックグラウンド自動処理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | string | Yes | コントローラーインスタンスの識別名 | 非空文字列 |
| workers | int | Yes | 同時実行ワーカー数 | 正の整数 |
| handler | func(context.Context, *CSR) error | Yes | CSR処理関数（署名ロジック） | nil不可 |

### 入力データソース

- CSR Informer: CertificateSigningRequestのadd/update/deleteイベント

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| CSR.Status.Certificate | []byte | 署名された証明書（PEM形式） |

### 出力先

- API Server: CSRステータス更新（Certificate フィールド）

## 処理フロー

### 処理シーケンス

```
1. NewCertificateController()でコントローラーを初期化する
   └─ CSR Informerのイベントハンドラ登録、カスタムレートリミッター設定
2. Run()でワーカーを起動する
   └─ キャッシュ同期待機後、workers個のワーカーを起動
3. syncFunc()でCSRを処理する
   └─ CSR取得 → Certificate存在チェック → handler呼び出し
4. handler()がCSRの署名処理を実行する
   └─ 具体的な署名ロジックは注入されたhandler関数に依存
```

### フローチャート

```mermaid
flowchart TD
    A[CSRイベント検知] --> B[キューに投入]
    B --> C[syncFunc]
    C --> D{CSR存在?}
    D -->|NotFound| E[ログ出力し終了]
    D -->|存在| F{Certificate発行済み?}
    F -->|Yes| G[スキップ]
    F -->|No| H[handler呼び出し]
    H --> I{成功?}
    I -->|Yes| J[queue.Forget]
    I -->|ignorableError| K[高レベルログ出力しリトライ]
    I -->|その他エラー| L[エラーログ出力しリトライ]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-26-01 | 発行済みスキップ | Status.Certificateが既に存在するCSRは処理をスキップする | syncFunc内 |
| BR-26-02 | 不変参照 | 処理前にDeepCopy()でCSRをコピーし、共有キャッシュの変更を防止する | syncFunc内 |
| BR-26-03 | ignorableError | ユーザーエラー等のspammy errorはV(4)レベルでログ出力 | processNextWorkItem内 |
| BR-26-04 | カスタムレートリミッター | エクスポネンシャルバックオフ(200ms-1000s) + バケットリミッター(10qps/100バースト) | キュー設定 |

### 計算ロジック

特段の計算ロジックはなし。署名処理は注入されたhandler関数に委譲される。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| CSR取得 | CSR（etcd） | SELECT | CSRオブジェクトの取得 |
| CSRステータス更新 | CSR（etcd） | UPDATE | Certificate等のステータス更新（handler内） |

### テーブル別操作詳細

#### CertificateSigningRequest（etcd経由）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| UPDATE | status.certificate | 署名された証明書（PEM） | handler関数内で実行 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NotFound | API応答 | CSRが削除済み | ログ出力し正常終了 |
| ignorableError | 内部 | ユーザー入力エラー等 | V(4)でログ出力しRateLimitedリトライ |
| その他エラー | 内部/API | handler処理失敗 | HandleErrorでログ出力しRateLimitedリトライ |

### リトライ仕様

- エクスポネンシャルバックオフ: 200ms初期値、最大1000秒
- バケットリミッター: 全体で10qps、バースト100
- 成功時: queue.Forget()でバックオフリセット

## トランザクション仕様

CSRのステータス更新はResourceVersionによる楽観的並行性制御が適用される。handler内での証明書署名とステータス更新はアトミックではない。

## パフォーマンス要件

- カスタムレートリミッターにより、大量CSR発行時のAPI Server負荷を制御
- handler処理はCSRごとに独立しており、workers数で並行度を制御

## セキュリティ考慮事項

- CA秘密鍵の管理が最重要。署名コントローラーはCA鍵にアクセスするため、適切なアクセス制御が必要
- signerNameによる署名者の分離
- handler関数でCSRの内容検証（リクエストされたSAN等）を実施する必要がある

## 備考

- 本コントローラーは抽象フレームワークであり、具体的な署名ロジックはhandler関数として注入される
- kube-controller-managerではKubeAPIServerClientSigner、KubeAPIServerClientKubeletSigner、LegacyUnknownSigner等が実装されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | CertificateController構造体（42-54行目）。name、kubeClient、csrLister、handler（プラグイン可能な関数）、queueを保持 |
| 1-2 | certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | ignorableError型（212-216行目）。spammy errorを区別するための型 |

**読解のコツ**: handler fieldはfunc(context.Context, *certificates.CertificateSigningRequest) errorのシグネチャを持つ。この関数が実際の署名ロジックを担当する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | NewCertificateController（56-113行目）。カスタムレートリミッター設定（67-76行目）、CSR InformerにAdd/Update/Deleteハンドラ登録（81-109行目） |
| 2-2 | certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | Run（116-139行目）。キャッシュ同期待機、workers個のワーカー起動 |

**主要処理フロー**:
1. **67-76行目**: エクスポネンシャルバックオフ(200ms-1000s) + バケットリミッター(10qps/100)のMaxOfRateLimiter
2. **81-109行目**: Add/Update/Deleteでenqueue。Delete時はTombstone対応あり
3. **129-131行目**: キャッシュ同期待機
4. **133-137行目**: workers個のworker goroutine起動

#### Step 3: 同期処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | syncFunc（179-202行目）。CSR取得、Certificate存在チェック（194行目）、DeepCopy（200行目）、handler呼び出し（201行目） |
| 3-2 | certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | processNextWorkItem（148-168行目）。ignorableError判定（157行目）でログレベルを分岐 |

### プログラム呼び出し階層図

```
CertificateController.Run()
    │
    └─ worker()
           └─ processNextWorkItem()
                  └─ syncFunc()
                         ├─ csrLister.Get() ── CSR取得
                         ├─ Certificate存在チェック
                         ├─ DeepCopy()
                         └─ handler() ── 注入された署名ロジック実行
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CSRイベント ──────────▶ processNextWorkItem()
(Informer)                 │
                          └─ syncFunc()
                                │
                                ├─ Certificate存在確認
                                │
                                └─ handler() ──▶ CSR Status更新
                                    (署名ロジック)    (Certificate設定)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| certificate_controller.go | `pkg/controller/certificates/certificate_controller.go` | ソース | 抽象Certificateコントローラーフレームワーク |
| signer/ | `pkg/controller/certificates/signer/` | ソース | 具体的な署名者実装（KubeAPIServerClient等） |
| approver/ | `pkg/controller/certificates/approver/` | ソース | CSR自動承認ロジック |
