# 機能設計書 3-認可（Authorization）

## 概要

本ドキュメントは、Kubernetes API Serverにおける認可（Authorization）機能の設計書である。RBAC、ABAC、Webhook等のモードによるリクエストの認可判定を行う。

### 本機能の処理概要

認可機能は、認証済みリクエストに対して、要求されたリソース操作が許可されているかを判定する。複数の認可モードをチェーンとして構成し、いずれかのモードが許可を返した時点で認可成功とする。

**業務上の目的・背景**：認証で特定されたユーザーが、要求するリソース操作（get, list, create, update, delete等）を実行する権限を持つかどうかを判定する。最小権限の原則に基づき、ユーザーには必要最小限のアクセス権のみを付与する。マルチテナント環境でのNamespace分離、機密リソースへのアクセス制限、監査コンプライアンス対応など、クラスターの安全な運用に不可欠な機能である。

**機能の利用シーン**：認証成功後の全てのAPIリクエストに対して自動的に実行される。開発者のNamespace内リソースへのアクセス制限、管理者のクラスター全体操作の許可、CI/CDパイプラインのServiceAccountに対する限定的な権限付与、kubectl auth can-iコマンドによる権限確認など。

**主要な処理内容**：
1. RBAC（Role-Based Access Control）による認可判定
2. ABAC（Attribute-Based Access Control）による認可判定
3. Webhook認可（外部認可サービスへの委譲）
4. Node認可（Kubeletのリクエストに特化した認可）
5. AlwaysAllow/AlwaysDeny（テスト・開発用）
6. SubjectAccessReview APIによるプログラマティックな認可確認

**関連システム・外部連携**：Webhook認可サービス、etcd（RBAC定義の永続化）

**権限による制御**：RBAC定義（Role, ClusterRole, RoleBinding, ClusterRoleBinding）により、ユーザー/グループ/ServiceAccountごとのリソースアクセスを制御。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 26 | kubectl auth | API連携 | RBAC等の認可モードに対してアクセス権限を問い合わせる |

## 機能種別

認可処理 / セキュリティ基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| User | user.Info | Yes | 認証済みユーザー情報 | 認証済みであること |
| Verb | string | Yes | APIリクエストの操作（get, list, create, update, patch, delete, watch等） | 有効な動詞であること |
| Resource | string | Yes | 対象リソースタイプ（pods, services等） | 有効なリソースタイプ |
| Namespace | string | No | 対象Namespace（Namespace scopedリソースの場合） | 存在するNamespace |
| Name | string | No | 対象リソース名 | - |
| APIGroup | string | No | APIグループ（core, apps, batch等） | 有効なAPIグループ |
| Subresource | string | No | サブリソース（status, scale等） | 有効なサブリソース |
| Path | string | No | 非リソースパス（/healthz等） | 有効なパス |

### 入力データソース

認証結果のユーザー情報、HTTPリクエストから抽出されたリソース属性

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Decision | int | 認可判定結果（Allowed, Denied, NoOpinion） |
| Reason | string | 判定理由の説明 |

### 出力先

API Serverの認可チェーン結果（後続のアドミッション制御への進行可否を決定）

## 処理フロー

### 処理シーケンス

```
1. 認証済みリクエストの受信
   └─ ユーザー情報とリクエスト属性（verb, resource, namespace等）を取得
2. 認可チェーンの実行
   └─ 設定された認可モードを順次実行
3. RBAC認可
   └─ RoleBinding/ClusterRoleBindingを検索し、対応するRole/ClusterRoleのルールと照合
4. Node認可
   └─ Kubeletのリクエストに対してノードが管理するリソースへのアクセスのみ許可
5. Webhook認可
   └─ SubjectAccessReviewをWebhookサービスに送信して判定を委譲
6. 判定結果の返却
   └─ いずれかのモードがAllowを返せば許可、全てがNoOpinionまたはDenyなら拒否
```

### フローチャート

```mermaid
flowchart TD
    A[認可チェーン開始] --> B{RBAC認可}
    B -->|Allow| G[認可成功]
    B -->|NoOpinion| C{Node認可}
    B -->|Deny| F[403 Forbidden]
    C -->|Allow| G
    C -->|NoOpinion| D{Webhook認可}
    D -->|Allow| G
    D -->|NoOpinion/Deny| F
    G --> H[アドミッション制御へ]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-1 | Allow優先 | チェーン内でいずれかのモードがAllowを返せば認可成功 | 全リクエスト |
| BR-2 | 明示的Deny | Denyを返すモードがあれば即時拒否 | 全リクエスト |
| BR-3 | RBAC最小権限 | RBACはデフォルトで全て拒否、明示的にRoleで許可されたもののみ許可 | RBACモード有効時 |
| BR-4 | ClusterRoleのAggregation | aggregationRuleを持つClusterRoleは、matchLabelsに一致する他のClusterRoleのルールを自動集約 | ClusterRole更新時 |
| BR-5 | system:masters特権 | system:mastersグループに所属するユーザーは全リソースへのアクセスが許可される | RBAC有効時 |

### 計算ロジック

RBAC認可判定は、Subjectに一致するRoleBinding/ClusterRoleBindingを検索し、参照先のRole/ClusterRoleのルール（resources, verbs, apiGroups）がリクエスト属性と一致するかを判定する。ワイルドカード（*）はすべてにマッチする。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| RBAC定義読み取り | etcd (roles, clusterroles, rolebindings, clusterrolebindings) | SELECT | RBAC定義をInformerキャッシュ経由で読み取り |
| SubjectAccessReview | etcd | SELECT | 認可確認API（can-i）の結果を返却 |

### テーブル別操作詳細

#### etcd (RBAC)

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | /registry/clusterroles/{name} | ClusterRole定義の読み取り | Informerキャッシュで高速化 |
| SELECT | /registry/clusterrolebindings/{name} | ClusterRoleBinding定義の読み取り | Informerキャッシュで高速化 |
| SELECT | /registry/roles/{namespace}/{name} | Role定義の読み取り | Namespace scoped |
| SELECT | /registry/rolebindings/{namespace}/{name} | RoleBinding定義の読み取り | Namespace scoped |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 403 | Forbidden | 認可に失敗した場合 | 適切なRBACロールを付与 |
| 500 | InternalServerError | RBAC定義の読み取りエラー | etcdクラスターの状態を確認 |
| 500 | WebhookError | Webhook認可サービスへの通信エラー | Webhookサービスの状態を確認 |

### リトライ仕様

Webhook認可にはリトライ設定が可能。failurePolicy設定により、Webhook到達不能時の動作（Fail/Ignore）を制御可能。

## トランザクション仕様

認可処理はステートレスであり、RBACルールの読み取りはInformerキャッシュから行われるため、トランザクション管理は不要。

## パフォーマンス要件

- RBAC認可判定: Informerキャッシュからの読み取りのため、サブミリ秒レベル
- Webhook認可: 外部サービスのレイテンシに依存

## セキュリティ考慮事項

- RBAC定義の変更は管理者権限に限定すべき
- system:mastersグループへの所属は厳格に管理する必要がある
- ABACポリシーファイルはファイルシステムレベルで保護が必要
- Webhook認可サービスとの通信はTLSで暗号化すべき

## 備考

RBACが推奨認可モードであり、Kubernetes 1.6以降でデフォルト有効。ABACは非推奨に向かっている。複数の認可モードを組み合わせることが可能（例: RBAC + Node）。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | authorizer/interfaces.go | `staging/src/k8s.io/apiserver/pkg/authorization/authorizer/` | Authorizerインターフェース（Authorize メソッド）、Attributes インターフェース |
| 1-2 | rbac types | `staging/src/k8s.io/api/rbac/v1/types.go` | Role, ClusterRole, RoleBinding, ClusterRoleBinding, PolicyRule の構造体定義 |

**読解のコツ**: Authorizer インターフェースの`Authorize(ctx context.Context, a Attributes) (Decision, string, error)`が認可判定の核心。Decision は Allowed, Denied, NoOpinion の3値。

#### Step 2: RBAC認可を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | rbac/ | `pkg/registry/rbac/` | RBAC関連リソースのレジストリ（CRUD操作） |
| 2-2 | rbac authorizer | `plugin/pkg/auth/authorizer/rbac/` | RBAC認可の実装 |

#### Step 3: ABAC認可を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | abac.go | `pkg/auth/authorizer/abac/abac.go` | ABACポリシーのファイル読み込みと認可判定。PolicyList型とNewFromFile関数 |

**主要処理フロー**:
- **56行目**: NewFromFileでポリシーファイルを読み込み
- ポリシーファイルは1行1ポリシーのJSON形式

### プログラム呼び出し階層図

```
Authorization Chain
    │
    ├─ RBAC Authorizer [plugin/pkg/auth/authorizer/rbac/]
    │      ├─ RuleResolver.RulesFor()
    │      │      ├─ ClusterRoleBinding検索
    │      │      └─ RoleBinding検索
    │      └─ PolicyRule照合
    │
    ├─ Node Authorizer [plugin/pkg/auth/authorizer/node/]
    │      └─ NodeAuthorizer.Authorize()
    │
    ├─ ABAC Authorizer [pkg/auth/authorizer/abac/]
    │      └─ PolicyList.Authorize()
    │
    └─ Webhook Authorizer [staging/.../plugin/pkg/authorizer/webhook/]
           └─ SubjectAccessReview送信
```

### データフロー図

```
[入力]                      [処理]                         [出力]

認証済みユーザー情報 ──▶ Authorization Chain ──────▶ Decision
リクエスト属性            │                            (Allow/Deny/NoOpinion)
├─ Verb                  ├─ RBAC Authorizer
├─ Resource              │    └─ Informer Cache ◀── etcd
├─ Namespace             ├─ Node Authorizer
├─ APIGroup              ├─ ABAC Authorizer
└─ Subresource           └─ Webhook Authorizer
                              └─ External Service
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| authorizer/ | `staging/src/k8s.io/apiserver/pkg/authorization/authorizer/` | ソース | 認可インターフェース定義 |
| rbac/ | `plugin/pkg/auth/authorizer/rbac/` | ソース | RBAC認可実装 |
| abac.go | `pkg/auth/authorizer/abac/abac.go` | ソース | ABAC認可実装 |
| node/ | `plugin/pkg/auth/authorizer/node/` | ソース | Node認可実装 |
| webhook/ | `staging/src/k8s.io/apiserver/plugin/pkg/authorizer/webhook/` | ソース | Webhook認可実装 |
| rbac registry | `pkg/registry/rbac/` | ソース | RBACリソースのレジストリ |
