# 機能設計書 34-Replicationコントローラー

## 概要

本ドキュメントは、Kubernetes kube-controller-managerに含まれるReplicationコントローラー（ReplicationManager）の機能設計を記述する。ReplicationControllerリソースで定義されたPodのレプリカ数を維持する処理を担当する。

### 本機能の処理概要

**業務上の目的・背景**：ReplicationControllerは旧来のKubernetesリソースであり、指定されたレプリカ数のPodを維持する役割を持つ。現在はReplicaSetが推奨されるが、後方互換性のためReplicationControllerもサポートされている。ReplicationManagerはReplicaSetControllerのラッパーであり、ReplicationControllerをReplicaSetの旧バージョンAPIとして変換レイヤーを通じて処理する。

**機能の利用シーン**：ReplicationControllerリソースの作成・更新・削除時、およびPodの障害やスケーリング時に自動的に動作する。

**主要な処理内容**：
1. ReplicationControllerリソースの変更を監視する
2. 現在のPod数とspec.replicasの差分を計算する
3. Podが不足している場合、新しいPodを作成する
4. Podが過剰な場合、余分なPodを削除する
5. ReplicationControllerのステータスを更新する

**関連システム・外部連携**：ReplicaSetController（内部ラッパー元）、Pod、Kubernetes API Server

**権限による制御**：ReplicationController、Podに対するCRUD権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | kubectl get rc | 参照画面 | ReplicationControllerの一覧・状態確認 |
| - | kubectl scale rc | 主機能 | レプリカ数の変更トリガー |

## 機能種別

レプリカ管理 / Pod作成・削除操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| RC.Spec.Replicas | *int32 | Yes | 望ましいレプリカ数 | 0以上 |
| RC.Spec.Selector | map[string]string | Yes | Podのラベルセレクタ | 空でないこと |
| RC.Spec.Template | v1.PodTemplateSpec | Yes | Podテンプレート | 有効なPod仕様 |
| burstReplicas | int | Yes | 一度に作成・削除するPodの最大数 | デフォルト500 |

### 入力データソース

- ReplicationController Informer: RCの変更イベント
- Pod Informer: Podの変更イベント

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Pod作成 | v1.Pod | 不足分のPod作成 |
| Pod削除 | - | 過剰分のPod削除 |
| RC.Status | v1.ReplicationControllerStatus | レプリカ数ステータス更新 |

### 出力先

- Kubernetes API Server（Pod作成・削除、RC Status更新）

## 処理フロー

### 処理シーケンス

```
1. RC/Pod変更イベントを受信
   └─ RCのキーをワークキューに追加
2. ワーカーがキューからキーを取得
3. syncReplicationController
   └─ セレクタに一致するPodの一覧を取得
   └─ 現在のアクティブPod数を計算
   └─ spec.replicasとの差分を計算
4. 差分に応じた操作
   └─ 不足 → slowStartBatchでPodをバッチ作成
   └─ 過剰 → Pod削除（Not Readyを優先）
5. RC Status更新
   └─ replicas, readyReplicas, availableReplicas等
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-34-01 | レプリカ維持 | spec.replicasで指定されたPod数を維持する | RC存在時 |
| BR-34-02 | バーストリミット | 一度に作成・削除するPod数をBurstReplicas（500）に制限する | Pod数調整時 |
| BR-34-03 | SlowStart | Pod作成は1, 2, 4, 8...と倍増バッチで実行する | Pod不足時 |
| BR-34-04 | 削除優先順位 | NotReady PodをReady Podより優先的に削除する | Pod過剰時 |
| BR-34-05 | ReplicaSet委譲 | 実装はReplicaSetControllerに完全に委譲し、変換レイヤーで対応する | 全操作 |
| BR-34-06 | terminatingReplicas無効 | ReplicationControllerではterminatingReplicasフィールドをサポートしない | ステータス更新時 |

### 計算ロジック

- 差分計算: `diff = currentPodCount - desiredReplicas`
- バッチサイズ: `min(diff, BurstReplicas)` で制限

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| Pod作成 | Pod | CREATE | テンプレートに基づくPod作成 |
| Pod削除 | Pod | DELETE | 過剰Podの削除 |
| Status更新 | ReplicationController | UPDATE (Status) | レプリカ数の更新 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NotFound | RC不在 | RCが既に削除されている | 正常終了扱い |
| Conflict | 競合エラー | 同時更新 | レートリミット付きリトライ |

## トランザクション仕様

Pod作成・削除とRC Status更新は別々のAPI呼び出し。バッチ内のPod作成は並行実行される。

## パフォーマンス要件

- BurstReplicas: 500
- ワーカー数: kube-controller-managerの設定に依存

## セキュリティ考慮事項

- ownerReferencesを通じてRCとPodの所有関係を管理
- ControllerRefでPodの所有者を厳密に確認

## 備考

- ReplicationManagerはReplicaSetControllerの薄いラッパーであり、独自のロジックはほとんどない
- RC固有の処理として`EnableStatusTerminatingReplicas: false`が設定されている（replication_controller.go 69行目）
- informerAdapter, clientsetAdapter, podControlAdapterの変換レイヤーでRC/RSの違いを吸収する

---

## コードリーディングガイド

### 推奨読解順序

#### Step 1: ラッパー構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | replication_controller.go | `pkg/controller/replication/replication_controller.go` | ReplicationManager構造体（48-50行目）がReplicaSetControllerを埋め込み |

**読解のコツ**: ReplicationManagerは`replicaset.ReplicaSetController`を埋め込んでおり、全メソッドをそのまま委譲する。独自ロジックはほぼない。

#### Step 2: 変換アダプタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | conversion.go | `pkg/controller/replication/conversion.go` | informerAdapter, clientsetAdapter, podControlAdapterの変換ロジック |

**読解のコツ**: RC <-> RS間の型変換をアダプターパターンで実現。RCのインターフェースをRSのインターフェースに変換する。

#### Step 3: NewReplicationManagerの初期化を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | replication_controller.go | `pkg/controller/replication/replication_controller.go` | NewReplicationManager関数（53-73行目） |

**主要処理**:
- **57行目**: replicaset.NewBaseControllerを呼び出し、RS用コントローラーを作成
- **58行目**: GVKとして`v1.SchemeGroupVersion.WithKind("ReplicationController")`を指定
- **66-70行目**: ReplicaSetControllerFeaturesでterminatingReplicasを無効化

### プログラム呼び出し階層図

```
NewReplicationManager (53行目)
    └─ replicaset.NewBaseController (57行目)
        ├─ informerAdapter: RCInformerをRSInformer化
        ├─ clientsetAdapter: KubeClientをRS用化
        └─ podControlAdapter: PodControlをRS用化

[以降はReplicaSetControllerの処理フロー]
Run
    └─ worker
        └─ syncReplicationController
            ├─ claimPods (OwnerReference管理)
            ├─ manageReplicas (Pod作成・削除)
            │   ├─ slowStartBatch (Pod作成)
            │   └─ deletePod (Pod削除)
            └─ updateStatus (RC Status更新)
```

### データフロー図

```
[入力]                          [処理]                          [出力]

RC変更イベント ──▶ informerAdapter ──▶ ReplicaSetController ──▶ Pod作成/削除
                  (RC→RS変換)                                    │
Pod変更イベント ──▶ claimPods ──▶ manageReplicas              ──▶ RC Status更新
                                     │
                              clientsetAdapter
                              (RS→RC変換で API呼び出し)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| replication_controller.go | `pkg/controller/replication/replication_controller.go` | ソース | ReplicationManager本体（ラッパー） |
| conversion.go | `pkg/controller/replication/conversion.go` | ソース | RC-RS変換アダプター |
| replicaset.go | `pkg/controller/replicaset/replica_set.go` | ソース | 実際のロジック（ReplicaSetController） |
