# 機能設計書 54-リソース管理（CPU/メモリ）

## 概要

本ドキュメントは、KubeletのCPUManager、MemoryManager、TopologyManagerによるリソースの割り当てと管理の設計を記載する。

### 本機能の処理概要

Kubeletのリソース管理機能は、ノード上のCPUコアとメモリをPod/コンテナに対して精密に割り当てるための3つのマネージャー（CPUManager、MemoryManager、TopologyManager）で構成される。NUMA（Non-Uniform Memory Access）アーキテクチャを考慮した最適なリソース配置を実現し、高性能ワークロードの要件を満たす。

**業務上の目的・背景**：テレコム、HPC（High-Performance Computing）、ML推論などのワークロードでは、CPUコアの排他的割り当てやNUMAノードを意識したメモリ配置が性能に大きく影響する。本機能により、Kubernetes上でもベアメタル並みのリソース制御が可能となる。

**機能の利用シーン**：Guaranteed QoSクラスのPodで整数CPUリクエストを指定した場合のCPU排他割り当て、NUMAアウェアなメモリ割り当て、GPUなどのデバイスとCPU/メモリの同一NUMAノード配置。

**主要な処理内容**：
1. CPUManager: CPUコアのExclusive割り当てとShared Pool管理
2. MemoryManager: NUMAノード単位でのメモリ割り当て管理
3. TopologyManager: CPU、メモリ、デバイスのNUMAアフィニティの統合調整
4. 状態永続化（チェックポイント）とreconcileループによる整合性維持
5. Pod Admission時のリソース割り当て可否判定

**関連システム・外部連携**：CRI（コンテナリソース更新）、cAdvisor（ハードウェアトポロジー情報）、Device Plugin（デバイスのトポロジーヒント）。

**権限による制御**：CPUの排他割り当てはGuaranteed QoSクラスかつ整数CPUリクエストのコンテナに限定される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | Kubelet内部機能のため画面関連なし |

## 機能種別

リソース管理 / 計算処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| cpuManagerPolicy | string | Yes | CPUManagerポリシー（none/static） | none or static |
| memoryManagerPolicy | string | Yes | MemoryManagerポリシー（None/Static） | None or Static |
| topologyManagerPolicy | string | Yes | TopologyManagerポリシー（none/best-effort/restricted/single-numa-node） | 4種のいずれか |
| topologyManagerScope | string | Yes | TopologyManagerスコープ（container/pod） | container or pod |
| reservedCPUs | cpuset.CPUSet | No | システム予約CPUセット | 有効なCPU番号 |

### 入力データソース

- cAdvisor: CPUトポロジー情報（ソケット、コア、スレッド）、NUMAノード情報
- Pod仕様: CPUリクエスト/リミット、メモリリクエスト/リミット
- チェックポイントファイル: 前回の割り当て状態

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| CPUSet | cpuset.CPUSet | コンテナに割り当てられたCPUセット |
| TopologyHints | map[string][]TopologyHint | NUMAアフィニティヒント |
| ContainerResources | *runtimeapi.ContainerResources | CRI経由でコンテナに設定するリソース |

### 出力先

- CRI（UpdateContainerResources API）
- チェックポイントファイル（状態永続化）
- TopologyManager（ヒント情報）

## 処理フロー

### 処理シーケンス

```
1. Kubelet起動時のマネージャー初期化
   └─ CPUトポロジー検出、チェックポイント復元、ポリシー初期化
2. Pod Admission時の割り当て
   └─ TopologyManagerがヒント収集 → ポリシー判定 → 各マネージャーAllocate
3. コンテナ追加時のマッピング記録
   └─ AddContainer でcontainerID → CPU/メモリの対応を記録
4. Reconcileループ
   └─ 実際のコンテナ状態とdesired stateの差分を検出・修復
5. コンテナ削除時のリソース解放
   └─ RemoveContainer でCPU/メモリを解放
```

### フローチャート

```mermaid
flowchart TD
    A[Pod Admission] --> B[TopologyManager.Admit]
    B --> C[各HintProviderからヒント収集]
    C --> D{ポリシー判定}
    D -->|best-effort| E[最善のアフィニティで割り当て]
    D -->|restricted| F{最適なアフィニティあり?}
    D -->|single-numa-node| G{単一NUMAで充足?}
    F -->|No| H[Admission拒否]
    G -->|No| H
    F -->|Yes| I[割り当て実行]
    G -->|Yes| I
    E --> I
    I --> J[CPUManager.Allocate]
    I --> K[MemoryManager.Allocate]
    J --> L[チェックポイント保存]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-54-01 | CPU排他割り当て | Guaranteed QoSかつ整数CPUリクエストのコンテナにCPUコアを排他的に割り当て | Static ポリシー時 |
| BR-54-02 | Shared Pool | 排他割り当て対象外のコンテナはShared CPUプールを共有 | 全ポリシー時 |
| BR-54-03 | NUMA整合性 | TopologyManager restrictedポリシーでは最適なNUMAアフィニティがない場合Podを拒否 | restrictedポリシー時 |
| BR-54-04 | 単一NUMA | single-numa-nodeポリシーでは全リソースが単一NUMAノードに収まる必要がある | single-numa-nodeポリシー時 |
| BR-54-05 | 状態復元 | Kubelet再起動時にチェックポイントから前回の割り当て状態を復元 | Kubelet起動時 |

### 計算ロジック

- CPUアフィニティスコア = NUMAノード数の逆数（少ないNUMAノードほど高スコア）
- Shared Pool = 全CPU - Reserved CPU - Exclusively Allocated CPU

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 状態永続化 | cpu_manager_state（ファイル） | INSERT/UPDATE | CPU割り当て状態のチェックポイント |
| メモリ状態 | memory_manager_state（ファイル） | INSERT/UPDATE | メモリ割り当て状態のチェックポイント |

### テーブル別操作詳細

#### cpu_manager_state（チェックポイントファイル）

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| UPDATE | assignments | containerID → CPUSet | Allocate/RemoveContainer時 |
| UPDATE | defaultCPUSet | Shared Pool CPUSet | 変更時 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TopologyAffinityError | Admission拒否 | NUMAアフィニティを満たせない | Podを別ノードにスケジュール |
| InsufficientCPU | リソース不足 | 排他CPU不足 | ノードのCPU容量を確認 |
| CheckpointCorrupted | 状態エラー | チェックポイント破損 | チェックポイントを再生成 |

### リトライ仕様

- Reconcileループ: デフォルト10秒間隔で継続実行
- CRI UpdateContainerResources失敗時は次回reconcileで再試行

## トランザクション仕様

各マネージャーはsync.Mutexで排他制御。チェックポイントへの書き込みはアトミック操作。

## パフォーマンス要件

- Reconcileループ間隔: 10秒（デフォルト）
- CPU割り当て計算は整数線形計画法に近い最適化を実行

## セキュリティ考慮事項

- チェックポイントファイルはKubeletの作業ディレクトリに保存、適切なファイルパーミッション設定
- CPU排他割り当てによりサイドチャネル攻撃のリスクを軽減

## 備考

- CPUManager Staticポリシーのオプション: full-pcpus-only、distribute-cpus-across-numa、align-by-socket、strict-cpu-reservation
- TopologyManagerは「ヒント提供者」パターンにより、CPUManager、MemoryManager、DeviceManagerからのヒントを統合する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | topology_manager.go | `pkg/kubelet/cm/topologymanager/topology_manager.go` | TopologyManager インターフェースとStore |
| 1-2 | cpu_manager.go | `pkg/kubelet/cm/cpumanager/cpu_manager.go` | Manager インターフェース |
| 1-3 | memory_manager.go | `pkg/kubelet/cm/memorymanager/memory_manager.go` | Manager インターフェース |

**読解のコツ**: TopologyManagerは「ヒント提供者」パターンを採用。各マネージャーがTopologyHintを提供し、TopologyManagerがそれらを統合してNUMAアフィニティを決定する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | cpu_manager.go | `pkg/kubelet/cm/cpumanager/cpu_manager.go` | Manager.Start、Allocate、RemoveContainer |

**主要処理フロー**:
1. **56-60行目**: Manager.Start - reconcileループ開始
2. **65行目**: Allocate - Pod Admission時のCPU割り当て
3. **74行目**: RemoveContainer - コンテナ削除時のCPU解放

#### Step 3: ポリシー実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | policy_static.go | `pkg/kubelet/cm/cpumanager/policy_static.go` | Static CPUポリシーの割り当てアルゴリズム |
| 3-2 | policy_best_effort.go | `pkg/kubelet/cm/topologymanager/policy_best_effort.go` | TopologyManager best-effortポリシー |
| 3-3 | policy_restricted.go | `pkg/kubelet/cm/topologymanager/policy_restricted.go` | TopologyManager restrictedポリシー |

### プログラム呼び出し階層図

```
Kubelet.HandlePodAdditions()
    |
    +-- TopologyManager.Admit()
    |       +-- CPUManager.GetTopologyHints()
    |       +-- MemoryManager.GetTopologyHints()
    |       +-- DeviceManager.GetTopologyHints()
    |       +-- Policy.Merge() → NUMAアフィニティ決定
    |       +-- CPUManager.Allocate()
    |       +-- MemoryManager.Allocate()
    |
    +-- CPUManager.AddContainer()
    |       +-- CRI UpdateContainerResources()
    |
    +-- CPUManager.reconcileState() [定期ループ]
            +-- CRI UpdateContainerResources()
```

### データフロー図

```
[入力]                   [処理]                          [出力]

cAdvisor              ──▶ CPUトポロジー検出       ──▶ CPUSet割り当て
(CPU/NUMAトポロジー)

Pod仕様              ──▶ TopologyManager.Admit()  ──▶ Admission判定
(CPU/メモリrequest)       |
                          +-- CPUManager.Allocate()
                          +-- MemoryManager.Allocate()
                                                    ──▶ チェックポイント保存

チェックポイント      ──▶ 状態復元                ──▶ CRI UpdateContainerResources
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| cpu_manager.go | `pkg/kubelet/cm/cpumanager/cpu_manager.go` | ソース | CPUManager主実装 |
| policy_static.go | `pkg/kubelet/cm/cpumanager/policy_static.go` | ソース | Static CPUポリシー |
| policy_none.go | `pkg/kubelet/cm/cpumanager/policy_none.go` | ソース | None CPUポリシー |
| cpu_assignment.go | `pkg/kubelet/cm/cpumanager/cpu_assignment.go` | ソース | CPU割り当てアルゴリズム |
| memory_manager.go | `pkg/kubelet/cm/memorymanager/memory_manager.go` | ソース | MemoryManager主実装 |
| topology_manager.go | `pkg/kubelet/cm/topologymanager/topology_manager.go` | ソース | TopologyManager主実装 |
| policy_best_effort.go | `pkg/kubelet/cm/topologymanager/policy_best_effort.go` | ソース | best-effortポリシー |
| policy_restricted.go | `pkg/kubelet/cm/topologymanager/policy_restricted.go` | ソース | restrictedポリシー |
| policy_single_numa_node.go | `pkg/kubelet/cm/topologymanager/policy_single_numa_node.go` | ソース | single-numa-nodeポリシー |
| state/ | `pkg/kubelet/cm/cpumanager/state/` | ソース | CPU状態管理・チェックポイント |
