# Kubernetes 業務運用マニュアル

## 目次

**【基本操作】**
1. [システム概要](#1-システム概要)
2. [ログイン・ログアウト手順（クラスタ認証）](#2-ログインログアウト手順クラスタ認証)
3. [トップページ（初期操作画面）](#3-トップページ初期操作画面)

**【業務フローに沿った機能】**
4. [【STEP1】案内・通知機能（イベント・通知の配信・受信）](#4-step1案内通知機能イベント通知の配信受信)
5. [【STEP2-3】依頼・届出機能（リソースの定義・適用）](#5-step2-3依頼届出機能リソースの定義適用)
6. [【STEP3-4】ファイル共有機能（マニフェスト・設定の管理）](#6-step3-4ファイル共有機能マニフェスト設定の管理)
7. [【STEP4-5】管理・処理完了（クラスタ管理者向け）](#7-step4-5管理処理完了クラスタ管理者向け)

**【サポート機能】**
8. [FAQ機能（ヘルプ・リファレンス）](#8-faq機能ヘルプリファレンス)
9. [お問合せ機能（トラブルシューティング・デバッグ）](#9-お問合せ機能トラブルシューティングデバッグ)
10. [マイページ機能（kubeconfig設定管理）](#10-マイページ機能kubeconfig設定管理)

**【運用ガイド】**
11. [日常業務の流れ](#11-日常業務の流れ)
12. [よくある質問と対処法](#12-よくある質問と対処法)
13. [トラブルシューティング](#13-トラブルシューティング)

---

## 1. システム概要

### 1.1 Kubernetesとは

Kubernetes（K8s）は、複数のホストにまたがるコンテナ化されたアプリケーションの展開、スケーリング、運用管理を自動化するためのオープンソースのコンテナオーケストレーションプラットフォームです。Cloud Native Computing Foundation（CNCF）がホストするプロジェクトであり、CLIツール（kubectl / kubeadm）を通じてクラスタの操作・管理を行います。

本システムはGUI（Webブラウザ画面）を持たないCLIベースのシステムです。操作はすべてターミナル（コマンドライン）から実行します。

### 1.2 本システムで行う主要業務

本システムは、主に以下の業務を支援します：

| 業務カテゴリ | 具体的な業務内容 |
|------------|----------------|
| **ワークロード管理** | Deployment/StatefulSet/DaemonSet/Jobによるアプリケーションのデプロイ・管理、Pod のスケーリング |
| **クラスタ管理** | ノードの追加・メンテナンス・監視、Namespace管理、リソースクォータ設定 |
| **ネットワーク管理** | Serviceの公開・ルーティング、EndpointSliceによるサービスディスカバリ |
| **ストレージ管理** | PersistentVolume/PersistentVolumeClaimの管理、CSIドライバーとの連携 |
| **セキュリティ管理** | RBAC/ABACによるアクセス制御、証明書管理、ServiceAccount管理 |
| **バッチ処理** | Job/CronJobによるバッチジョブの実行・スケジューリング |
| **オートスケーリング** | HPAによるPodの自動スケーリング |
| **監視・可観測性** | リソース使用量の監視、イベント・ログの確認、メトリクス収集 |

### 1.3 業務フロー概要

Kubernetesにおける業務は、以下の流れで進行します：

```
+-------------------------------------------------------------------+
|                        業務フロー                                  |
+-------------------------------------------------------------------+
|                                                                    |
|  【STEP 1】管理者がクラスタをセットアップし、通知・設定を配信       |
|      |                                                             |
|  【STEP 2】オペレーターが設定・通知を確認し、要件を把握            |
|      |                                                             |
|  【STEP 3】オペレーターがマニフェストを作成し、リソースを適用       |
|      |                                                             |
|  【STEP 4】管理者がリソースの状態を確認・処理                      |
|      |                                                             |
|  【STEP 5】処理完了の確認と運用状態の監視                          |
|                                                                    |
+-------------------------------------------------------------------+
```

**役割別の業務概要：**

| 役割 | 主な業務 |
|------|---------|
| **クラスタ管理者** | クラスタのセットアップ・アップグレード、ノード管理、RBAC設定、リソースクォータ設定、証明書管理、クラスタ全体の監視 |
| **オペレーター（開発者/運用者）** | アプリケーションのデプロイ・更新、スケーリング、ログ確認、デバッグ、ファイルコピー、ポートフォワード |

### 1.4 主な機能

| 機能 | 説明 | 業務フローでの位置 |
|------|------|------------------|
| クラスタセットアップ（kubeadm） | クラスタの初期化、ノード参加、証明書管理 | STEP 1 |
| リソース作成・適用（create/apply） | マニフェストからKubernetesリソースを作成・更新 | STEP 2-3 |
| リソース監視・確認（get/describe） | リソースの一覧取得、詳細情報の表示 | STEP 2, 4 |
| デプロイ管理（rollout/scale） | ロールアウト制御、スケーリング、オートスケール設定 | STEP 3-4 |
| ノード管理（cordon/drain/taint） | ノードのメンテナンス、スケジュール制御 | STEP 4 |
| デバッグ・トラブルシュート（logs/exec/debug） | コンテナログの確認、コマンド実行、デバッグ | 随時利用 |
| 設定管理（config） | kubeconfig設定の管理、コンテキスト切り替え | 随時利用 |
| ヘルプ・リファレンス（explain/api-resources） | リソースフィールドの説明、API情報の参照 | 随時参照 |

### 1.5 ユーザー種別

システムには以下のユーザー種別があります：

| ユーザー種別 | 説明 |
|-------------|------|
| クラスタ管理者（cluster-admin） | クラスタ全体の管理権限を持つユーザー。kubeadmの操作、RBAC設定、ノード管理など全操作が可能 |
| 名前空間管理者（namespace-admin） | 特定のNamespace内のリソースを管理する権限を持つユーザー |
| オペレーター（developer/operator） | アプリケーションのデプロイ・運用を行うユーザー。割り当てられたNamespace内でリソースの作成・更新・削除が可能 |
| 閲覧者（viewer） | リソースの参照のみ可能なユーザー。get/describe/logsなどの読み取り操作のみ |

### 1.6 対応環境

推奨環境：
- ターミナル（bash / zsh / PowerShell）
- kubectl バイナリ（クラスタバージョンと互換性のあるバージョン）
- kubeadm バイナリ（クラスタセットアップ時のみ）
- kubeconfigファイル（クラスタへの接続情報）

対応OS：
- Linux（amd64 / arm64）
- macOS（amd64 / arm64）
- Windows（amd64）

---

## 2. ログイン・ログアウト手順（クラスタ認証）

### 2.1 クラスタへの接続方法（ログイン相当）

Kubernetesではブラウザによるログインではなく、kubeconfigファイルを使用してクラスタに接続します。

1. 管理者から提供されたkubeconfigファイルを所定のディレクトリに配置します

```bash
mkdir -p ~/.kube
cp <提供されたkubeconfig> ~/.kube/config
```

2. 接続テストを行います

```bash
kubectl cluster-info
```

3. 正常に接続できると、以下のような出力が表示されます

```
Kubernetes control plane is running at https://<API-SERVER-ADDRESS>:6443
CoreDNS is running at https://<API-SERVER-ADDRESS>:6443/api/v1/namespaces/kube-system/services/kube-dns:dns/proxy
```

**接続に必要な情報：**

| 項目 | 説明 |
|------|---------|
| kubeconfigファイル | クラスタのAPIサーバーアドレス、認証情報を含む設定ファイル |
| コンテキスト名 | 接続先クラスタを識別する名前 |
| 認証方式 | クライアント証明書、トークン、OIDC等のいずれか |

### 2.2 クラスタからの切断方法（ログアウト相当）

Kubernetesには明示的なログアウト操作はありませんが、以下の方法でセッションを終了できます。

1. 現在のコンテキストを解除する場合：

```bash
kubectl config unset current-context
```

2. kubeconfigから特定のコンテキストを削除する場合：

```bash
kubectl config delete-context <コンテキスト名>
```

### 2.3 認証情報を確認したい場合

現在の認証情報やアクセス権限を確認するには：

```bash
# 現在のユーザー情報を確認
kubectl auth whoami

# 特定の操作権限を確認
kubectl auth can-i <操作> <リソース>
# 例: kubectl auth can-i create deployments
```

---

## 3. トップページ（初期操作画面）

### 3.1 オペレーター向け初期操作

ターミナルを起動し、クラスタに接続した後、以下のコマンドで現在の状態を把握します。

**クラスタ状態の確認：**

```bash
# 接続先クラスタ情報を確認
kubectl cluster-info

# 現在のコンテキストを確認
kubectl config current-context

# 利用可能なNamespaceを確認
kubectl get namespaces
```

**自分のリソース一覧の確認：**

```bash
# デフォルトNamespaceのPod一覧
kubectl get pods

# 特定のNamespaceのリソース一覧
kubectl get all -n <Namespace名>

# 全Namespaceのリソース一覧
kubectl get all --all-namespaces
```

### 3.2 クラスタ管理者向け初期操作

管理者はクラスタ全体の状態を把握するため、以下のコマンドを使用します。

```bash
# ノード一覧と状態の確認
kubectl get nodes

# ノードのリソース使用量を確認
kubectl top nodes

# 全Namespaceのイベントを確認
kubectl events --all-namespaces

# クラスタのバージョン情報を確認
kubectl version
```

---

## 4. 【STEP1】案内・通知機能（イベント・通知の配信・受信）

> **業務フローでの位置**: 管理者がクラスタの構成情報やイベントを配信し、オペレーターがそれを受信・確認するステップです。Kubernetesではイベントリソースやノードステータスが通知の役割を果たします。

### 4.1 イベント・通知を確認する（オペレーター向け）

Kubernetesのイベントは、リソースの状態変化（Pod起動、エラー発生、スケジューリング完了など）を通知するメカニズムです。

**イベント一覧の確認：**

```bash
# デフォルトNamespaceのイベントを確認
kubectl events

# 特定Namespaceのイベントを確認
kubectl events -n <Namespace名>

# 全Namespaceのイベントを確認
kubectl events --all-namespaces

# Warning（警告）イベントのみ表示
kubectl events --types=Warning
```

**特定リソースのイベント確認：**

```bash
# 特定のPodに関連するイベントを確認
kubectl describe pod <Pod名>
# 出力の「Events:」セクションにイベント情報が表示されます
```

### 4.2 イベント・通知の詳細

Kubernetesが発行する主要なイベント（通知）の種類は以下の通りです：

| カテゴリ | イベント名 | 説明 | 重要度 |
|---------|-----------|------|--------|
| コンテナ | Created | コンテナが正常に作成された | Normal |
| コンテナ | Started | コンテナが正常に起動した | Normal |
| コンテナ | BackOff | コンテナ起動のバックオフが発生した | Warning |
| コンテナ | Killing | コンテナが強制終了される | Normal |
| イメージ | Pulling | コンテナイメージのプルを開始した | Normal |
| イメージ | Pulled | コンテナイメージのプルが完了した | Normal |
| イメージ | Failed (PullImage) | コンテナイメージのプルに失敗した | Warning |
| ノード | NodeReady | ノードがReady状態になった | Normal |
| ノード | NodeNotReady | ノードがNotReady状態になった | Warning |
| スケジューリング | Scheduled | Podがノードにスケジュールされた | Normal |
| スケジューリング | FailedScheduling | Podのスケジューリングに失敗した | Warning |
| Deployment | ScalingReplicaSet | ReplicaSetのスケーリングが実行された | Normal |
| HPA | SuccessfulRescale | HPAによるスケール変更が成功した | Normal |
| エビクション | EvictionThresholdMet | リソース使用量がエビクション閾値に到達した | Warning |

> **注意**: Warningイベントが発生した場合は、速やかに内容を確認し、必要に応じて対処してください。

### 4.3 クラスタ情報の確認（管理者向け）

管理者はクラスタ全体のステータスを定期的に確認し、問題を早期に検知します。

**ノード状態の監視：**

```bash
# ノード一覧と状態を確認
kubectl get nodes -o wide

# 特定ノードの詳細情報を確認
kubectl describe node <ノード名>

# ノードのリソース使用量を確認
kubectl top node
```

**クラスタコンポーネントの確認：**

```bash
# APIサーバーのバージョンを確認
kubectl version

# 利用可能なAPIバージョンを確認
kubectl api-versions

# 利用可能なAPIリソースを確認
kubectl api-resources
```

---

## 5. 【STEP2-3】依頼・届出機能（リソースの定義・適用）

> **業務フローでの位置**: オペレーターがアプリケーションのデプロイ要件をマニフェストとして定義し、クラスタに適用するステップです。

### 5.1 リソース操作の種類

Kubernetesで作成・管理できる主要なリソースの種類は以下の通りです：

| リソース種別 | 説明 | 主なユースケース |
|-------------|------|-----------------|
| Deployment | アプリケーションの宣言的なデプロイ管理 | Webアプリケーション、APIサーバー等のステートレスアプリ |
| StatefulSet | ステートフルアプリケーションのデプロイ管理 | データベース、メッセージキュー等 |
| DaemonSet | 全ノードに1つずつPodを配置 | ログ収集エージェント、監視エージェント等 |
| Job | 一度きりのバッチ処理 | データ移行、バッチ処理等 |
| CronJob | 定期実行するバッチ処理 | 定期バックアップ、レポート生成等 |
| Service | ネットワークアクセスの公開 | アプリケーションの外部公開、内部通信 |
| ConfigMap | 設定データの管理 | アプリケーション設定の外部化 |
| Secret | 機密データの管理 | パスワード、APIキー等の管理 |
| Namespace | リソースの論理的な分離 | 環境分離（開発/ステージング/本番） |

### 5.2 リソース作成の基本的な流れ

1. マニフェストファイル（YAML）を作成します
2. 「kubectl apply」コマンドでマニフェストをクラスタに適用します
3. 「kubectl get」コマンドでリソースが正常に作成されたか確認します
4. 「kubectl describe」コマンドで詳細な状態を確認します
5. 問題がある場合は「kubectl logs」や「kubectl events」で原因を調査します

### 5.3 アプリケーションのデプロイ手順

**方法1: マニフェストファイルを使用した宣言的適用（推奨）**

```bash
# マニフェストファイルを適用
kubectl apply -f deployment.yaml

# 適用前に差分を確認（ドライラン）
kubectl diff -f deployment.yaml

# ディレクトリ内の全マニフェストを適用
kubectl apply -f ./manifests/

# Kustomizeを使用した適用
kubectl apply -k ./overlays/production/
```

**方法2: コマンドラインからの直接作成**

```bash
# Deploymentを作成
kubectl create deployment <名前> --image=<イメージ名>

# Serviceを作成（Deploymentを公開）
kubectl expose deployment <名前> --port=<ポート番号> --type=<タイプ>

# Podを直接実行
kubectl run <名前> --image=<イメージ名>

# Namespaceを作成
kubectl create namespace <Namespace名>

# ConfigMapを作成
kubectl create configmap <名前> --from-file=<ファイルパス>

# Secretを作成
kubectl create secret generic <名前> --from-literal=<キー>=<値>

# CronJobを作成
kubectl create cronjob <名前> --image=<イメージ名> --schedule="*/5 * * * *"
```

**方法3: リソースの更新**

```bash
# マニフェストでリソースを置換
kubectl replace -f deployment.yaml

# リソースのフィールドを部分更新（パッチ）
kubectl patch deployment <名前> -p '{"spec":{"replicas":3}}'

# コンテナイメージを更新
kubectl set image deployment/<名前> <コンテナ名>=<新イメージ>

# 環境変数を設定
kubectl set env deployment/<名前> KEY=VALUE

# リソースをエディタで編集
kubectl edit deployment <名前>
```

### 5.4 リソースの状態確認

リソースの状態を確認し、正常にデプロイされたか検証します。

```bash
# リソース一覧の確認
kubectl get <リソースタイプ>
# 例: kubectl get deployments
# 例: kubectl get pods
# 例: kubectl get services

# 詳細情報の表示
kubectl describe <リソースタイプ> <リソース名>
# 例: kubectl describe deployment my-app

# リソースの条件待機
kubectl wait --for=condition=available deployment/<名前> --timeout=120s

# Pod一覧（詳細表示）
kubectl get pods -o wide
```

---

## 6. 【STEP3-4】ファイル共有機能（マニフェスト・設定の管理）

> **業務フローでの位置**: オペレーターがマニフェストファイルや設定ファイルをPodとの間でやり取りし、管理者がそれを確認するステップです。

### 6.1 ファイルコピー（Pod間のファイル転送）

Podとローカルマシンとのファイルのやり取りには「kubectl cp」を使用します。

**ローカルからPodへのコピー：**

```bash
kubectl cp <ローカルファイルパス> <Pod名>:<コンテナ内パス>
# 例: kubectl cp ./config.yaml my-pod:/etc/config/config.yaml

# 特定のNamespaceのPodにコピー
kubectl cp <ローカルファイルパス> <Namespace>/<Pod名>:<コンテナ内パス>
```

**Podからローカルへのコピー：**

```bash
kubectl cp <Pod名>:<コンテナ内パス> <ローカルファイルパス>
# 例: kubectl cp my-pod:/var/log/app.log ./app.log
```

### 6.2 ConfigMap / Secretによる設定管理

アプリケーションの設定ファイルはConfigMapおよびSecretリソースとして管理します。

**ConfigMapの作成と確認：**

```bash
# ファイルからConfigMapを作成
kubectl create configmap <名前> --from-file=<ファイルパス>

# リテラル値からConfigMapを作成
kubectl create configmap <名前> --from-literal=key1=value1

# ConfigMapの内容を確認
kubectl get configmap <名前> -o yaml

# ConfigMapの一覧を確認
kubectl get configmaps
```

**Secretの作成と確認：**

```bash
# リテラル値からSecretを作成
kubectl create secret generic <名前> --from-literal=password=<パスワード>

# ファイルからSecretを作成
kubectl create secret generic <名前> --from-file=<ファイルパス>

# Secretの一覧を確認（値はBase64エンコードされています）
kubectl get secret <名前> -o yaml
```

### 6.3 ラベル・アノテーションによるメタデータ管理

リソースにメタデータを付与して管理します。

```bash
# ラベルを追加
kubectl label <リソースタイプ> <名前> <キー>=<値>
# 例: kubectl label pod my-pod env=production

# ラベルを変更（上書き）
kubectl label <リソースタイプ> <名前> <キー>=<新しい値> --overwrite

# ラベルを削除
kubectl label <リソースタイプ> <名前> <キー>-

# アノテーションを追加
kubectl annotate <リソースタイプ> <名前> <キー>=<値>

# アノテーションを削除
kubectl annotate <リソースタイプ> <名前> <キー>-
```

---

## 7. 【STEP4-5】管理・処理完了（クラスタ管理者向け）

> **業務フローでの位置**: 管理者がオペレーターが適用したリソースの状態を確認・処理し、クラスタの安定稼働を確認するステップです。

### 7.1 デプロイメントの管理

**ロールアウト管理：**

```bash
# ロールアウト状態の確認
kubectl rollout status deployment/<名前>

# ロールアウト履歴の確認
kubectl rollout history deployment/<名前>

# ロールバック（前のバージョンに戻す）
kubectl rollout undo deployment/<名前>

# 特定リビジョンにロールバック
kubectl rollout undo deployment/<名前> --to-revision=<リビジョン番号>

# ロールアウトの一時停止
kubectl rollout pause deployment/<名前>

# ロールアウトの再開
kubectl rollout resume deployment/<名前>

# ロールアウトの再起動
kubectl rollout restart deployment/<名前>
```

**スケーリング：**

```bash
# レプリカ数を手動で変更
kubectl scale deployment/<名前> --replicas=<数>

# HPA（Horizontal Pod Autoscaler）を設定
kubectl autoscale deployment/<名前> --min=<最小> --max=<最大> --cpu-percent=<閾値>

# HPAの状態を確認
kubectl get hpa
```

### 7.2 ノード管理

**ノードのメンテナンス：**

```bash
# ノードをスケジュール不可に設定（新しいPodの配置を停止）
kubectl cordon <ノード名>

# ノードからPodを退避（メンテナンス準備）
kubectl drain <ノード名> --ignore-daemonsets --delete-emptydir-data

# メンテナンス完了後、ノードをスケジュール可能に戻す
kubectl uncordon <ノード名>
```

**Taint管理：**

```bash
# ノードにTaintを追加
kubectl taint nodes <ノード名> <キー>=<値>:<効果>
# 例: kubectl taint nodes node1 maintenance=true:NoSchedule

# Taintを削除
kubectl taint nodes <ノード名> <キー>-
# 例: kubectl taint nodes node1 maintenance-
```

### 7.3 リソースの削除

```bash
# リソースを削除
kubectl delete <リソースタイプ> <名前>
# 例: kubectl delete deployment my-app

# マニフェストファイルを使用して削除
kubectl delete -f deployment.yaml

# Namespace内の全リソースを削除
kubectl delete all --all -n <Namespace名>

# ラベルセレクタで一括削除
kubectl delete pods -l app=my-app
```

### 7.4 アクセス制御の管理（RBAC）

```bash
# Roleの作成
kubectl create role <ロール名> --verb=get,list,watch --resource=pods -n <Namespace名>

# RoleBindingの作成
kubectl create rolebinding <バインディング名> --role=<ロール名> --user=<ユーザー名> -n <Namespace名>

# ClusterRoleの作成
kubectl create clusterrole <ロール名> --verb=get,list,watch --resource=nodes

# ClusterRoleBindingの作成
kubectl create clusterrolebinding <バインディング名> --clusterrole=<ロール名> --user=<ユーザー名>

# 権限の確認
kubectl auth can-i <操作> <リソース> --as <ユーザー名>
# 例: kubectl auth can-i create deployments --as developer1

# RBAC設定の整合性チェック
kubectl auth reconcile -f rbac-config.yaml
```

### 7.5 証明書管理

```bash
# CertificateSigningRequestの一覧を確認
kubectl get csr

# CSRを承認
kubectl certificate approve <CSR名>

# CSRを拒否
kubectl certificate deny <CSR名>
```

### 7.6 クラスタセットアップ（kubeadm）

**クラスタの初期化：**

```bash
# コントロールプレーンノードを初期化
kubeadm init --pod-network-cidr=<CIDR>

# kubeconfigを設定
mkdir -p $HOME/.kube
cp -i /etc/kubernetes/admin.conf $HOME/.kube/config
```

**ノードの参加：**

```bash
# ワーカーノードをクラスタに参加させる
kubeadm join <APIサーバーアドレス>:<ポート> --token <トークン> --discovery-token-ca-cert-hash sha256:<ハッシュ>

# 参加用トークンの作成
kubeadm token create --print-join-command

# トークンの一覧を確認
kubeadm token list
```

**クラスタのアップグレード：**

```bash
# アップグレードプランの確認
kubeadm upgrade plan

# コントロールプレーンのアップグレード実行
kubeadm upgrade apply v<バージョン>
```

**クラスタのリセット：**

```bash
# kubeadm init / joinの変更を元に戻す
kubeadm reset
```

---

## 8. FAQ機能（ヘルプ・リファレンス）

### 8.1 APIリソースのドキュメントを参照する

kubectlに組み込まれたヘルプ機能で、リソースの仕様を確認できます。

```bash
# リソースのフィールドドキュメントを表示
kubectl explain <リソースタイプ>
# 例: kubectl explain deployment

# 特定フィールドの詳細を表示
kubectl explain <リソースタイプ>.<フィールド>
# 例: kubectl explain deployment.spec.replicas

# 再帰的に全フィールドを表示
kubectl explain <リソースタイプ> --recursive
```

### 8.2 利用可能なAPIリソースを確認する

```bash
# 利用可能なAPIリソースタイプの一覧
kubectl api-resources

# 特定のAPIグループのリソースを表示
kubectl api-resources --api-group=apps

# Namespaced/非Namespacedリソースの区別
kubectl api-resources --namespaced=true
kubectl api-resources --namespaced=false

# サポートされるAPIバージョンの一覧
kubectl api-versions
```

### 8.3 コマンドのヘルプを確認する

```bash
# kubectlのヘルプ
kubectl --help

# 特定コマンドのヘルプ
kubectl <コマンド> --help
# 例: kubectl create --help
# 例: kubectl apply --help

# 全コマンド共通のフラグ一覧
kubectl options

# kubeadmのヘルプ
kubeadm --help
```

---

## 9. お問合せ機能（トラブルシューティング・デバッグ）

### 9.1 ログの確認

問題が発生した際に、まずコンテナのログを確認します。

```bash
# Podのログを表示
kubectl logs <Pod名>

# 特定コンテナのログを表示
kubectl logs <Pod名> -c <コンテナ名>

# ログをリアルタイムで追跡
kubectl logs <Pod名> -f

# 前回の終了コンテナのログを表示
kubectl logs <Pod名> --previous

# 直近1時間のログを表示
kubectl logs <Pod名> --since=1h

# ラベルセレクタで複数Podのログを表示
kubectl logs -l app=my-app
```

### 9.2 コンテナ内でのコマンド実行

```bash
# コンテナ内でコマンドを実行
kubectl exec <Pod名> -- <コマンド>
# 例: kubectl exec my-pod -- ls /app

# コンテナ内でインタラクティブシェルを起動
kubectl exec -it <Pod名> -- /bin/bash
# または
kubectl exec -it <Pod名> -- /bin/sh

# 特定コンテナでコマンドを実行
kubectl exec <Pod名> -c <コンテナ名> -- <コマンド>
```

### 9.3 デバッグ機能

```bash
# エフェメラルコンテナでデバッグ
kubectl debug <Pod名> --image=<デバッグイメージ> -it

# Pod をコピーしてデバッグ
kubectl debug <Pod名> --copy-to=debug-pod --image=<デバッグイメージ> -it

# ノードレベルのデバッグ
kubectl debug node/<ノード名> --image=<デバッグイメージ> -it
```

### 9.4 ポートフォワード

ローカルマシンからPodへのネットワーク接続を確立します。

```bash
# ポートフォワードを設定
kubectl port-forward <Pod名> <ローカルポート>:<コンテナポート>
# 例: kubectl port-forward my-pod 8080:80

# Serviceに対するポートフォワード
kubectl port-forward service/<Service名> <ローカルポート>:<Serviceポート>
```

### 9.5 プロキシ

Kubernetes APIサーバーへのローカルプロキシを起動します。

```bash
# APIプロキシを起動
kubectl proxy --port=8001

# 特定のアドレスでリッスン
kubectl proxy --address=0.0.0.0 --accept-hosts='.*'
```

### 9.6 コンテナへのアタッチ

```bash
# 実行中のコンテナの標準入出力にアタッチ
kubectl attach <Pod名> -it
```

---

## 10. マイページ機能（kubeconfig設定管理）

### 10.1 kubeconfig設定の確認

```bash
# 現在の設定を表示
kubectl config view

# 現在のコンテキストを確認
kubectl config current-context

# 利用可能なコンテキスト一覧
kubectl config get-contexts

# 利用可能なクラスター一覧
kubectl config get-clusters

# 利用可能なユーザー一覧
kubectl config get-users
```

### 10.2 コンテキストの切り替え

複数のクラスタを管理する場合、コンテキストを切り替えて操作します。

```bash
# コンテキストを切り替え
kubectl config use-context <コンテキスト名>

# 新しいコンテキストを設定
kubectl config set-context <コンテキスト名> \
  --cluster=<クラスタ名> \
  --user=<ユーザー名> \
  --namespace=<デフォルトNamespace>

# コンテキストの名前を変更
kubectl config rename-context <旧名> <新名>

# コンテキストを削除
kubectl config delete-context <コンテキスト名>
```

### 10.3 クラスタ接続設定の管理

```bash
# クラスタ情報を設定
kubectl config set-cluster <クラスタ名> \
  --server=https://<APIサーバーアドレス>:6443 \
  --certificate-authority=<CA証明書パス>

# ユーザー認証情報を設定
kubectl config set-credentials <ユーザー名> \
  --client-certificate=<証明書パス> \
  --client-key=<秘密鍵パス>

# クラスタ情報を削除
kubectl config delete-cluster <クラスタ名>

# ユーザー情報を削除
kubectl config delete-user <ユーザー名>

# 設定値を解除
kubectl config unset <キー>
```

### 10.4 シェル補完の設定

kubectlコマンドの入力を効率化するため、シェル補完を設定します。

```bash
# bash補完の設定
kubectl completion bash > ~/.kubectl-completion.bash
echo 'source ~/.kubectl-completion.bash' >> ~/.bashrc

# zsh補完の設定
kubectl completion zsh > ~/.kubectl-completion.zsh
echo 'source ~/.kubectl-completion.zsh' >> ~/.zshrc

# kubeadm補完の設定（bash）
kubeadm completion bash > ~/.kubeadm-completion.bash
echo 'source ~/.kubeadm-completion.bash' >> ~/.bashrc
```

### 10.5 kuberc設定（エイリアス・デフォルト設定）

kubercファイルを使用して、kubectlのエイリアスやデフォルト設定をカスタマイズできます（feature gate有効時のみ利用可能）。

```bash
# kuberc設定の管理
kubectl kuberc
```

### 10.6 プラグイン管理

kubectlの機能をプラグインで拡張できます。

```bash
# インストール済みプラグインの一覧を確認
kubectl plugin list
```

---

## 11. 日常業務の流れ

本セクションでは、業務フローに沿った日常業務の流れを説明します。

```
業務フロー概要（再掲）
=====================================================================
【STEP 1】管理者がクラスタをセットアップし、通知・設定を配信 -> 4章・7章参照
    |
【STEP 2】オペレーターが設定・通知を確認し、要件を把握 -> 4章参照
    |
【STEP 3】オペレーターがマニフェストを作成し、リソースを適用 -> 5章参照
    |
【STEP 4】管理者がリソースの状態を確認・処理 -> 7章参照
    |
【STEP 5】処理完了の確認と運用状態の監視 -> 7章参照
=====================================================================
```

### 11.1 【管理者】STEP1: クラスタのセットアップと設定配信

業務開始時に、クラスタの環境を整備し、オペレーターに必要な設定を配信します。

**やるべきこと：**
- クラスタの健全性を確認する
- 必要なNamespaceを作成する
- RBAC設定を行い、オペレーターに権限を付与する
- ResourceQuotaを設定する
- kubeconfigファイルをオペレーターに配布する

**操作手順：**

```bash
# 1. クラスタの健全性を確認
kubectl get nodes
kubectl get pods -n kube-system

# 2. Namespaceを作成
kubectl create namespace <Namespace名>

# 3. ResourceQuotaを設定
kubectl apply -f resource-quota.yaml

# 4. RBAC設定を適用
kubectl apply -f rbac-config.yaml

# 5. kubeconfig生成（kubeadm使用時）
kubeadm kubeconfig user --client-name=<ユーザー名>
```

### 11.2 【オペレーター】STEP2: 設定の確認と要件把握

管理者から配布された情報を確認し、デプロイの準備を行います。

**やるべきこと：**
- kubeconfigを設定してクラスタに接続する
- 自分に割り当てられたNamespaceと権限を確認する
- クラスタのイベントや通知を確認する

**操作手順：**

```bash
# 1. クラスタへの接続確認
kubectl cluster-info
kubectl auth whoami

# 2. 権限の確認
kubectl auth can-i --list

# 3. 割り当てNamespaceの確認
kubectl get namespaces

# 4. イベントの確認
kubectl events -n <Namespace名>
```

### 11.3 【オペレーター】STEP3: マニフェストの作成とリソース適用

要件に基づいてマニフェストを作成し、クラスタに適用します。

**やるべきこと：**
- マニフェストファイル（YAML）を作成する
- 適用前に差分を確認する
- リソースをクラスタに適用する
- 適用結果を確認する

**操作手順：**

```bash
# 1. 適用前の差分確認
kubectl diff -f manifests/

# 2. リソースの適用
kubectl apply -f manifests/

# 3. デプロイ状態の確認
kubectl rollout status deployment/<名前>

# 4. Pod状態の確認
kubectl get pods -o wide

# 5. 問題がある場合はログを確認
kubectl logs <Pod名>
kubectl describe pod <Pod名>
```

### 11.4 【管理者】STEP4-5: リソースの確認・処理・完了通知

オペレーターが適用したリソースの状態を確認し、問題があれば対処します。

**やるべきこと：**
- 全Namespaceのリソース状態を確認する
- 問題のあるPodを特定し対処する
- 必要に応じてスケーリングやロールバックを実施する
- ノードのリソース使用量を監視する

**操作手順：**

```bash
# 1. 全体のリソース状態確認
kubectl get pods --all-namespaces | grep -v Running

# 2. リソース使用量の確認
kubectl top nodes
kubectl top pods --all-namespaces

# 3. 問題のあるPodの調査
kubectl describe pod <Pod名> -n <Namespace名>
kubectl logs <Pod名> -n <Namespace名>

# 4. 必要に応じてスケーリング
kubectl scale deployment/<名前> --replicas=<数> -n <Namespace名>

# 5. 必要に応じてロールバック
kubectl rollout undo deployment/<名前> -n <Namespace名>
```

---

## 12. よくある質問と対処法

### 12.1 接続に関するQ&A

**Q: kubectlコマンドを実行すると「Unable to connect to the server」エラーが出ます**

A: 以下を確認してください：
1. kubeconfigファイルが正しく配置されているか確認する（デフォルト: `~/.kube/config`）
2. APIサーバーのアドレスが正しいか確認する（`kubectl config view` で確認）
3. ネットワーク接続が正常か確認する（APIサーバーへのping等）
4. 証明書の有効期限が切れていないか確認する

**Q: 「error: You must be logged in to the server (Unauthorized)」エラーが出ます**

A: 以下を確認してください：
1. 認証情報（トークン/証明書）が有効か確認する
2. 管理者に認証情報の再発行を依頼する
3. `kubectl config view` で現在の設定を確認する

### 12.2 リソース作成に関するQ&A

**Q: 「kubectl apply」を実行しても変更が反映されません**

A: 以下を確認してください：
1. `kubectl diff -f <ファイル>` で差分を確認する
2. マニフェストの文法に誤りがないか確認する
3. リソースのステータスを `kubectl describe` で確認する
4. イベントを `kubectl events` で確認する

**Q: Podが「Pending」状態のまま起動しません**

A: 以下を確認してください：
1. `kubectl describe pod <Pod名>` でイベントを確認する
2. ノードのリソース（CPU/メモリ）が十分か確認する（`kubectl top nodes`）
3. PersistentVolumeClaimがバインドされているか確認する
4. NodeSelectorやAffinityの設定が正しいか確認する

**Q: Podが「CrashLoopBackOff」状態になります**

A: 以下を確認してください：
1. `kubectl logs <Pod名>` でアプリケーションのエラーログを確認する
2. `kubectl logs <Pod名> --previous` で前回の終了時のログを確認する
3. コンテナイメージが正しいか確認する
4. コマンドや引数の設定が正しいか確認する
5. リソース制限（メモリ等）が適切か確認する

### 12.3 権限に関するQ&A

**Q: 「forbidden」エラーが出て操作できません**

A: 以下を確認してください：
1. `kubectl auth can-i <操作> <リソース>` で権限を確認する
2. 正しいNamespaceで操作しているか確認する（`-n <Namespace名>` を指定）
3. 管理者に必要な権限の付与を依頼する

---

## 13. トラブルシューティング

### 13.1 エラーメッセージと対処法

| エラーメッセージ | 原因 | 対処法 |
|----------------|------|--------|
| `Unable to connect to the server` | APIサーバーへの接続失敗 | ネットワーク接続とkubeconfig設定を確認する |
| `error: You must be logged in to the server (Unauthorized)` | 認証失敗 | 認証情報（トークン/証明書）を確認・再発行する |
| `Error from server (Forbidden)` | 権限不足 | 管理者にRBAC権限の付与を依頼する |
| `Error from server (NotFound)` | リソースが存在しない | リソース名とNamespaceを確認する |
| `Error from server (AlreadyExists)` | リソースが既に存在する | 既存リソースを削除または更新する |
| `Error from server (Conflict)` | リソースのバージョン競合 | 最新の状態を取得してから再度操作する |
| `error: unable to recognize "file.yaml"` | マニフェストの文法エラー | YAMLファイルの文法を確認する |
| `ImagePullBackOff` | コンテナイメージの取得失敗 | イメージ名、レジストリ接続、認証情報を確認する |
| `CrashLoopBackOff` | コンテナが繰り返しクラッシュ | `kubectl logs` でエラーを確認する |
| `OOMKilled` | メモリ不足によるコンテナ強制終了 | メモリリソース制限を増やす |
| `FailedScheduling` | スケジューリング失敗 | ノードのリソース状況とセレクタ設定を確認する |
| `FailedMount` | ボリュームのマウント失敗 | PVCの状態とストレージ設定を確認する |

### 13.2 よくあるトラブルと解決方法

**トラブル1: ノードがNotReady状態になった**

```bash
# 1. ノードの詳細を確認
kubectl describe node <ノード名>

# 2. ノード上のkubeletの状態を確認（ノードにSSH接続後）
systemctl status kubelet

# 3. kubeletのログを確認
journalctl -u kubelet -f
```

**トラブル2: Deploymentの更新が完了しない**

```bash
# 1. ロールアウト状態を確認
kubectl rollout status deployment/<名前>

# 2. ReplicaSetの状態を確認
kubectl get replicasets

# 3. 問題がある場合はロールバック
kubectl rollout undo deployment/<名前>
```

**トラブル3: Serviceにアクセスできない**

```bash
# 1. Serviceの状態を確認
kubectl get service <Service名>
kubectl describe service <Service名>

# 2. エンドポイントの確認
kubectl get endpoints <Service名>

# 3. バックエンドPodの状態を確認
kubectl get pods -l <セレクタ>

# 4. ポートフォワードでの接続テスト
kubectl port-forward service/<Service名> <ローカルポート>:<Serviceポート>
```

**トラブル4: PersistentVolumeClaimが「Pending」のまま**

```bash
# 1. PVCの状態を確認
kubectl describe pvc <PVC名>

# 2. 利用可能なPVを確認
kubectl get pv

# 3. StorageClassを確認
kubectl get storageclass
```

### 13.3 サポートへの連絡方法

問題が解決しない場合は、以下の情報を添えてクラスタ管理者に連絡してください：

1. 実行したコマンドと出力結果
2. `kubectl describe <リソースタイプ> <名前>` の出力
3. `kubectl logs <Pod名>` の出力（該当する場合）
4. `kubectl events` の出力
5. 使用しているkubectlのバージョン（`kubectl version --client`）

---

## 改訂履歴

| 版数 | 改訂日 | 改訂内容 |
|-----|--------|---------|
| 1.0 | 2026-02-02 | 初版作成 |

---

*本マニュアルは、Kubernetesのエンドユーザー向け業務運用マニュアルです。*
*システムの更新により、コマンドや操作方法が変更される場合があります。*
