# 非機能要件定義書

## 概要

本ドキュメントは、Kubernetesプロジェクト（kubernetes-master）のソースコードから抽出した非機能要件を定義するものである。Kubernetesは大規模な分散コンテナオーケストレーションシステムであり、kube-apiserver、kube-controller-manager、kube-scheduler、kubelet、kube-proxyなどの主要コンポーネントで構成される。本書では、コードベースに実装されている性能、可用性、セキュリティ、拡張性、保守性、運用・監視に関する要件を記載する。

## 性能要件

### レスポンスタイム

| 処理種別 | 目標値 | 備考 |
| --- | --- | --- |
| API応答（通常リクエスト） | RequestTimeout: 2分（デフォルト） | `cmd/kube-apiserver/app/options/options_test.go` にて `--request-timeout=2m` として設定。GenericServerRunOptions.RequestTimeoutで制御 |
| Kubelet HTTP通信 | 5秒以内 | `cmd/kube-apiserver/app/options/options.go` にて `HTTPTimeout: time.Duration(5) * time.Second` として定義 |
| etcdダイヤルタイムアウト | 20秒 | `cmd/kube-apiserver/app/testing/testserver.go` にて `DialTimeout: 20 * time.Second` として設定 |
| etcdヘルスチェックタイムアウト | 2秒（デフォルト） | `staging/src/k8s.io/apiserver/pkg/storage/storagebackend/config.go` にて `DefaultHealthcheckTimeout = 2 * time.Second` |
| etcdレディネスチェックタイムアウト | 2秒（デフォルト） | `staging/src/k8s.io/apiserver/pkg/storage/storagebackend/config.go` にて `DefaultReadinessTimeout = 2 * time.Second` |
| MinRequestTimeout | 1800秒（30分） | 長時間Watch系リクエスト用の最小タイムアウト |
| ストレージ初期化タイムアウト | 1分 | `StorageInitializationTimeout: time.Minute` |

### スループット

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 最大同時読み取りリクエスト数（MaxRequestsInFlight） | 400（デフォルト）/ 大規模クラスタ: 3000 | `options_test.go` にてデフォルト400。`cluster/gce/gci/configure-kubeapiserver.sh` にて大規模クラスタ向けに3000に設定 |
| 最大同時変更リクエスト数（MaxMutatingRequestsInFlight） | 200（デフォルト）/ 大規模クラスタ: 1000 | デフォルト総並行処理数は600（400+200）。大規模クラスタは1000 |
| テスト用クライアントQPS | 1000 QPS | `cmd/kube-apiserver/app/testing/testserver.go` にて `QPS = 1000` |
| テスト用クライアントBurst | 10000 | `cmd/kube-apiserver/app/testing/testserver.go` にて `Burst = 10000` |
| Controller Manager API QPS | 50.0（設定可能） | `--kube-api-qps=50.0` で設定。Namespaceコントローラ等は20倍に引き上げ |
| Controller Manager API Burst | 100（設定可能） | `--kube-api-burst=100` で設定。Namespaceコントローラ等は100倍に引き上げ |
| 監査ログバッチスロットルQPS | 49.5 | `--audit-log-batch-throttle-qps=49.5` |
| 監査Webhookバッチスロットル QPS | 43.5 | `--audit-webhook-batch-throttle-qps=43.5` |

### 並行処理（コントローラ同期ワーカー数）

| コントローラ | 設定項目 | 備考 |
| --- | --- | --- |
| ReplicaSetController | ConcurrentRSSyncs | 並行ReplicaSet同期数 |
| DeploymentController | ConcurrentDeploymentSyncs | 並行Deployment同期数 |
| JobController | ConcurrentJobSyncs | 並行Job同期数 |
| DeviceTaintEvictionController | ConcurrentSyncs | デフォルト50（テスト値） |
| ResourceClaimController | ConcurrentSyncs | リソースクレーム並行同期数 |

## 可用性要件

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 稼働率 | コード上の明示的な数値定義なし | Kubernetesは設計上高可用性を前提としており、リーダー選出・冗長構成により可用性を確保 |
| 計画停止時間 | コード上の明示的な数値定義なし | コントローラマネージャのGraceful Shutdownにより制御 |
| 障害復旧時間（RTO） | コード上の明示的な数値定義なし | リーダー選出機構により自動フェイルオーバーが可能 |
| 目標復旧時点（RPO） | コード上の明示的な数値定義なし | etcdへのデータ永続化により、RPOはetcdバックアップ戦略に依存 |

### リーダー選出（高可用性）

- kube-scheduler、kube-controller-managerはリーダー選出（Leader Election）機構を搭載
- `--leader-elect=true`（デフォルト有効）で有効化
- 設定可能パラメータ：`--leader-elect-lease-duration`、`--leader-elect-renew-deadline`、`--leader-elect-retry-period`
- 根拠: `cmd/kube-scheduler/app/config/config.go` にて `LeaderElection *leaderelection.LeaderElectionConfig` として定義

### ノードライフサイクル管理

| 項目 | 値 | 備考 |
| --- | --- | --- |
| NodeMonitorGracePeriod | 50秒（テスト値） | ノードが応答しなくなってからNotReadyと判定するまでの猶予期間 |
| NodeMonitorPeriod | 5秒（テスト値） | ノードの状態を確認するポーリング間隔 |

### ヘルスチェック

Kubernetesコンポーネントは以下のヘルスチェックエンドポイントを提供:
- `/healthz` - 総合ヘルスチェック
- `/readyz` - レディネスチェック
- `/livez` - ライブネスチェック

関連ポート:
| コンポーネント | ポート | 用途 |
| --- | --- | --- |
| Kubelet | 10248 | ヘルスチェック（KubeletHealthzPort） |
| Kubelet | 10250 | メインサーバーポート（KubeletPort） |
| Kubelet | 10255 | 読み取り専用ポート（KubeletReadOnlyPort） |
| Kube Proxy | 10249 | メトリクスサーバー（ProxyStatusPort） |
| Kube Proxy | 10256 | ヘルスチェック（ProxyHealthzPort） |
| Controller Manager | 10257 | ステータスサーバー（KubeControllerManagerPort） |
| Cloud Controller Manager | 10258 | ステータスサーバー（CloudControllerManagerPort） |

## セキュリティ要件

### 認証（Authentication）

Kubernetesは複数の認証方式をサポート:

1. **クライアント証明書認証**: `--kubelet-client-certificate`、`--kubelet-client-key` で設定。TLSクライアント証明書によるmTLS認証
2. **トークンファイル認証**: `TokenFileAuthenticationOptions` にて静的トークンファイルによる認証
3. **Webhook認証**: `WebHookAuthenticationOptions` にてWebhookトークン認証。キャッシュTTL設定可能（`--authentication-token-webhook-cache-ttl=3m`）
4. **OIDC認証**: `OIDCAuthenticationOptions` にてOpenID Connect認証。`UsernameClaim: "sub"`、`SigningAlgs: ["RS256"]` がデフォルト
5. **BootstrapToken認証**: ノード参加時のブートストラップトークン認証
6. **ServiceAccount認証**: サービスアカウントトークンによる認証
7. **匿名認証**: `--anonymous-auth` で制御（デフォルトfalseに設定可能）

### 認可（Authorization）

1. **RBAC**: ロールベースアクセス制御が主要な認可方式（`--authorization-mode=AlwaysDeny,RBAC`）
2. **Webhook認可**: Webhookによる外部認可。キャッシュTTL設定:
   - 認可済みキャッシュ: `--authorization-webhook-cache-authorized-ttl=3m`
   - 未認可キャッシュ: `--authorization-webhook-cache-unauthorized-ttl=1m`
3. **ABACポリシー**: `--authorization-policy-file` で属性ベースアクセス制御

### 通信暗号化

- kubeletへの通信はTLS（mTLS）で保護: `--kubelet-client-certificate`、`--kubelet-client-key`、`--kubelet-certificate-authority`
- etcdとの通信はTLSで保護: `--etcd-keyfile`、`--etcd-certfile`、`--etcd-cafile`

### PodSecurityContext

- `PodSecurityContext`: Pod単位のセキュリティ設定（RunAsUser、RunAsGroup、SELinux、SeccompProfile等）
- `SecurityContext`: コンテナ単位のセキュリティ設定。PodSecurityContextよりも優先

### API Priority and Fairness（フロー制御）

- `FlowSchema` と `PriorityLevelConfiguration` により、APIリクエストの優先度とフェアネスを制御
- `exempt` と `catch-all` のシステムプリセット優先レベルが定義
- `FlowSchemaMaxMatchingPrecedence`: 10000

## 拡張性要件

### 水平スケーリング

- kube-apiserverは複数インスタンスでの水平スケーリングが可能（`MasterCount` パラメータ、デフォルト1）
- Endpoint Reconcilerとして `LeaseEndpointReconcilerType` を使用し、複数apiserverインスタンス間の調整を実現
- コントローラマネージャ、スケジューラはリーダー選出により、アクティブ/スタンバイ構成でのスケーリングに対応

### APIリソースの拡張性

- FlowControl APIにより、ワイルドカード指定（`*`）でのAPIグループ、リソース、Verb、NonResource、Namespace対応が可能
- CRD（Custom Resource Definition）による拡張が前提の設計

### ストレージ拡張性

- etcd3をデフォルトストレージバックエンドとして使用（`--storage-backend=etcd3`）
- WatchCacheの有効化（`EnableWatchCache: true`、`DefaultWatchCacheSize: 100`）により読み取り性能を向上
- eventsリソースはWatchCacheサイズ0（キャッシュ無効）に設定（負荷軽減のため）

## 保守性要件

### コード構成

- コンポーネント別のパッケージ構成: `cmd/`（エントリポイント）、`pkg/`（コアロジック）、`staging/`（外部化予定のライブラリ）、`vendor/`（外部依存）
- `hack/` ディレクトリにビルド・テスト用のユーティリティスクリプトを配置
- `test/` ディレクトリに統合テスト、E2Eテスト、kubemarkテストを配置

### 設定の外部化

- コントローラマネージャのすべての主要パラメータはCLIフラグで外部設定可能
- kube-schedulerはComponentConfigファイルによる宣言的設定をサポート
- 各コンポーネントのデフォルト値は `defaults.go` ファイルに集約

### バージョニングと互換性

- ストレージバージョンマイグレーション（`storagemigration`）機能によりAPIバージョン間のデータ移行をサポート
- `EffectiveVersion` によるバージョン互換性管理
- Feature Gateによる段階的な機能の有効化/無効化制御

## 運用・監視要件

### ロギング

- `klog` パッケージによる構造化ログ出力
- `klog.FlushAndExit(klog.ExitFlushTimeout, 1)` による安全なログフラッシュ後の終了処理
- ログの詳細度（Verbosity）レベル制御: `klog.V(1)` 等で出力レベルを指定

### 監査ログ

- 監査ログの設定:
  - ファイルベース: `--audit-log-path=/var/log`
  - 最大保持日数: `--audit-log-maxage=11`
  - 最大バックアップ数: `--audit-log-maxbackup=12`
  - 最大サイズ: `--audit-log-maxsize=13`（MB）
  - モード: `--audit-log-mode=blocking`
  - バッチサイズ: `--audit-log-batch-max-size=47`
  - バッチバッファサイズ: `--audit-log-batch-buffer-size=46`
- Webhook監査:
  - `--audit-webhook-config-file` による外部Webhook送信
  - 初期バックオフ: `--audit-webhook-initial-backoff=2s`
  - トランケーション対応: `--audit-log-truncate-enabled`

### メトリクス

- Prometheusメトリクスのエクスポート機能
- `MaxRequestsInFlight` によるインフライトリクエスト制限とメトリクス
- 各コントローラにメトリクス収集機能が組み込み（例: `clustertrustbundlepublisher/metrics.go`、`rootcacertpublisher/metrics.go`）

### コントローラシャットダウン制御

- `ControllerShutdownTimeout` によるGraceful Shutdown制御
- コンテキストキャンセル後、指定タイムアウト時間内に全コントローラの終了を待機
- タイムアウト到達時にはログ出力（`shutdownTimeout / 3` 間隔でログ）して強制終了

### リトライ設計

- コントローラの作業キューにはリトライ機構が組み込み
- 証明書コントローラ、Namespaceコントローラ等でリトライ付きリソース削除を実装
- Webhook認証のリトライバックオフ: `DefaultAuthWebhookRetryBackoff()` で設定
- ノードIPAM割り当てにおけるタイムアウト・リトライ制御（`pkg/controller/nodeipam/ipam/timeout.go`）

## 備考

- 本ドキュメントに記載した数値の多くは、テストコードやデフォルト設定値から抽出したものであり、本番環境では運用要件に応じてチューニングが必要である
- Kubernetesはクラスタ規模に応じた設定調整が推奨されており、小規模クラスタと大規模クラスタ（5000ノード以上）では推奨値が大きく異なる
- 本書に記載のないパラメータについても、staging配下のapiserver/client-goライブラリにて詳細な設定が可能である
