# セキュリティ設計書

## 概要

本ドキュメントは、Legacy CMS（バージョン2.1）のセキュリティ設計について記載する。本システムはZend Framework 1.xをベースとしたPHP製CMSであり、認証・認可、データ保護、入出力対策などのセキュリティ機能を実装している。

## 認証設計

### 認証方式

本システムは、Zend Framework の `Zend_Auth` コンポーネントを使用したデータベースベースの認証を採用している。

#### 認証フロー

1. ユーザーがメールアドレスとパスワードを入力
2. `Zend_Auth_Adapter_DbTable` を使用してデータベース認証を実行
3. パスワードは `MD5(CONCAT(site_key, password, user_salt))` 形式でハッシュ化して照合
4. 認証成功時、ユーザー情報をセッションストレージに保存
5. 認証失敗時、エラーコードに応じたメッセージを表示

#### 認証コード（AuthController.php）

```php
$authAdapter = new Zend_Auth_Adapter_DbTable(
    $registry->db,
    'users',
    'user_email',
    'user_password',
    "MD5(CONCAT('". $registry->site->site->key . "', ?, user_salt))"
);
$select = $authAdapter->getDbSelect();
$select->where('user_status = "active"');
```

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション名 | PHPSESSID | 標準PHPセッション名 |
| セッション有効期限 | PHP設定に依存 | 明示的な設定なし |
| セッション固定化対策 | 部分的実装 | Flash対応のためURLからセッションID受付可能 |
| Remember Me機能 | Cookie（91日間） | 平文パスワードをCookieに保存（セキュリティリスク） |

#### セッション関連の懸念事項

- **Flash対応処理**: URLパラメータ `__tkn` からセッションIDを受け付ける実装があり、セッションハイジャックのリスクがある
- **Remember Me Cookie**: 平文パスワードがCookieに保存される（`setcookie('loginPassword', $input->password, ...)`）

### パスワードポリシー

#### パスワードハッシュ

| 項目 | 実装内容 |
| --- | --- |
| ハッシュアルゴリズム | MD5 |
| ソルト方式 | サイトキー + ユーザー固有ソルトの連結 |
| ソルト長 | 8文字（ランダム生成） |

#### パスワード強度チェック

`CMS_Password_Strength` クラスによるパスワード強度評価機能を提供：

- 文字長スコア（最小4文字必須）
- 文字種複雑性スコア（英数字、記号の混合評価）
- 文字セット複雑性スコア

## 認可設計

### 権限体系

本システムは、Zend Framework の `Zend_Acl` を使用したロールベースアクセス制御（RBAC）を実装している。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| Administrator (3) | 全リソースへのフルアクセス | システム管理者 |
| User (1) | 制限付きアクセス | 一般ユーザー |
| Guest (2) | 最小限のアクセス | 未認証ユーザー |

### アクセス制御

#### ACL構造

```
users_roles         - ロール定義テーブル
users_resources     - リソース定義テーブル
users_privileges    - ロール-リソース間の権限マッピング
```

#### リソースカテゴリ

| モジュール | リソース例 | 説明 |
| --- | --- | --- |
| global | gadmin | 管理画面アクセス |
| users | uusers, uedit, uview | ユーザー管理 |
| articles | aarticles, aarticleedit | 記事管理 |
| pages | ppages, ppageedit | ページ管理 |
| events | eevents, eeventedit | イベント管理 |
| assets | fassets, fassetupload | アセット管理 |
| mail | mmail, mmailsend | メール管理 |

#### 認可チェック実装

```php
// コントローラー内での権限チェック例
if($this->view->acl->isAllowed($this->view->user->user_role, 'aarticles')) {
    // 許可された処理
} else {
    $this->_forward('privileges','error','admin');
}
```

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | 設定ファイルで指定可能（インフラ依存） |
| HSTS | 未実装 |
| SMTP暗号化 | SSL/TLS（ポート465） |

### メール送信設定

```php
$config = array(
    'auth' => 'login',
    'port' => 465,
    'ssl' => 'ssl',
    'username' => $registry->site->site->email,
    'password' => $registry->site->smtp->pass
);
$transport = new Zend_Mail_Transport_Smtp($registry->site->smtp->server, $config);
```

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | SMTP: SSL/TLS、HTTP: インフラ依存 |
| パスワード保存 | MD5ハッシュ（サイトキー+ソルト連結） |
| セッションデータ | PHP標準セッション（暗号化なし） |

### 機密情報管理

#### 設定ファイル構造

機密情報は `config.ini` ファイルで管理：

| 設定項目 | 説明 |
| --- | --- |
| site.key | パスワードハッシュ用サイトキー |
| site.salt | 追加ソルト |
| smtp.pass | SMTP認証パスワード |
| resources.db.params.password | データベースパスワード |

#### データベース資格情報

- 設定ファイル（INI形式）で管理
- `.gitignore` に `config.ini` を除外推奨（`config.example.ini` をテンプレートとして提供）

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | ビューヘルパー `$this->escape()` による出力エスケープ |
| SQLインジェクション | Zend_Db のプリペアドステートメント使用 |
| CSRF | **明示的なトークン検証なし** |

### XSS対策詳細

#### 出力エスケープ

ビューテンプレートでは `$this->escape()` メソッドを使用：

```php
<?php echo $this->escape($this->userArray['user_alias']); ?>
```

#### 入力フィルタリング

`Zend_Filter_Input` を使用した入力検証：

```php
$filters = array(
    'email' => array('StringTrim', 'StringToLower')
);
$validators = array(
    'email' => array('NotEmpty', 'EmailAddress', 'breakChainOnFailure' => true)
);
$input = new Zend_Filter_Input($filters, $validators, $_POST, $options);
```

### SQLインジェクション対策

Zend_Db のクエリビルダーによるパラメータバインディング：

```php
$select = $registry->db->select()
    ->from(array('u' => 'users'))
    ->where('u.user_id = ?', $userid);  // パラメータバインド
```

### CSRF対策

**現状**: 明示的なCSRFトークン検証は実装されていない。一部のアクションでは `confirm` パラメータによる2段階確認を使用しているが、これはCSRF対策としては不十分。

```php
// 現行の確認フロー（CSRF対策としては不十分）
if ($confirm == '1' & isset($id)) {
    // 削除実行
}
```

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 認証ログ | 明示的な実装なし | N/A |
| アクセスログ | Webサーバー依存 | インフラ設定による |
| 操作ログ | 明示的な実装なし | N/A |

**備考**: 本システムには専用の監査ログ機能が実装されていない。セキュリティイベントの追跡にはWebサーバーログへの依存が必要。

## セキュリティ上の懸念事項

### 高リスク

| 項目 | リスク内容 | 推奨対応 |
| --- | --- | --- |
| MD5ハッシュ使用 | 脆弱なハッシュアルゴリズム | bcrypt/Argon2への移行 |
| Remember Me Cookie | 平文パスワード保存 | セキュアトークン方式への変更 |
| CSRF対策未実装 | クロスサイトリクエストフォージェリ脆弱性 | CSRFトークンの導入 |

### 中リスク

| 項目 | リスク内容 | 推奨対応 |
| --- | --- | --- |
| セッションID受付 | URLからのセッションID受付 | Flash対応コードの削除または改善 |
| 監査ログ未実装 | セキュリティイベント追跡不可 | ログ機能の追加 |

### 低リスク

| 項目 | リスク内容 | 推奨対応 |
| --- | --- | --- |
| HSTS未実装 | 中間者攻撃リスク | HSTSヘッダーの追加 |
| パスワードポリシー | 最小長4文字は弱い | 最小8文字以上への強化 |

## 備考

- 本システムはZend Framework 1.x時代のレガシーアプリケーションであり、現代のセキュリティ標準から見ると複数の改善点がある
- 運用環境ではHTTPS必須化、WAF導入、定期的なセキュリティ監査を推奨
- パスワードハッシュアルゴリズムのMD5からbcrypt/Argon2への移行を強く推奨
