# アーキテクチャ設計書

## 概要

本ドキュメントは、LLVM（Low Level Virtual Machine）プロジェクトのアーキテクチャ設計を記述したものである。LLVMは、高度に最適化されたコンパイラ、オプティマイザ、およびランタイム環境を構築するためのツールキットであり、モジュラーで再利用可能なコンパイラおよびツールチェーン技術のコレクションとして設計されている。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| ソースコード | ファイルシステム入力 | コンパイル対象のソースファイル（C/C++/Fortran等） |
| オブジェクトファイル | ファイルシステム出力 | コンパイル結果の出力 |
| 実行可能ファイル | ファイルシステム出力 | リンク結果の出力 |
| デバッグ情報（DWARF等） | 標準フォーマット | デバッガとの連携 |
| ターゲットハードウェア | 機械語生成 | 各種CPU/GPUアーキテクチャ |
| CMake | ビルドシステム | プロジェクト構成・ビルド管理 |

## レイヤー構成

### アーキテクチャスタイル

LLVMは**パイプライン・フィルタアーキテクチャ**と**モジュラーアーキテクチャ**を組み合わせた設計を採用している。コンパイル処理は複数のフェーズ（字句解析、構文解析、意味解析、IR生成、最適化、コード生成）をパイプラインとして実行し、各フェーズは独立したモジュールとして実装されている。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Frontend | ソース言語の解析、中間表現への変換 | Clang（C/C++）、Flang（Fortran）、字句解析、構文解析、AST生成 |
| Middle-End | 言語非依存の最適化、解析 | LLVM IR、Analysis、Transforms、パス管理 |
| Backend | ターゲット固有のコード生成 | CodeGen、Target、MC（Machine Code）、アセンブラ |
| Runtime | 実行時サポート | Compiler-RT、libc++、libunwind、OpenMP |
| Tools | 開発支援ツール | LLDB（デバッガ）、LLD（リンカ）、各種ユーティリティ |

### レイヤー間の依存関係ルール

- **Frontend → Middle-End**: フロントエンドはLLVM IRを生成し、Middle-Endに渡す
- **Middle-End → Backend**: 最適化されたIRをBackendに渡し、ターゲット固有のコードを生成
- **下位レイヤーから上位レイヤーへの依存は禁止**: 各レイヤーは下位レイヤーのAPIのみを使用
- **共通ライブラリ（Support, ADT）**: 全レイヤーから参照可能なユーティリティ

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Core LLVM | 中間表現と基本変換 | llvm/lib/IR、llvm/lib/Analysis、llvm/lib/Transforms |
| Clang | C/C++/Objective-C コンパイラ | clang/lib/AST、clang/lib/Sema、clang/lib/CodeGen |
| LLD | リンカ | lld/ELF、lld/COFF、lld/MachO、lld/wasm |
| LLDB | デバッガ | lldb/source/API、lldb/source/Core、lldb/source/Target |
| MLIR | 多層中間表現 | mlir/lib/IR、mlir/lib/Dialect、mlir/lib/Transforms |
| Compiler-RT | ランタイムライブラリ | compiler-rt/lib/builtins、compiler-rt/lib/sanitizer |
| Polly | ポリヘドラル最適化 | polly/lib/Analysis、polly/lib/Transform |
| BOLT | バイナリ最適化 | bolt/lib/Core、bolt/lib/Passes |
| Flang | Fortranコンパイラ | flang/lib/Parser、flang/lib/Semantics、flang/lib/Lower |

### パッケージ構造

```
llvm-project/
├── llvm/                    # LLVM Core
│   ├── include/llvm/        # 公開ヘッダファイル
│   ├── lib/                 # ライブラリ実装
│   │   ├── IR/              # 中間表現
│   │   ├── Analysis/        # 解析パス
│   │   ├── Transforms/      # 変換パス
│   │   ├── CodeGen/         # コード生成
│   │   ├── Target/          # ターゲットバックエンド
│   │   │   ├── X86/         # x86/x64
│   │   │   ├── AArch64/     # ARM64
│   │   │   ├── RISCV/       # RISC-V
│   │   │   └── ...          # その他アーキテクチャ
│   │   ├── MC/              # Machine Code
│   │   └── Support/         # 共通ユーティリティ
│   └── tools/               # コマンドラインツール
├── clang/                   # Clang C/C++ Compiler
│   ├── lib/
│   │   ├── Lex/             # 字句解析
│   │   ├── Parse/           # 構文解析
│   │   ├── AST/             # 抽象構文木
│   │   ├── Sema/            # 意味解析
│   │   └── CodeGen/         # LLVM IR生成
│   └── tools/
├── lld/                     # LLD Linker
│   ├── ELF/                 # ELFフォーマット
│   ├── COFF/                # COFFフォーマット（Windows）
│   ├── MachO/               # Mach-Oフォーマット（macOS）
│   └── wasm/                # WebAssembly
├── lldb/                    # LLDB Debugger
│   └── source/
├── mlir/                    # MLIR
│   └── lib/
├── compiler-rt/             # Compiler Runtime
│   └── lib/
├── libcxx/                  # libc++ Standard Library
├── libcxxabi/               # libc++abi
├── libunwind/               # Unwinding Library
├── openmp/                  # OpenMP Runtime
├── polly/                   # Polyhedral Optimizer
├── bolt/                    # Binary Optimization
├── flang/                   # Fortran Compiler
└── flang-rt/                # Flang Runtime
```

### コンポーネント依存関係

1. **Clang → LLVM Core**: ClangはLLVM IRを生成するためLLVM Coreに依存
2. **LLD → LLVM Core**: オブジェクトファイル処理でLLVMライブラリを使用
3. **LLDB → Clang, LLVM Core**: デバッグ情報解析とExpression Evaluationで使用
4. **Flang → MLIR, LLVM Core**: Fortranコンパイルにおいて多層IRを使用
5. **BOLT → LLVM Core**: バイナリ解析・変換でLLVMインフラを使用
6. **Polly → LLVM Core**: ポリヘドラル最適化パスとしてLLVMに統合

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| 該当なし | - | - | LLVMはデータベースを使用しない（ファイルベース） |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| ビルドキャッシュ | CMake統合 | コンパイル結果のキャッシュ | プロジェクト設定依存 |
| ccache/sccache | 外部ツール | コンパイラキャッシュ | ユーザー設定依存 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| 該当なし | - | LLVMはメッセージキューを使用しない |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| 該当なし | - | LLVMは検索エンジンを使用しない |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| zlib | オプション | 圧縮/展開サポート |
| libxml2 | オプション | XMLパース（一部機能で使用） |
| libedit/readline | オプション | 対話的ツールの行編集 |
| Python | 3.x | テスト実行、ビルドスクリプト |

## データフロー

### リクエスト処理の流れ

コンパイル処理の典型的なフロー：

1. **ソースファイル入力**: ドライバがソースファイルを受け取る
2. **プリプロセス**: プリプロセッサがマクロ展開、インクルード処理を実行
3. **字句解析（Lexer）**: ソースコードをトークン列に変換
4. **構文解析（Parser）**: トークン列から抽象構文木（AST）を構築
5. **意味解析（Sema）**: 型チェック、名前解決、セマンティクス検証
6. **IR生成（CodeGen）**: ASTからLLVM IRを生成
7. **最適化パス**: IRに対して各種最適化を適用
8. **コード生成**: IRからターゲット固有のマシンコードを生成
9. **アセンブリ/オブジェクト出力**: アセンブリファイルまたはオブジェクトファイルを出力

### 非同期処理の流れ

1. **並列コンパイル**: CMake/Ninjaによる複数ファイルの並列コンパイル
2. **LTO（Link Time Optimization）**: リンク時に複数モジュールを最適化
3. **ThinLTO**: 軽量なリンク時最適化で並列処理を実現
4. **DTLTO（Distributed ThinLTO）**: 分散環境でのThinLTO実行

### データ永続化の流れ

1. **ビットコード出力**: LLVM IRをビットコード形式（.bc）で保存
2. **オブジェクトファイル出力**: コンパイル結果を.oファイルとして保存
3. **プロファイルデータ**: PGO用のプロファイル情報を保存
4. **デバッグ情報**: DWARF形式でデバッグ情報を埋め込み/出力

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| 該当なし | - | LLVMはコンパイラであり認証機能を持たない |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| デバッグ出力 | DEBUG()マクロ | 標準エラー出力 |
| 最適化レポート | OptimizationRemark | 標準出力/ファイル |
| 統計情報 | Statistic | 標準エラー出力 |
| タイムレポート | TimePassesHandler | 標準エラー出力 |

### エラーハンドリング

LLVMでは以下のエラーハンドリング戦略を採用：

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| 構文エラー | Diagnostic | エラーメッセージ出力、コンパイル中断 |
| 型エラー | Diagnostic | エラーメッセージ出力、コンパイル中断 |
| 内部エラー | llvm::Error/Expected | エラー伝播、適切な処理 |
| 致命的エラー | report_fatal_error | 即座に終了 |
| アサーション | assert/llvm_unreachable | デバッグビルドでのチェック |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| 該当なし | - | コンパイラはトランザクション管理を使用しない |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| RAII | メモリ管理全般 | unique_ptr、スマートポインタの活用 |
| パス・パイプライン | 最適化フレームワーク | PassManager、AnalysisManager |
| Visitor パターン | AST走査、IR走査 | InstVisitor、RecursiveASTVisitor |
| Factory パターン | ターゲット生成 | TargetRegistry、TargetMachine生成 |
| Strategy パターン | コード生成 | 各ターゲット固有のStrategy実装 |
| TableGen | コード生成 | 命令定義、レジスタ定義の自動生成 |

### コーディング規約

LLVMプロジェクトは独自のコーディング規約（LLVM Coding Standards）を採用：

- **命名規則**: 型名は大文字キャメルケース、変数は小文字キャメルケース
- **インデント**: 2スペース
- **行長**: 80文字制限
- **ヘッダガード**: `LLVM_*_H` 形式
- **名前空間**: `llvm::` 名前空間を使用
- **assert**: 開発時の不変条件チェックに積極使用
- **スマートポインタ**: 所有権が明確な場合はスマートポインタを使用
- **RTTI**: デフォルトでは無効（isa<>/cast<>/dyn_cast<>を使用）
- **例外**: 使用しない（-fno-exceptions）
- **C++標準**: C++17を使用

## 備考

### サポートターゲットアーキテクチャ

LLVMは以下のターゲットアーキテクチャをサポート：

- X86/X86-64
- AArch64（ARM64）
- ARM
- RISC-V
- PowerPC
- MIPS
- SystemZ
- WebAssembly
- AMDGPU
- NVPTX
- SPARC
- Hexagon
- その他多数

### ビルドシステム

- **CMake**: 主要ビルドシステム
- **Ninja**: 推奨ビルドツール（高速ビルド）
- **C++17**: 必須C++標準
- **Python 3**: テスト・ユーティリティ用

### ライセンス

Apache License 2.0 with LLVM Exceptions
