# 機能設計書 13-libc++

## 概要

本ドキュメントは、LLVMプロジェクトにおけるlibc++（LLVM C++標準ライブラリ）の機能設計書である。libc++はC++標準ライブラリの実装であり、ISO C++規格に準拠したコンテナ、アルゴリズム、入出力、スレッド等の機能を提供する。

### 本機能の処理概要

libc++は、C++プログラムが必要とする標準ライブラリ機能を提供する。これには、コンテナ（vector, map等）、アルゴリズム（sort, find等）、文字列処理、入出力ストリーム、スレッドサポート、ファイルシステム操作などが含まれる。

**業務上の目的・背景**：C++言語で開発されるアプリケーションは、標準ライブラリに大きく依存している。libc++は、高性能かつモダンなC++標準ライブラリ実装を提供し、特にClangコンパイラとの親和性が高い。GCCのlibstdc++に代わる選択肢として、macOS、iOS、FreeBSD等で標準採用されている。また、最新のC++規格（C++17, C++20, C++23等）への迅速な対応を特徴とする。

**機能の利用シーン**：
- C++アプリケーション開発における標準ライブラリの利用
- Clangベースのツールチェーンでのビルド
- クロスプラットフォーム開発
- 組み込みシステムでの軽量C++ランタイム

**主要な処理内容**：
1. **コンテナ**：vector, list, map, set, unordered_map等のデータ構造
2. **アルゴリズム**：sort, find, transform等の汎用アルゴリズム
3. **文字列**：string, string_view, 正規表現
4. **入出力**：iostream, fstream, stringstream
5. **並行処理**：thread, mutex, future, atomic
6. **ファイルシステム**：path, directory_iterator（C++17）
7. **フォーマット**：std::format（C++20）
8. **範囲**：ranges（C++20）

**関連システム・外部連携**：
- Clang：推奨コンパイラ
- libc++abi：C++ ABI実装（例外処理、RTTI）
- libunwind：スタックアンワインド
- LLVM：ビルドシステム統合

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリのため画面なし |

## 機能種別

標準ライブラリ / ランタイムライブラリ

## 入力仕様

### 入力パラメータ

libc++はライブラリであり、コンパイラオプションを通じて使用される。

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| -stdlib=libc++ | flag | No | libc++の明示的な使用指定 | - |
| -std=c++XX | flag | No | C++標準バージョン指定 | 有効なバージョン |
| -D_LIBCPP_DEBUG | macro | No | デバッグモード有効化 | - |
| -D_LIBCPP_HARDENING_MODE | macro | No | ハードニングモード指定 | none/fast/extensive/debug |

### 入力データソース

- C++ソースコード（#include文を通じて）
- コンパイラオプション

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| libc++.a | 静的ライブラリ | リンク時に結合 |
| libc++.so / libc++.dylib | 動的ライブラリ | 実行時にロード |
| ヘッダーファイル群 | .h / ヘッダー | コンパイル時にインクルード |

### 出力先

- インストールディレクトリのlib/およびinclude/

## 処理フロー

### 処理シーケンス（std::vector例）

```
1. コンパイル時
   └─ #include <vector>でヘッダーをインクルード
   └─ テンプレートのインスタンス化
   └─ インライン関数の展開

2. リンク時
   └─ libc++.a/libc++.soとリンク
   └─ メモリアロケータ等の共通コードを参照

3. 実行時
   └─ 動的ライブラリの場合はロード
   └─ コンテナ操作（push_back等）の実行
   └─ メモリ管理（new/delete）との連携
```

### フローチャート

```mermaid
flowchart TD
    A[C++ソースコード] --> B[#include <header>]
    B --> C[コンパイラ処理]
    C --> D[テンプレートインスタンス化]
    D --> E[オブジェクトファイル生成]
    E --> F{静的/動的リンク?}
    F -->|静的| G[libc++.aをリンク]
    F -->|動的| H[libc++.so参照を追加]
    G --> I[実行ファイル]
    H --> I
    I --> J[実行時]
    J --> K{動的リンク?}
    K -->|Yes| L[libc++.soロード]
    K -->|No| M[直接実行]
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ABI互換性 | 異なるバージョン間でABI互換性を維持 | 同一メジャーバージョン内 |
| BR-02 | 標準準拠 | ISO C++規格に準拠 | 常時 |
| BR-03 | libc++abi依存 | 例外処理・RTTIにlibc++abiが必要 | 例外/RTTI有効時 |
| BR-04 | スレッドサポート | pthreads等のスレッドライブラリが必要 | マルチスレッド使用時 |

### 計算ロジック

該当なし（汎用ライブラリ）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| std::bad_alloc | 例外 | メモリ確保失敗 | メモリ管理を見直す |
| std::out_of_range | 例外 | 範囲外アクセス | インデックスをチェック |
| std::runtime_error | 例外 | 実行時エラー | エラー原因を調査 |
| assertion failure | アボート | ハードニングモードでの違反 | コードを修正 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 標準規格で規定された計算量を満たす（例：sort O(n log n)）
- インライン展開による最適化
- コピーエリジョン、ムーブセマンティクスの活用

## セキュリティ考慮事項

- ハードニングモードによる境界チェック
- _FORTIFY_SOURCE互換機能
- スタックプロテクタとの連携

## 備考

- macOS、iOSではシステム標準ライブラリとして採用
- FreeBSDでも標準採用
- Androidでは一部バージョンで採用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 設定と構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CMakeLists.txt | `libcxx/CMakeLists.txt` | ビルドオプション、設定項目 |
| 1-2 | __config | `libcxx/include/__config` | マクロ定義、プラットフォーム設定 |

**読解のコツ**: `__config`はlibc++の中核設定ファイル。`_LIBCPP_*`マクロで各種機能の有効/無効を制御。

#### Step 2: コンテナの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | vector | `libcxx/include/vector` | vectorの公開インターフェース |
| 2-2 | __vector/ | `libcxx/include/__vector/` | vector実装の詳細 |
| 2-3 | __memory/ | `libcxx/include/__memory/` | アロケータ、スマートポインタ |

**読解のコツ**: 最近のlibc++はヘッダーをモジュール化。`<vector>`は`__vector/`ディレクトリ内のヘッダーを集約。

#### Step 3: アルゴリズムの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | algorithm | `libcxx/include/algorithm` | algorithmの公開インターフェース |
| 3-2 | __algorithm/ | `libcxx/include/__algorithm/` | 各アルゴリズムの実装 |

#### Step 4: 入出力の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | iostream | `libcxx/include/iostream` | iostream定義 |
| 4-2 | __locale | `libcxx/include/__locale` | ロケールサポート |
| 4-3 | src/iostream.cpp | `libcxx/src/iostream.cpp` | iostream実装 |

#### Step 5: スレッドサポートを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | thread | `libcxx/include/thread` | threadクラス定義 |
| 5-2 | __thread/ | `libcxx/include/__thread/` | スレッド実装詳細 |
| 5-3 | mutex | `libcxx/include/mutex` | 同期プリミティブ |

### プログラム呼び出し階層図

```
ユーザーコード
    │
    ├─ #include <vector>
    │      │
    │      └─ libcxx/include/vector
    │             │
    │             ├─ __vector/vector.h
    │             │      └─ std::vector<T>テンプレート
    │             │
    │             └─ __memory/allocator.h
    │                    └─ std::allocator<T>
    │
    ├─ #include <algorithm>
    │      │
    │      └─ libcxx/include/algorithm
    │             │
    │             └─ __algorithm/sort.h
    │                    └─ std::sort()
    │
    └─ リンク時
           │
           └─ libc++.a / libc++.so
                  │
                  ├─ src/algorithm.cpp
                  ├─ src/string.cpp
                  ├─ src/iostream.cpp
                  └─ ...
```

### データフロー図

```
[入力]                    [処理]                         [出力]

C++ソースコード ─────────▶ Clang/GCC ─────────────────▶ オブジェクトファイル
       │                        │
       │                        └─▶ libc++ヘッダー解析
       │                               │
       │                               └─▶ テンプレートインスタンス化
       │
       └─▶ リンカ ──────────────────────────────────▶ 実行ファイル
                  │
                  └─▶ libc++ライブラリリンク
                         │
                         └─▶ 実行時: コンテナ操作
                                    │
                                    └─▶ メモリ管理（new/delete）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CMakeLists.txt | `libcxx/CMakeLists.txt` | 設定 | ビルド設定 |
| __config | `libcxx/include/__config` | ヘッダー | 全体設定マクロ |
| vector | `libcxx/include/vector` | ヘッダー | vectorコンテナ |
| string | `libcxx/include/string` | ヘッダー | 文字列クラス |
| algorithm | `libcxx/include/algorithm` | ヘッダー | アルゴリズム群 |
| memory | `libcxx/include/memory` | ヘッダー | スマートポインタ |
| thread | `libcxx/include/thread` | ヘッダー | スレッドサポート |
| mutex | `libcxx/include/mutex` | ヘッダー | 同期プリミティブ |
| iostream | `libcxx/include/iostream` | ヘッダー | 入出力ストリーム |
| filesystem | `libcxx/include/filesystem` | ヘッダー | ファイルシステム |
| format | `libcxx/include/format` | ヘッダー | フォーマット（C++20） |
| ranges | `libcxx/include/ranges` | ヘッダー | 範囲（C++20） |
| __algorithm/ | `libcxx/include/__algorithm/` | ヘッダー群 | アルゴリズム実装 |
| __chrono/ | `libcxx/include/__chrono/` | ヘッダー群 | 時間ライブラリ |
| __filesystem/ | `libcxx/include/__filesystem/` | ヘッダー群 | ファイルシステム実装 |
| __format/ | `libcxx/include/__format/` | ヘッダー群 | フォーマット実装 |
| __functional/ | `libcxx/include/__functional/` | ヘッダー群 | 関数オブジェクト |
| __iterator/ | `libcxx/include/__iterator/` | ヘッダー群 | イテレータ |
| __memory/ | `libcxx/include/__memory/` | ヘッダー群 | メモリ管理 |
| __ranges/ | `libcxx/include/__ranges/` | ヘッダー群 | 範囲実装 |
| __thread/ | `libcxx/include/__thread/` | ヘッダー群 | スレッド実装 |
| src/ | `libcxx/src/` | ソース | 非インラインコード実装 |
| src/algorithm.cpp | `libcxx/src/algorithm.cpp` | ソース | アルゴリズム実装 |
| src/string.cpp | `libcxx/src/string.cpp` | ソース | 文字列実装 |
| src/iostream.cpp | `libcxx/src/iostream.cpp` | ソース | iostream実装 |
| src/filesystem/ | `libcxx/src/filesystem/` | ソース | ファイルシステム実装 |
