# 機能設計書 14-libc++abi

## 概要

本ドキュメントは、LLVMプロジェクトにおけるlibc++abi（C++ Application Binary Interface）の機能設計書である。libc++abiは、C++のABI（Application Binary Interface）を実装するライブラリであり、例外処理、RTTI（Run-Time Type Information）、およびその他のC++ランタイム機能を提供する。

### 本機能の処理概要

libc++abiは、Itanium C++ ABIに準拠した実装を提供し、libc++と組み合わせて使用される。主に例外処理機構、動的型情報、および低レベルのC++ランタイムサポートを担当する。

**業務上の目的・背景**：C++プログラムが例外を投げたり、dynamic_castやtypeidを使用したりする際には、ABIレベルでの統一された実装が必要である。libc++abiは、libc++が正しく動作するために必要なABI層を提供し、異なるコンパイラやライブラリ間での相互運用性を確保する。特にmacOS、iOS、FreeBSD等では標準のC++ ABIライブラリとして採用されている。

**機能の利用シーン**：
- C++例外のthrow/catch処理
- dynamic_castによる動的型変換
- typeidによる実行時型情報取得
- new/delete演算子のデフォルト実装
- スレッドローカルストレージの初期化
- 静的オブジェクトのガードと初期化

**主要な処理内容**：
1. **例外処理**：__cxa_throw, __cxa_catch等の例外関連関数
2. **RTTI**：type_info実装、dynamic_cast支援
3. **デマングリング**：__cxa_demangleによるシンボル名の復元
4. **ガード**：静的ローカル変数の一度だけの初期化保証
5. **new/delete**：標準のメモリ確保/解放演算子
6. **atexit**：プログラム終了時のデストラクタ呼び出し

**関連システム・外部連携**：
- libc++：C++標準ライブラリ本体
- libunwind：スタックアンワインド（例外伝播）
- Clang：コンパイラ

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ランタイムライブラリのため画面なし |

## 機能種別

ABIライブラリ / ランタイムライブラリ / 例外処理

## 入力仕様

### 入力パラメータ

libc++abiは自動的にリンクされ、通常は明示的なオプション指定は不要。

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| -lc++abi | flag | No | 明示的なlibc++abiリンク | - |
| LIBCXXABI_USE_LLVM_UNWINDER | cmake | No | libunwindの使用 | ON/OFF |
| LIBCXXABI_ENABLE_EXCEPTIONS | cmake | No | 例外サポートの有効化 | ON/OFF |

### 入力データソース

- C++ソースコード（例外、RTTI使用時に自動的に呼び出される）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| libc++abi.a | 静的ライブラリ | リンク時に結合 |
| libc++abi.so / libc++abi.dylib | 動的ライブラリ | 実行時にロード |

### 出力先

- インストールディレクトリのlib/

## 処理フロー

### 処理シーケンス（例外処理例）

```
1. 例外のthrow
   └─ throw文がコンパイラにより__cxa_allocate_exception + __cxa_throw呼び出しに変換

2. スタックアンワインド
   └─ libunwindが呼び出され、スタックフレームを巻き戻し
   └─ __cxa_personality関数がランディングパッドを検索

3. 例外のcatch
   └─ __cxa_begin_catchで例外オブジェクト取得
   └─ ユーザーのcatchブロック実行
   └─ __cxa_end_catchで例外オブジェクト解放

4. 例外オブジェクトの解放
   └─ __cxa_free_exceptionでメモリ解放
```

### フローチャート

```mermaid
flowchart TD
    A[throw expression] --> B[__cxa_allocate_exception]
    B --> C[例外オブジェクト構築]
    C --> D[__cxa_throw]
    D --> E[libunwindでスタックアンワインド]
    E --> F{catchブロック発見?}
    F -->|Yes| G[__cxa_begin_catch]
    F -->|No| H[std::terminate]
    G --> I[catchブロック実行]
    I --> J[__cxa_end_catch]
    J --> K[正常継続]
    H --> L[プログラム終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | Itanium ABI準拠 | Itanium C++ ABIに準拠 | 常時 |
| BR-02 | libunwind依存 | 例外処理にはlibunwindが必要 | 例外有効時 |
| BR-03 | MSVC非対応 | MSVCターゲットでは使用不可（MS ABIを使用） | - |
| BR-04 | libc++連携 | libc++と組み合わせて使用 | 常時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| std::terminate | 終了 | 未捕捉例外 | try-catchを追加 |
| std::bad_alloc | 例外 | 例外オブジェクト確保失敗 | メモリを確保 |
| std::bad_cast | 例外 | 不正なdynamic_cast | 型を確認 |
| std::bad_typeid | 例外 | nullポインタのtypeid | ポインタをチェック |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 例外を投げない通常パスではゼロオーバーヘッド
- 例外発生時のみスタックアンワインドコストが発生

## セキュリティ考慮事項

- 例外オブジェクトのメモリ管理は安全に実装
- type_info比較は安全に実装

## 備考

- macOS、iOSではシステム標準として採用
- GCCのlibsupc++に相当する機能を提供
- 例外を無効にしたビルド（-fno-exceptions）も可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 全体構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CMakeLists.txt | `libcxxabi/CMakeLists.txt` | ビルドオプション、依存関係 |
| 1-2 | include/cxxabi.h | `libcxxabi/include/cxxabi.h` | 公開APIの定義 |

**読解のコツ**: LIBCXXABI_USE_LLVM_UNWINDERオプションでlibunwindとの連携を制御。MSVCターゲットでは使用不可（Itanium ABI専用）。

#### Step 2: 例外処理の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | cxa_exception.h | `libcxxabi/src/cxa_exception.h` | 例外オブジェクトの内部構造 |
| 2-2 | cxa_exception.cpp | `libcxxabi/src/cxa_exception.cpp` | 例外処理の実装 |
| 2-3 | cxa_personality.cpp | `libcxxabi/src/cxa_personality.cpp` | パーソナリティルーチン |

**主要関数**:
- `__cxa_allocate_exception`: 例外オブジェクトのメモリ確保
- `__cxa_throw`: 例外の投げ出し
- `__cxa_begin_catch`: 例外の捕捉開始
- `__cxa_end_catch`: 例外の捕捉終了

#### Step 3: RTTIの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | private_typeinfo.h | `libcxxabi/src/private_typeinfo.h` | type_infoの内部表現 |
| 3-2 | private_typeinfo.cpp | `libcxxabi/src/private_typeinfo.cpp` | RTTI実装 |

#### Step 4: ガードとnew/deleteを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | cxa_guard.cpp | `libcxxabi/src/cxa_guard.cpp` | 静的初期化ガード |
| 4-2 | stdlib_new_delete.cpp | `libcxxabi/src/stdlib_new_delete.cpp` | new/delete実装 |

#### Step 5: デマングリングを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | cxa_demangle.cpp | `libcxxabi/src/cxa_demangle.cpp` | デマングラーのエントリーポイント |
| 5-2 | demangle/ | `libcxxabi/src/demangle/` | デマングラー実装詳細 |

### プログラム呼び出し階層図

```
[例外throw時]
throw expression (ユーザーコード)
    │
    ├─ __cxa_allocate_exception() [cxa_exception.cpp]
    │      └─ 例外オブジェクト用メモリ確保
    │
    ├─ 例外オブジェクトのコンストラクタ
    │
    └─ __cxa_throw() [cxa_exception.cpp]
           │
           └─ _Unwind_RaiseException() [libunwind]
                  │
                  └─ __gxx_personality_v0() [cxa_personality.cpp]
                         └─ ランディングパッド検索

[例外catch時]
catch block entry
    │
    ├─ __cxa_begin_catch() [cxa_exception.cpp]
    │      └─ 例外オブジェクト参照カウント増加
    │
    ├─ ユーザーcatchブロック実行
    │
    └─ __cxa_end_catch() [cxa_exception.cpp]
           └─ 例外オブジェクト参照カウント減少/解放
```

### データフロー図

```
[入力]                    [処理]                         [出力]

throw文 ─────────────────▶ __cxa_throw ─────────────────▶ 例外オブジェクト
                              │
                              └─▶ libunwind呼び出し
                                     │
                                     └─▶ スタックアンワインド
                                            │
catch文 ◀──────────────────────────────────┘
    │
    └─▶ __cxa_begin_catch
           │
           └─▶ 例外オブジェクト取得
                  │
                  └─▶ catchブロック実行
                         │
                         └─▶ __cxa_end_catch
                                │
                                └─▶ 例外オブジェクト解放
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CMakeLists.txt | `libcxxabi/CMakeLists.txt` | 設定 | ビルド設定 |
| cxxabi.h | `libcxxabi/include/cxxabi.h` | ヘッダー | 公開API定義 |
| __cxxabi_config.h | `libcxxabi/include/__cxxabi_config.h` | ヘッダー | 設定マクロ |
| cxa_exception.h | `libcxxabi/src/cxa_exception.h` | ヘッダー | 例外内部構造 |
| cxa_exception.cpp | `libcxxabi/src/cxa_exception.cpp` | ソース | 例外処理実装 |
| cxa_exception_storage.cpp | `libcxxabi/src/cxa_exception_storage.cpp` | ソース | 例外ストレージ |
| cxa_personality.cpp | `libcxxabi/src/cxa_personality.cpp` | ソース | パーソナリティルーチン |
| cxa_guard.cpp | `libcxxabi/src/cxa_guard.cpp` | ソース | 静的初期化ガード |
| cxa_guard_impl.h | `libcxxabi/src/cxa_guard_impl.h` | ヘッダー | ガード実装詳細 |
| cxa_handlers.cpp | `libcxxabi/src/cxa_handlers.cpp` | ソース | ハンドラ実装 |
| cxa_demangle.cpp | `libcxxabi/src/cxa_demangle.cpp` | ソース | デマングラー |
| cxa_vector.cpp | `libcxxabi/src/cxa_vector.cpp` | ソース | ベクトル例外 |
| cxa_virtual.cpp | `libcxxabi/src/cxa_virtual.cpp` | ソース | 仮想関数サポート |
| cxa_thread_atexit.cpp | `libcxxabi/src/cxa_thread_atexit.cpp` | ソース | スレッドatexit |
| private_typeinfo.h | `libcxxabi/src/private_typeinfo.h` | ヘッダー | RTTI内部構造 |
| private_typeinfo.cpp | `libcxxabi/src/private_typeinfo.cpp` | ソース | RTTI実装 |
| stdlib_new_delete.cpp | `libcxxabi/src/stdlib_new_delete.cpp` | ソース | new/delete |
| stdlib_exception.cpp | `libcxxabi/src/stdlib_exception.cpp` | ソース | 標準例外 |
| stdlib_stdexcept.cpp | `libcxxabi/src/stdlib_stdexcept.cpp` | ソース | stdexcept |
| stdlib_typeinfo.cpp | `libcxxabi/src/stdlib_typeinfo.cpp` | ソース | type_info |
| fallback_malloc.cpp | `libcxxabi/src/fallback_malloc.cpp` | ソース | フォールバックアロケータ |
| abort_message.cpp | `libcxxabi/src/abort_message.cpp` | ソース | アボートメッセージ |
| demangle/ | `libcxxabi/src/demangle/` | ソース群 | デマングラー詳細実装 |
