# 機能設計書 15-libc

## 概要

本ドキュメントは、LLVMプロジェクトにおけるlibc（LLVM C Library）の機能設計書である。LLVM libcは、C標準ライブラリの新規実装であり、高い移植性、正確性、および性能を目標として設計されている。

### 本機能の処理概要

LLVM libcは、ISO C標準およびPOSIX標準で定義されたC言語ライブラリ関数を提供する。文字列操作、数学関数、入出力、メモリ管理、スレッド、時間処理などの基本的なシステム機能を実装している。

**業務上の目的・背景**：従来のCライブラリ（glibc、musl等）には歴史的な複雑さやレガシーコードが含まれている。LLVM libcは、モダンなC++で実装され、テスト駆動開発、高品質なコード、良好な移植性を重視した新しいCライブラリを目指している。特にGPUや組み込みシステムなど、従来のCライブラリが使用できない環境でも利用可能な設計となっている。

**機能の利用シーン**：
- 組み込みシステムでの軽量Cランタイム
- GPUプログラミングでのデバイスサイドCライブラリ
- 高精度な数学関数が必要なアプリケーション
- クリーンルーム実装が必要な環境

**主要な処理内容**：
1. **string.h**：文字列操作関数（strlen, strcmp, memcpy等）
2. **math.h**：数学関数（sin, cos, exp, log等）
3. **stdio.h**：入出力関数（printf, fopen, fread等）
4. **stdlib.h**：ユーティリティ関数（malloc, qsort, atoi等）
5. **pthread.h**：POSIXスレッド
6. **time.h**：時間関数
7. **ctype.h**：文字分類関数
8. **errno.h**：エラー処理

**関連システム・外部連携**：
- Clang：コンパイラ統合
- LLVM：ビルドシステム
- Linux/POSIX：システムコールインターフェース

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | ライブラリのため画面なし |

## 機能種別

標準Cライブラリ / ランタイムライブラリ

## 入力仕様

### 入力パラメータ

LLVM libcはライブラリであり、コンパイラオプションを通じて使用される。

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| LIBC_TARGET_TRIPLE | cmake | No | ターゲットトリプル | - |
| LIBC_NAMESPACE | cmake | No | 内部名前空間 | __llvm_libcで開始 |
| LLVM_LIBC_FULL_BUILD | cmake | No | フルビルドモード | ON/OFF |

### 入力データソース

- Cソースコード（#include文を通じて）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| libc.a | 静的ライブラリ | リンク時に結合 |
| ヘッダーファイル群 | .h | コンパイル時にインクルード |

### 出力先

- インストールディレクトリのlib/およびinclude/

## 処理フロー

### 処理シーケンス（数学関数例）

```
1. ユーザーコードでの呼び出し
   └─ #include <math.h>
   └─ sin(x)の呼び出し

2. ヘッダーでの宣言参照
   └─ include/llvm-libc-types/で型定義を読み込み
   └─ 関数プロトタイプを参照

3. 実装の呼び出し
   └─ src/math/sin.cpp内の実装が呼び出される
   └─ 高精度な多項式近似で計算

4. 結果の返却
   └─ 計算結果をdouble型で返却
```

### フローチャート

```mermaid
flowchart TD
    A[Cソースコード] --> B[#include <header.h>]
    B --> C[コンパイラ処理]
    C --> D[ヘッダー解析]
    D --> E[オブジェクトファイル生成]
    E --> F[libc.aとリンク]
    F --> G[実行ファイル]
    G --> H[実行時]
    H --> I[ライブラリ関数呼び出し]
    I --> J[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ISO C準拠 | ISO C標準に準拠 | 常時 |
| BR-02 | POSIX対応 | POSIX標準の部分実装 | Linux等 |
| BR-03 | 名前空間分離 | 内部シンボルは__llvm_libc名前空間 | 常時 |
| BR-04 | GPU対応 | デバイスサイド利用可能 | GPUターゲット |

### 計算ロジック

**数学関数の精度**：
- 正しく丸められた結果（correctly rounded results）を目標
- ULP（Unit in the Last Place）精度の検証

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ERANGE | errno | 範囲エラー | 入力値をチェック |
| EDOM | errno | 定義域エラー | 入力値をチェック |
| EINVAL | errno | 無効な引数 | 引数を修正 |
| ENOMEM | errno | メモリ不足 | メモリを確保 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 数学関数は高精度かつ高速な実装
- 文字列関数はSIMD最適化
- メモリ関数は効率的な実装

## セキュリティ考慮事項

- バッファオーバーフロー対策
- フォーマット文字列脆弱性対策
- 安全な乱数生成

## 備考

- 現在開発中のプロジェクト
- GPU（CUDA/HIP）でのデバイスサイドライブラリとして使用可能
- フルビルドモードと部分ビルドモードがある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 全体構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CMakeLists.txt | `libc/CMakeLists.txt` | ビルド設定、名前空間定義 |
| 1-2 | docs/ | `libc/docs/` | プロジェクトドキュメント |

**読解のコツ**: LIBC_NAMESPACEで内部シンボルの名前空間を制御。デフォルトは`__llvm_libc_XX_YY_ZZ`形式。

#### Step 2: ヘッダー構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | include/ | `libc/include/` | 公開ヘッダー |
| 2-2 | hdr/ | `libc/hdr/` | ヘッダーマクロ定義 |

#### Step 3: 文字列関数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | string/ | `libc/src/string/` | 文字列関数実装 |
| 3-2 | __support/ | `libc/src/__support/` | サポートユーティリティ |

#### Step 4: 数学関数の実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | math/ | `libc/src/math/` | 数学関数実装 |

#### Step 5: pthread実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | pthread/ | `libc/src/pthread/` | POSIXスレッド実装 |

### プログラム呼び出し階層図

```
ユーザーコード
    │
    ├─ #include <string.h>
    │      │
    │      └─ libc/include/string.h
    │             │
    │             └─ libc/src/string/strlen.cpp等
    │
    ├─ #include <math.h>
    │      │
    │      └─ libc/include/math.h
    │             │
    │             └─ libc/src/math/sin.cpp等
    │                    │
    │                    └─ libc/src/__support/FPUtil/
    │
    └─ #include <stdio.h>
           │
           └─ libc/include/stdio.h
                  │
                  └─ libc/src/stdio/printf.cpp等
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Cソースコード ───────────▶ Clang ─────────────────────▶ オブジェクトファイル
       │                        │
       │                        └─▶ libcヘッダー解析
       │
       └─▶ リンカ ──────────────────────────────────▶ 実行ファイル
                  │
                  └─▶ libc.aリンク
                         │
                         └─▶ 実行時: ライブラリ関数呼び出し
                                    │
                                    └─▶ システムコール（必要時）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CMakeLists.txt | `libc/CMakeLists.txt` | 設定 | ビルド設定 |
| include/ | `libc/include/` | ヘッダー群 | 公開ヘッダー |
| hdr/ | `libc/hdr/` | ヘッダー群 | マクロ定義 |
| src/ | `libc/src/` | ソース群 | 関数実装 |
| src/string/ | `libc/src/string/` | ソース群 | 文字列関数 |
| src/math/ | `libc/src/math/` | ソース群 | 数学関数 |
| src/stdio/ | `libc/src/stdio/` | ソース群 | 入出力関数 |
| src/stdlib/ | `libc/src/stdlib/` | ソース群 | ユーティリティ |
| src/pthread/ | `libc/src/pthread/` | ソース群 | スレッド関数 |
| src/time/ | `libc/src/time/` | ソース群 | 時間関数 |
| src/ctype/ | `libc/src/ctype/` | ソース群 | 文字分類 |
| src/errno/ | `libc/src/errno/` | ソース群 | エラー処理 |
| src/fenv/ | `libc/src/fenv/` | ソース群 | 浮動小数点環境 |
| src/setjmp/ | `libc/src/setjmp/` | ソース群 | 非ローカルジャンプ |
| src/signal/ | `libc/src/signal/` | ソース群 | シグナル処理 |
| src/unistd/ | `libc/src/unistd/` | ソース群 | UNIX関数 |
| src/__support/ | `libc/src/__support/` | ソース群 | サポートユーティリティ |
| startup/ | `libc/startup/` | ソース群 | スタートアップコード |
| test/ | `libc/test/` | テスト | ユニットテスト |
| benchmarks/ | `libc/benchmarks/` | ベンチマーク | 性能テスト |
