# 機能設計書：llvm-dwarfdump

## 1. 機能概要

### 1.1 機能名
llvm-dwarfdump - DWARFデバッグ情報をダンプするツール

### 1.2 機能説明
llvm-dwarfdumpは、オブジェクトファイルやデバッグ情報アーカイブ（.dSYMバンドル）に含まれるDWARFデバッグ情報を人間が読める形式で出力するツールです。デバッグ情報の検証、アドレス検索、名前検索、統計収集などの機能を提供します。

### 1.3 関連する画面/API
- CLI（コマンドラインインターフェース）
- 入力：オブジェクトファイル、アーカイブファイル、.dSYMバンドル
- 出力：DWARF情報のテキスト表示、JSON統計、検証結果

---

## 2. 入出力設計

### 2.1 入力
| 項目 | 型 | 必須 | 説明 |
|------|-----|------|------|
| input files | ファイルパス | No | オブジェクトファイルまたは.dSYMバンドル（デフォルト: a.out） |
| -o | ファイルパス | No | 出力ファイル（デフォルト: 標準出力） |

### 2.2 出力
| 項目 | 型 | 説明 |
|------|-----|------|
| DWARF dump | テキスト | DWARFセクションの内容 |
| statistics | JSON | デバッグ情報の品質メトリクス |
| verification | テキスト | 検証結果 |
| exit code | 整数 | 0: 成功（または検証エラーなし）、1: エラー |

### 2.3 セクションダンプオプション
| オプション | 説明 |
|-----------|------|
| --all (-a) | すべてのデバッグセクションをダンプ |
| --debug-info | .debug_infoセクションをダンプ |
| --debug-abbrev | .debug_abbrevセクションをダンプ |
| --debug-line | .debug_lineセクションをダンプ |
| --debug-frame | .debug_frame/.eh_frameセクションをダンプ |
| --debug-aranges | .debug_arangesセクションをダンプ |
| --debug-ranges | .debug_rangesセクションをダンプ |
| --debug-str | .debug_strセクションをダンプ |
| --debug-loc | .debug_locセクションをダンプ |
| --debug-pubnames | .debug_pubnamesセクションをダンプ |
| --debug-pubtypes | .debug_pubtypesセクションをダンプ |

### 2.4 検索・フィルタオプション
| オプション | 説明 |
|-----------|------|
| --lookup=<address> | 指定アドレスのデバッグ情報を検索 |
| --name=<pattern> (-n) | 名前でDIEを検索（完全一致/正規表現） |
| --find=<name> (-f) | アクセラレータテーブルで名前を検索 |
| --find-all-apple | すべてのAppleアクセラレータエントリを出力 |
| --regex (-x) | 名前検索で正規表現を使用 |
| --ignore-case (-i) | 大文字小文字を無視 |

### 2.5 表示制御オプション
| オプション | 説明 |
|-----------|------|
| --show-children (-c) | 子DIEを表示 |
| --show-parents (-p) | 親DIEを表示 |
| --show-form (-F) | DWARF形式を表示 |
| --recurse-depth=<N> (-r) | 子の再帰深度を指定 |
| --parent-recurse-depth=<N> | 親の再帰深度を指定 |
| --verbose (-v) | 詳細情報を表示 |
| --diff | diff用出力（オフセット・アドレス省略） |
| --summarize-types | 型ユニットの簡略表示 |

### 2.6 検証オプション
| オプション | 説明 |
|-----------|------|
| --verify | DWARF情報を検証 |
| --verify-num-threads=<N> | 検証スレッド数 |
| --error-display=<level> | エラー表示レベル（quiet/summary/details/full） |
| --verify-json=<file> | JSON形式でエラーサマリを出力 |
| --quiet | 検証時にSTDOUT出力を抑制 |

### 2.7 その他のオプション
| オプション | 説明 |
|-----------|------|
| --arch=<arch> | 特定アーキテクチャのみダンプ（複数指定可） |
| --dwo | .dwo/.dwpファイルの非スケルトンDIEもダンプ |
| --statistics | JSON形式で品質メトリクスを出力 |
| --show-section-sizes | デバッグセクションサイズを表示 |
| --show-sources | 全コンパイルユニットのソースを表示 |
| --uuid (-u) | UUIDを表示 |
| --show-variable-coverage | 変数カバレッジメトリクスを表示 |

---

## 3. 処理フロー

### 3.1 基本フロー

```
[開始]
   │
   ▼
[コマンドライン引数解析]
   │
   ▼
[オプション検証・調整]
   │
   ▼
[入力ファイル展開（.dSYMバンドル対応）]
   │
   ▼
[各ファイルを処理]
   │
   ├─► --verify ──► verifyObjectFile()
   │
   ├─► --statistics ──► collectStatsForObjectFile()
   │
   ├─► --show-section-sizes ──► collectObjectSectionSizes()
   │
   ├─► --show-sources ──► collectObjectSources()
   │
   └─► 通常ダンプ ──► dumpObjectFile()
   │
   ▼
[終了（Success/Failure）]
```

### 3.2 ファイル処理フロー

```
handleFile()
   │
   ▼
[MemoryBuffer読み込み]
   │
   ▼
handleBuffer()
   │
   ▼
[Binary作成]
   │
   ├─► ObjectFile ──► DWARFContext::create() ──► HandleObj()
   │
   ├─► MachOUniversalBinary ──► 各アーキテクチャを処理
   │
   └─► Archive ──► handleArchive() ──► 各メンバーを処理
```

### 3.3 オブジェクトダンプフロー

```
dumpObjectFile()
   │
   ▼
[MCRegisterInfo作成]
   │
   ▼
[ファイル情報出力]
   │
   ▼
[処理分岐]
   │
   ├─► --lookup ──► lookup()
   │
   ├─► --name ──► filterByName()
   │
   ├─► --find ──► filterByAccelName()
   │
   ├─► --find-all-apple ──► findAllApple()
   │
   └─► 通常 ──► DICtx.dump()
```

---

## 4. データ構造

### 4.1 オプション構造

```cpp
// オフセットオプション（オプショナル値付き）
struct OffsetOption {
    uint64_t Val = 0;        // オフセット値
    bool HasValue = false;   // 値が指定されたか
    bool IsRequested = false; // オプションが要求されたか
};

// ブールオプション
struct BoolOption : public OffsetOption {};
```

### 4.2 エラー詳細レベル

```cpp
enum ErrorDetailLevel {
    OnlyDetailsNoSummary,   // 詳細のみ、サマリなし
    NoDetailsOnlySummary,   // サマリのみ、詳細なし
    NoDetailsOrSummary,     // どちらもなし（quiet）
    BothDetailsAndSummary,  // 両方表示（full）
    Unspecified             // 未指定
};
```

### 4.3 ダンプオプション

```cpp
DIDumpOptions DumpOpts;
DumpOpts.DumpType = DumpType;           // ダンプするセクション
DumpOpts.ChildRecurseDepth = ...;       // 子の再帰深度
DumpOpts.ParentRecurseDepth = ...;      // 親の再帰深度
DumpOpts.ShowAddresses = !Diff;         // アドレス表示
DumpOpts.ShowChildren = ShowChildren;   // 子DIE表示
DumpOpts.ShowParents = ShowParents;     // 親DIE表示
DumpOpts.ShowForm = ShowForm;           // フォーム表示
DumpOpts.Verbose = Verbose;             // 詳細表示
```

---

## 5. 主要機能

### 5.1 アドレス検索（--lookup）

```cpp
bool lookup(ObjectFile &Obj, DWARFContext &DICtx, uint64_t Address, raw_ostream &OS) {
    auto DIEsForAddr = DICtx.getDIEsForAddress(Address, DumpNonSkeleton);
    // コンパイルユニット、関数、ブロック、行情報を出力
}
```

### 5.2 名前検索（--name）

```cpp
void filterByName(const StringSet<> &Names, DWARFContext::unit_iterator_range CUs, ...) {
    // 全DIEを走査し、DW_AT_name/DW_AT_linkage_nameでマッチング
    // --regexで正規表現、--ignore-caseで大文字小文字無視
}
```

### 5.3 アクセラレータテーブル検索（--find）

```cpp
void filterByAccelName(ArrayRef<std::string> Names, DWARFContext &DICtx, ...) {
    // Apple Names/Types/Namespacesテーブルと.debug_namesを検索
}
```

### 5.4 DWARF検証（--verify）

```cpp
bool verifyObjectFile(ObjectFile &Obj, DWARFContext &DICtx, ...) {
    bool Result = DICtx.verify(stream, getDumpOpts(DICtx));
    // エラー有無を返す
}
```

---

## 6. エラーハンドリング

### 6.1 エラーケース
| エラー | 原因 | 対応 |
|--------|------|------|
| ファイル読み込み失敗 | ファイル不存在など | エラーメッセージ出力、exit(1) |
| バイナリ作成失敗 | 不正な形式 | エラーメッセージ出力、exit(1) |
| 正規表現エラー | 無効な正規表現 | エラーメッセージ出力、exit(1) |
| --diffと--verbose同時指定 | 非互換オプション | エラーメッセージ出力、終了コード1 |

### 6.2 回復可能エラー

```cpp
auto RecoverableErrorHandler = [&](Error E) {
    Result = false;
    WithColor::defaultErrorHandler(std::move(E));
};
```

---

## 7. 技術仕様

### 7.1 サポートファイル形式
- ELF、MachO、COFF、Wasm
- MachO Universal Binary（Fat Binary）
- Archive（.a）
- .dSYMバンドル

### 7.2 アーキテクチャフィルタリング
- `--arch`で特定アーキテクチャのみ処理
- 名前または番号で指定可能

### 7.3 .dSYMバンドル展開
```cpp
MachOObjectFile::findDsymObjectMembers(F)
// バンドル内のオブジェクトファイルを展開
```

### 7.4 並列検証
```cpp
if (!VerifyNumThreads)
    parallel::strategy = hardware_concurrency(...);
else
    parallel::strategy = hardware_concurrency(VerifyNumThreads);
```

---

## 8. 制約事項

### 8.1 オプション制約
- `--diff`と`--verbose`は同時指定不可
- `--error-display`または`--verify-json`指定時は自動的に`--verify`が有効

### 8.2 デフォルト動作
- 入力ファイル未指定時は`a.out`を使用
- ダンプタイプ未指定時は`.debug_info`のみダンプ
- `--verbose`または`--verify`時は全セクションダンプ

---

## 9. 非機能要件

### 9.1 パフォーマンス
- マルチスレッド検証による高速化
- スレッド数は`--verify-num-threads`で制御

### 9.2 出力形式
- diff用出力モードで比較しやすい形式
- JSON形式の統計・エラーサマリ出力

---

## 10. コードリーディングガイド

### 10.1 エントリーポイント
- **ファイル**: `llvm/tools/llvm-dwarfdump/llvm-dwarfdump.cpp`
- **関数**: `main()` (864-980行目)

### 10.2 重要な関数
| 関数名 | 行番号 | 説明 |
|--------|--------|------|
| main() | 864-980 | エントリーポイント |
| getDumpOpts() | 378-402 | ダンプオプション構築 |
| filterByName() | 436-491 | 名前でDIEをフィルタリング |
| filterByAccelName() | 533-550 | アクセラレータテーブルで検索 |
| findAllApple() | 553-581 | 全Appleアクセラレータエントリを出力 |
| lookup() | 589-616 | アドレスでデバッグ情報を検索 |
| collectObjectSources() | 637-697 | ソースファイル一覧を収集 |
| dumpObjectFile() | 714-769 | オブジェクトファイルをダンプ |
| verifyObjectFile() | 771-784 | DWARF検証を実行 |
| handleBuffer() | 806-853 | バッファ内のバイナリを処理 |
| handleFile() | 855-862 | ファイルを処理 |

### 10.3 プログラム呼び出し階層図

```
main()
├── cl::ParseCommandLineOptions()              [876-879行目]
├── [オプション検証]                            [883-891行目]
├── ToolOutputFile()                           [894行目]
├── [DumpType設定]                             [904-923行目]
├── MachOObjectFile::findDsymObjectMembers()   [937行目]
└── [処理分岐]
    ├── handleFile(verifyObjectFile)           [955行目]
    │   └── verifyObjectFile()                 [771-784行目]
    │       └── DICtx.verify()                 [778行目]
    ├── handleFile(collectStatsForObjectFile)  [958行目]
    ├── handleFile(collectObjectSectionSizes)  [961行目]
    ├── handleFile(collectObjectSources)       [964行目]
    │   └── collectObjectSources()             [637-697行目]
    └── handleFile(dumpObjectFile)             [967行目]
        └── dumpObjectFile()                   [714-769行目]
            ├── createRegInfo()                [717行目]
            ├── lookup()                       [739行目]
            ├── filterByName()                 [747-748行目]
            ├── filterByAccelName()            [754行目]
            ├── findAllApple()                 [760行目]
            └── DICtx.dump()                   [767行目]
```

### 10.4 データフロー図

```
入力ファイル (オブジェクト/.dSYM/アーカイブ)
         │
         ▼
    ┌─────────────────────┐
    │ findDsymObjectMembers│ ← .dSYMバンドル展開
    └─────────────────────┘
         │
         ▼
    ┌─────────────────────┐
    │ handleFile()        │
    │ MemoryBuffer読み込み │
    └─────────────────────┘
         │
         ▼
    ┌─────────────────────┐
    │ handleBuffer()      │
    │ object::createBinary│
    └─────────────────────┘
         │
         ├─► ObjectFile
         │        │
         │        ▼
         │   ┌─────────────────────┐
         │   │ DWARFContext::create│
         │   └─────────────────────┘
         │        │
         │        ▼
         │   ┌─────────────────────┐
         │   │ filterArch()        │ ← アーキテクチャフィルタ
         │   └─────────────────────┘
         │        │
         │        ▼
         │   ┌─────────────────────┐
         │   │ HandleObj()         │
         │   │ (dump/verify/stats) │
         │   └─────────────────────┘
         │
         ├─► MachOUniversalBinary
         │        │
         │        ▼
         │   [各アーキテクチャを処理]
         │
         └─► Archive
                  │
                  ▼
             [各メンバーを処理]
         │
         ▼
      出力 (テキスト/JSON)
```

### 10.5 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| llvm/tools/llvm-dwarfdump/llvm-dwarfdump.cpp | メインソースファイル（981行） |
| llvm/tools/llvm-dwarfdump/llvm-dwarfdump.h | ヘッダーファイル |
| llvm/include/llvm/DebugInfo/DWARF/DWARFContext.h | DWARFコンテキストクラス |
| llvm/include/llvm/DebugInfo/DIContext.h | デバッグ情報コンテキストインターフェース |
| llvm/include/llvm/DebugInfo/DWARF/DWARFDie.h | DWARF DIEクラス |
| llvm/include/llvm/DebugInfo/DWARF/DWARFCompileUnit.h | コンパイルユニットクラス |
| llvm/include/llvm/DebugInfo/DWARF/DWARFAcceleratorTable.h | アクセラレータテーブルクラス |
| llvm/include/llvm/Object/ObjectFile.h | オブジェクトファイルクラス |
| llvm/include/llvm/Object/Archive.h | アーカイブクラス |
| llvm/include/llvm/Object/MachOUniversal.h | MachO Universal Binaryクラス |

---

## 11. 使用例

### 11.1 基本的なダンプ
```bash
llvm-dwarfdump program.o
```

### 11.2 全セクションをダンプ
```bash
llvm-dwarfdump --all program.o
```

### 11.3 特定セクションをダンプ
```bash
llvm-dwarfdump --debug-line program.o
```

### 11.4 アドレスで検索
```bash
llvm-dwarfdump --lookup=0x400000 program.o
```

### 11.5 名前で検索
```bash
llvm-dwarfdump --name=main program.o
```

### 11.6 正規表現で検索
```bash
llvm-dwarfdump --name='test.*' --regex program.o
```

### 11.7 DWARF情報を検証
```bash
llvm-dwarfdump --verify program.o
```

### 11.8 統計を出力
```bash
llvm-dwarfdump --statistics program.o
```

### 11.9 ソースファイル一覧を出力
```bash
llvm-dwarfdump --show-sources program.o
```

### 11.10 特定アーキテクチャのみダンプ
```bash
llvm-dwarfdump --arch=x86_64 universal.o
```

### 11.11 .dSYMバンドルをダンプ
```bash
llvm-dwarfdump program.dSYM
```

---

## 12. 改訂履歴

| 版数 | 日付 | 変更内容 | 担当者 |
|------|------|----------|--------|
| 1.0 | 2026-02-01 | 初版作成 | llvm-dwarfdump機能設計書生成 |
