# 機能設計書：llvm-dwp

## 1. 機能概要

### 1.1 機能名
llvm-dwp - DWARF Package Fileを作成するツール

### 1.2 機能説明
llvm-dwpは、DWARF 5 Split DWARF形式の.dwo（DWARF Object）ファイルを.dwp（DWARF Package）ファイルにマージするユーティリティです。Split DWARFを使用したビルドで生成される複数の.dwoファイルを1つのパッケージファイルにまとめることで、デバッグ時のファイルアクセスを効率化します。

### 1.3 関連する画面/API
- CLI（コマンドラインインターフェース）
- 入力：.dwoファイル群、または実行ファイル（.dwo参照を抽出）
- 出力：.dwpファイル

---

## 2. 入出力設計

### 2.1 入力
| 項目 | 型 | 必須 | 説明 |
|------|-----|------|------|
| input files | ファイルパス群 | No | マージする.dwoファイル |
| -e/--execuable | ファイルパス | No | .dwo参照を抽出する実行ファイル（複数指定可） |
| -o/--output | ファイルパス | Yes | 出力.dwpファイル名 |

### 2.2 出力
| 項目 | 型 | 説明 |
|------|-----|------|
| output file | .dwpファイル | マージされたDWARF Packageファイル |
| exit code | 整数 | 0: 成功、1: エラー |

### 2.3 コマンドラインオプション
| オプション | 説明 |
|-----------|------|
| -o, --output | 出力ファイル名を指定 |
| -e, --executable | .dwo参照を抽出する実行ファイルを指定（複数指定可） |
| --continue-on-cu-index-overflow[=<value>] | CUインデックスオーバーフロー時の動作（soft-stop/continue） |
| --dwarf64-str-offsets-promotion[=<value>] | DWARF64文字列オフセットプロモーション（disabled/enabled/always） |
| --help | ヘルプを表示 |
| --version | バージョンを表示 |

---

## 3. 処理フロー

### 3.1 基本フロー

```
[開始]
   │
   ▼
[コマンドライン引数解析]
   │
   ▼
[ターゲット初期化]
   │
   ▼
[実行ファイルから.dwo参照を抽出]（-e指定時）
   │
   ▼
[.dwoファイル一覧を構築]
   │
   ▼
[ターゲットトリプル読み取り]
   │
   ▼
[MCオブジェクト初期化]
   │
   ▼
[MCStreamer作成]
   │
   ▼
[.dwoファイルをマージ] ←─── write()
   │
   ▼
[出力ファイル書き込み]
   │
   ▼
[終了]
```

### 3.2 .dwo参照抽出処理

```
getDWOFilenames()
   │
   ▼
[ObjectFile::createObjectFile()]
   │
   ▼
[DWARFContext作成]
   │
   ▼
[各コンパイルユニットを走査]
   │
   ├─► DW_AT_dwo_name / DW_AT_GNU_dwo_name を取得
   │
   ├─► DW_AT_comp_dir を取得
   │
   └─► 絶対パスを構築
         │
         ├─► パスが存在 → 追加
         │
         └─► 存在しない場合、相対パスで再試行
```

### 3.3 MCオブジェクト初期化フロー

```
MCオブジェクト初期化
   │
   ├─► TargetRegistry::lookupTarget()
   │
   ├─► createMCRegInfo()
   │
   ├─► createMCAsmInfo()
   │
   ├─► createMCSubtargetInfo()
   │
   ├─► MCContext作成
   │
   ├─► createMCObjectFileInfo()
   │
   ├─► createMCAsmBackend()
   │
   ├─► createMCInstrInfo()
   │
   ├─► createMCCodeEmitter()
   │
   └─► createMCObjectStreamer()
```

---

## 4. データ構造

### 4.1 CUインデックスオーバーフロー処理

```cpp
enum class OnCuIndexOverflow {
    HardStop,  // エラーで停止（デフォルト）
    SoftStop,  // 警告して停止
    Continue   // 継続
};
```

### 4.2 DWARF64文字列オフセットプロモーション

```cpp
enum class Dwarf64StrOffsetsPromotion {
    Disabled,  // 無効（デフォルト）
    Enabled,   // 有効
    Always     // 常に有効
};
```

### 4.3 主要な変数

```cpp
static std::vector<std::string> ExecFilenames;  // 実行ファイル一覧
static std::string OutputFilename;              // 出力ファイル名
static std::vector<std::string> DWOFilenames;   // .dwoファイル一覧
```

---

## 5. 主要機能

### 5.1 .dwo参照の自動抽出
- 実行ファイルからDWARF情報を解析
- DW_AT_dwo_name / DW_AT_GNU_dwo_name属性を検索
- コンパイルディレクトリを考慮した絶対パス構築

### 5.2 ターゲット自動検出
- 最初の.dwoファイルからターゲットトリプルを読み取り
- 適切なMCコンポーネントを自動選択

### 5.3 .dwpファイル生成
- 複数の.dwoファイルをマージ
- CUインデックスとTUインデックスを構築
- 文字列プールを統合

---

## 6. エラーハンドリング

### 6.1 エラーケース
| エラー | 原因 | 対応 |
|--------|------|------|
| 出力ファイルオープン失敗 | パーミッション等 | エラーメッセージ出力、終了コード1 |
| 実行ファイル読み込み失敗 | ファイル不存在・形式不正 | エラーメッセージ出力、終了コード1 |
| ターゲット未検出 | 不明なターゲット | エラーメッセージ出力、終了コード1 |
| MCオブジェクト作成失敗 | ターゲットサポート不足 | エラーメッセージ出力、終了コード1 |
| CUインデックスオーバーフロー | インデックス上限超過 | オプションに応じて停止/継続 |
| .dwo参照なし | 実行ファイルに.dwo参照がない | 警告出力、終了コード0 |

### 6.2 警告
| 警告 | 状況 |
|------|------|
| .dwo参照なし | 実行ファイルに.dwo参照がない場合 |

---

## 7. 技術仕様

### 7.1 DWARF属性サポート
- `DW_AT_dwo_name`：DWARF 5標準
- `DW_AT_GNU_dwo_name`：GNU拡張（DWARF 4以前）

### 7.2 パス解決
```cpp
// コンパイルディレクトリを基準に絶対パスを構築
SmallString<16> DWOPath(DWOName);
sys::path::make_absolute(DWOCompDir, DWOPath);
// 絶対パスが存在しない場合、相対パスで再試行
if (!sys::fs::exists(DWOPath) && sys::fs::exists(DWOName))
    DWOPaths.push_back(std::move(DWOName));
```

### 7.3 出力ストリーム
- seekサポートがある場合：直接出力
- seekサポートがない場合：buffer_ostreamを使用

---

## 8. 制約事項

### 8.1 入力制約
- .dwoファイルまたは実行ファイルの指定が必要
- 全ての.dwoファイルは同じターゲットアーキテクチャ

### 8.2 CUインデックス制約
- デフォルトではオーバーフロー時にエラー
- `--continue-on-cu-index-overflow`で動作変更可能

---

## 9. 非機能要件

### 9.1 パフォーマンス
- 複数の.dwoファイルを効率的にマージ
- 文字列プールの重複排除

### 9.2 互換性
- DWARF 4（GNU拡張）とDWARF 5の両方に対応
- 各種ターゲットアーキテクチャに対応

---

## 10. コードリーディングガイド

### 10.1 エントリーポイント
- **ファイル**: `llvm/tools/llvm-dwp/llvm-dwp.cpp`
- **関数**: `llvm_dwp_main()` (123-310行目)

### 10.2 重要な関数
| 関数名 | 行番号 | 説明 |
|--------|--------|------|
| llvm_dwp_main() | 123-310 | エントリーポイント |
| getDWOFilenames() | 77-107 | 実行ファイルから.dwo参照を抽出 |
| readTargetTriple() | 115-121 | ターゲットトリプルを読み取り |
| error() | 109-113 | エラー出力ヘルパー |
| write() | (外部) | .dwoファイルをマージして出力 |

### 10.3 プログラム呼び出し階層図

```
llvm_dwp_main()
├── DwpOptTable::parseArgs()                    [131-135行目]
├── Args.getLastArgValue(OPT_outputFileName)    [148行目]
├── [--continue-on-cu-index-overflow処理]        [149-165行目]
├── [--dwarf64-str-offsets-promotion処理]        [167-186行目]
├── [ExecFilenames収集]                          [188-189行目]
├── [DWOFilenames収集]                           [191-193行目]
├── InitializeAllTargetInfos()                  [195行目]
├── InitializeAllTargetMCs()                    [196行目]
├── InitializeAllTargets()                      [197行目]
├── InitializeAllAsmPrinters()                  [198行目]
├── getDWOFilenames() [各実行ファイル]           [201-215行目]
│   ├── ObjectFile::createObjectFile()          [79行目]
│   ├── DWARFContext::create()                  [84行目]
│   └── [CU走査、DW_AT_dwo_name抽出]            [87-105行目]
├── readTargetTriple()                          [226行目]
│   └── ObjectFile::createObjectFile()          [116行目]
├── TargetRegistry::lookupTarget()              [239-240行目]
├── [MCオブジェクト初期化]
│   ├── createMCRegInfo()                       [247行目]
│   ├── createMCAsmInfo()                       [252-253行目]
│   ├── createMCSubtargetInfo()                 [257-258行目]
│   ├── MCContext()                             [262行目]
│   ├── createMCObjectFileInfo()                [263-264行目]
│   ├── createMCAsmBackend()                    [268行目]
│   ├── createMCInstrInfo()                     [272行目]
│   └── createMCCodeEmitter()                   [276行目]
├── ToolOutputFile()                            [282行目]
├── createMCObjectStreamer()                    [294-297行目]
├── write()                                     [301-302行目]
├── MS->finish()                                [307行目]
└── OutFile.keep()                              [308行目]
```

### 10.4 データフロー図

```
入力ファイル
   │
   ├─► 実行ファイル (-e)
   │        │
   │        ▼
   │   ┌─────────────────────┐
   │   │ getDWOFilenames()   │
   │   │ DWARFContext解析    │
   │   └─────────────────────┘
   │        │
   │        ▼ .dwoファイルパス群
   │
   └─► .dwoファイル（直接指定）
            │
            ▼
       DWOFilenames[]
            │
            ▼
    ┌─────────────────────┐
    │ readTargetTriple()  │ ← 最初の.dwoから
    └─────────────────────┘
            │
            ▼ Triple
    ┌─────────────────────┐
    │ TargetRegistry::    │
    │ lookupTarget()      │
    └─────────────────────┘
            │
            ▼ Target
    ┌─────────────────────┐
    │ MCオブジェクト初期化 │
    │ MCContext           │
    │ MCAsmBackend        │
    │ MCCodeEmitter       │
    └─────────────────────┘
            │
            ▼ MCStreamer
    ┌─────────────────────┐
    │ write()             │ ← .dwoファイルマージ
    │ (DWP.h)             │
    └─────────────────────┘
            │
            ▼
    ┌─────────────────────┐
    │ ToolOutputFile      │
    │ MS->finish()        │
    └─────────────────────┘
            │
            ▼
       出力ファイル (.dwp)
```

### 10.5 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| llvm/tools/llvm-dwp/llvm-dwp.cpp | メインソースファイル（311行） |
| llvm/tools/llvm-dwp/Opts.td | オプション定義（TableGen） |
| llvm/include/llvm/DWP/DWP.h | DWP処理API |
| llvm/include/llvm/DWP/DWPError.h | DWPエラー定義 |
| llvm/include/llvm/DWP/DWPStringPool.h | 文字列プール管理 |
| llvm/include/llvm/DebugInfo/DWARF/DWARFContext.h | DWARFコンテキスト |
| llvm/include/llvm/MC/MCStreamer.h | MCストリーマー |
| llvm/include/llvm/MC/MCContext.h | MCコンテキスト |
| llvm/include/llvm/MC/TargetRegistry.h | ターゲットレジストリ |

---

## 11. 使用例

### 11.1 .dwoファイルを直接マージ
```bash
llvm-dwp -o output.dwp file1.dwo file2.dwo file3.dwo
```

### 11.2 実行ファイルから.dwo参照を抽出してマージ
```bash
llvm-dwp -e program -o program.dwp
```

### 11.3 複数の実行ファイルから抽出
```bash
llvm-dwp -e program1 -e program2 -o combined.dwp
```

### 11.4 実行ファイルと.dwoファイルを組み合わせ
```bash
llvm-dwp -e program -o output.dwp additional.dwo
```

### 11.5 CUインデックスオーバーフロー時に継続
```bash
llvm-dwp -o output.dwp --continue-on-cu-index-overflow *.dwo
```

### 11.6 soft-stopモード
```bash
llvm-dwp -o output.dwp --continue-on-cu-index-overflow=soft-stop *.dwo
```

### 11.7 DWARF64文字列オフセットプロモーション有効
```bash
llvm-dwp -o output.dwp --dwarf64-str-offsets-promotion=enabled *.dwo
```

---

## 12. 改訂履歴

| 版数 | 日付 | 変更内容 | 担当者 |
|------|------|----------|--------|
| 1.0 | 2026-02-01 | 初版作成 | llvm-dwp機能設計書生成 |
