# 通知設計書 10-DirectoryWatcher Removed

## 概要

本ドキュメントは、LLVM/Clangプロジェクトの DirectoryWatcher コンポーネントが発行する Removed イベント通知の仕様を定義する。このイベントは、監視対象ディレクトリ内のファイルが削除された場合に発行される。

### 本通知の処理概要

DirectoryWatcher Removed イベントは、ファイルシステム監視機能の一部として、ディレクトリ内のファイル削除を検出し、登録されたコールバック関数に通知する機能である。

**業務上の目的・背景**：clangdやその他のLLVMツールでは、プロジェクトファイル（ソースファイル、ヘッダファイル等）の削除を検出し、インデックスからの削除やキャッシュの無効化を行う必要がある。DirectoryWatcherはこの削除検出を担当し、Removedイベントを通じて削除を通知する。

**通知の送信タイミング**：
- Linux: inotifyによるIN_DELETE, IN_MOVED_FROMイベント検出時
- macOS: FSEventsによるkFSEventStreamEventFlagItemRemovedイベント検出時、またはModifyingFlagsイベントでファイルが存在しない場合

**通知の受信者**：DirectoryWatcher::create()で登録されたReceiverコールバック関数が受信者となる。

**通知内容の概要**：EventKind::RemovedとファイルのFilenameが含まれる。

**期待されるアクション**：受信者はファイルの削除に対応した処理（インデックスからの削除、キャッシュ無効化等）を行う。

## 通知種別

内部イベント通知（DirectoryWatcher API）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（コールバック） |
| 優先度 | 中 |
| リトライ | なし |

### 送信先決定ロジック

DirectoryWatcher::create()で登録されたReceiverコールバック関数に送信される。

## 通知テンプレート

### イベント形式

| 項目 | 内容 |
|-----|------|
| イベント種別 | DirectoryWatcher::Event::EventKind::Removed |
| 形式 | DirectoryWatcher::Event構造体 |

### 構造体定義

```cpp
struct Event {
  enum class EventKind {
    Removed,
    Modified,
    WatchedDirRemoved,
    WatcherGotInvalidated
  };

  EventKind Kind;
  std::string Filename;

  Event(EventKind Kind, llvm::StringRef Filename)
      : Kind(Kind), Filename(Filename) {}
};
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| Kind | イベント種別 | EventKind::Removed固定 | Yes |
| Filename | 削除されたファイル名 | inotify/FSEventsから取得 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ファイルシステム | ファイル削除 | IN_DELETE (Linux) / kFSEventStreamEventFlagItemRemoved (macOS) | ファイル削除検出 |
| ファイルシステム | ファイル移動（出） | IN_MOVED_FROM (Linux) | ディレクトリからのファイル移動 |
| ファイルシステム | ファイル不存在 | ModifyingFlags && !getFileStatus() (macOS) | 変更イベント時にファイルが存在しない場合 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ディレクトリイベント | ファイルではなくディレクトリの削除（別イベントWatchedDirRemovedとして処理） |
| WatcherGotInvalidated後 | ウォッチャーが無効化された後のイベント |

## 処理フロー

### 送信フロー（Linux - inotify）

```mermaid
flowchart TD
    A[inotify_add_watch] --> B[InotifyPollingLoop]
    B --> C[epoll_wait]
    C --> D[inotifyイベント読み取り]
    D --> E{イベントマスク確認}
    E -->|IN_DELETE/MOVED_FROM| F[Queue.push_back Removed]
    E -->|IN_DELETE_SELF/MOVE_SELF| G[WatchedDirRemoved]
    F --> H[EventReceivingLoop]
    H --> I[Receiver コールバック呼び出し]
```

### 送信フロー（macOS - FSEvents）

```mermaid
flowchart TD
    A[FSEventStreamCreate] --> B[eventStreamCallback]
    B --> C{フラグ確認}
    C -->|ItemRemoved| D[Events.emplace_back Removed]
    C -->|ModifyingFlags| E{ファイル存在確認}
    E -->|存在しない| F[Events.emplace_back Removed]
    E -->|存在する| G[Events.emplace_back Modified]
    D --> H[Receiver コールバック呼び出し]
    F --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（ファイルシステム監視のみ）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| inotify_init1失敗 | カーネルリソース不足 | エラー返却 |
| inotify_add_watch失敗 | パーミッション不足等 | エラー返却 |
| epoll_wait失敗 | システムエラー | StopWork()でウォッチャー停止 |
| FSEventStream作成失敗 | リソース不足 | エラー返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント駆動） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（リアルタイム配信）

## セキュリティ考慮事項

- ファイル名情報のみを含み、ファイル内容は含まれない
- ローカルファイルシステムイベントのみを監視
- 監視対象ディレクトリへのアクセス権限が必要

## 備考

- プラットフォーム依存の実装（Linux: inotify, macOS: FSEvents）
- 初期スキャン前にファイルが削除された場合、Added通知なしにRemoved通知が発生する可能性がある
- IN_DELETE_SELF/IN_MOVE_SELFは監視対象ディレクトリ自体の削除/移動であり、WatchedDirRemovedとして処理される
- macOSではModifyingFlagsイベント時にファイル存在確認を行い、存在しなければRemovedとして処理

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: APIインターフェースを理解する

DirectoryWatcher APIの公開インターフェースを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | Event構造体定義（行66-100） |
| 1-2 | DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | EventKind::Removed定義（行68） |
| 1-3 | DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | 既知の不整合についてのコメント（行34-36） |

**読解のコツ**: Removedは列挙型の最初に定義されており、コメントはないがドキュメントの行34-36に「初期スキャン前の削除ではAdded通知なしにRemoved通知が発生する可能性」が記載されている。

#### Step 2: Linux実装を理解する

inotifyを使用したLinux実装を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | InotifyPollingLoop実装（行171-273） |
| 2-2 | DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | Removed判定条件（行255-257） |
| 2-3 | DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | WatchedDirRemoved判定（行258-262） |

**主要処理フロー**:
1. **行255-257**: `IN_DELETE | IN_MOVED_FROM`でRemovedイベント発行
2. **行256**: `Queue.push_back(DirectoryWatcher::Event::EventKind::Removed, Event->name)`
3. **行258-262**: `IN_DELETE_SELF | IN_MOVE_SELF`でWatchedDirRemovedイベント発行

#### Step 3: macOS実装を理解する

FSEventsを使用したmacOS実装を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | eventStreamCallback実装（行105-158） |
| 3-2 | DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | ItemRemoved判定（行134-137） |
| 3-3 | DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | ファイル不存在でRemoved（行139-141） |

**主要処理フロー**:
1. **行134-137**: `kFSEventStreamEventFlagItemRemoved`でRemovedイベント発行
2. **行135-136**: `Events.emplace_back(EventKind::Removed, llvm::sys::path::filename(Path))`
3. **行139-141**: ModifyingFlagsでファイルが存在しない場合もRemovedイベント発行

### プログラム呼び出し階層図

```
[Linux]
DirectoryWatcher::create()
    │
    └─ DirectoryWatcherLinux コンストラクタ
           │
           ├─ InotifyPollingThread
           │      └─ InotifyPollingLoop()
           │             ├─ epoll_wait()
           │             └─ inotifyイベント処理
           │                    ├─ [IN_DELETE/MOVED_FROM]
           │                    │      └─ Queue.push_back(Removed, ...)
           │                    └─ [IN_DELETE_SELF/MOVE_SELF]
           │                           └─ Queue.push_back(WatchedDirRemoved, ...)
           │
           └─ EventsReceivingThread
                  └─ EventReceivingLoop()
                         └─ Receiver(Event, IsInitial)

[macOS]
DirectoryWatcher::create()
    │
    └─ DirectoryWatcherMac コンストラクタ
           │
           └─ dispatch_queue
                  └─ eventStreamCallback()
                         ├─ [ItemRemoved]
                         │      └─ Events.emplace_back(Removed, ...)
                         └─ [ModifyingFlags && !exists]
                                └─ Events.emplace_back(Removed, ...)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

[Linux]
inotify_event ─────────▶ イベントマスク判定 ───▶ DirectoryWatcher::Event
    │                           │                        │
    │ mask                      ▼                        ▼
    │ name              IN_DELETE/MOVED_FROM?    Kind = Removed
    │                           │                Filename = name
    └───────────────────────────▶ Queue.push_back()
                                         │
                                         ▼
                                   Receiver コールバック

[macOS]
FSEventStreamEventFlags ──▶ フラグ判定 ───▶ DirectoryWatcher::Event
    │                           │                        │
    │ Flags                     ▼                        ▼
    │ Path              ItemRemoved? or          Kind = Removed
    │                   (ModifyingFlags &&       Filename = basename(Path)
    │                    !exists)?
    └───────────────────────────▶ Events.emplace_back()
                                         │
                                         ▼
                                   Receiver コールバック
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | ヘッダ | 公開API定義 |
| DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | ソース | Linux実装（inotify） |
| DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | ソース | macOS実装（FSEvents） |
| DirectoryScanner.h | `clang/lib/DirectoryWatcher/DirectoryScanner.h` | ヘッダ | 初期スキャン用ユーティリティ |
