# 通知設計書 9-DirectoryWatcher Modified

## 概要

本ドキュメントは、LLVM/Clangプロジェクトの DirectoryWatcher コンポーネントが発行する Modified イベント通知の仕様を定義する。このイベントは、監視対象ディレクトリ内のファイルが作成または変更された場合に発行される。

### 本通知の処理概要

DirectoryWatcher Modified イベントは、ファイルシステム監視機能の一部として、ディレクトリ内のファイル変更を検出し、登録されたコールバック関数に通知する機能である。

**業務上の目的・背景**：clangdやその他のLLVMツールでは、プロジェクトファイル（compile_commands.json等）やヘッダファイルの変更を検出し、自動的にインデックスの再構築やキャッシュの無効化を行う必要がある。DirectoryWatcherはこの変更検出を担当し、Modifiedイベントを通じて変更を通知する。

**通知の送信タイミング**：
- Linux: inotifyによるIN_CREATE, IN_MODIFY, IN_MOVED_TOイベント検出時
- macOS: FSEventsによるkFSEventStreamEventFlagItemCreated, kFSEventStreamEventFlagItemModified, kFSEventStreamEventFlagItemRenamedイベント検出時
- 初期スキャン時に既存ファイルを検出した場合

**通知の受信者**：DirectoryWatcher::create()で登録されたReceiverコールバック関数が受信者となる。

**通知内容の概要**：EventKind::ModifiedとファイルのFilenameが含まれる。

**期待されるアクション**：受信者はファイルの変更に対応した処理（インデックス更新、キャッシュ無効化等）を行う。

## 通知種別

内部イベント通知（DirectoryWatcher API）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（コールバック） |
| 優先度 | 中 |
| リトライ | なし |

### 送信先決定ロジック

DirectoryWatcher::create()で登録されたReceiverコールバック関数に送信される。

## 通知テンプレート

### イベント形式

| 項目 | 内容 |
|-----|------|
| イベント種別 | DirectoryWatcher::Event::EventKind::Modified |
| 形式 | DirectoryWatcher::Event構造体 |

### 構造体定義

```cpp
struct Event {
  enum class EventKind {
    Removed,
    Modified,
    WatchedDirRemoved,
    WatcherGotInvalidated
  };

  EventKind Kind;
  std::string Filename;

  Event(EventKind Kind, llvm::StringRef Filename)
      : Kind(Kind), Filename(Filename) {}
};
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| Kind | イベント種別 | EventKind::Modified固定 | Yes |
| Filename | 変更されたファイル名 | inotify/FSEventsから取得 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ファイルシステム | ファイル作成 | IN_CREATE (Linux) / kFSEventStreamEventFlagItemCreated (macOS) | 新規ファイル作成 |
| ファイルシステム | ファイル変更 | IN_MODIFY (Linux) / kFSEventStreamEventFlagItemModified (macOS) | 既存ファイル変更 |
| ファイルシステム | ファイル移動（入） | IN_MOVED_TO (Linux) / kFSEventStreamEventFlagItemRenamed (macOS) | ディレクトリへのファイル移動 |
| 初期スキャン | 既存ファイル検出 | ディレクトリスキャン完了時 | 初期状態同期 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ディレクトリイベント | ファイルではなくディレクトリの変更（macOS: !kFSEventStreamEventFlagItemIsFile） |
| WatcherGotInvalidated後 | ウォッチャーが無効化された後のイベント |

## 処理フロー

### 送信フロー（Linux - inotify）

```mermaid
flowchart TD
    A[inotify_add_watch] --> B[InotifyPollingLoop]
    B --> C[epoll_wait]
    C --> D[inotifyイベント読み取り]
    D --> E{イベントマスク確認}
    E -->|IN_CREATE/MODIFY/MOVED_TO| F[Queue.push_back Modified]
    E -->|IN_DELETE/MOVED_FROM| G[Queue.push_back Removed]
    F --> H[EventReceivingLoop]
    H --> I[Receiver コールバック呼び出し]
```

### 送信フロー（macOS - FSEvents）

```mermaid
flowchart TD
    A[FSEventStreamCreate] --> B[eventStreamCallback]
    B --> C{フラグ確認}
    C -->|ItemIsFile & ModifyingFlags| D{ファイル存在確認}
    D -->|存在する| E[Events.emplace_back Modified]
    D -->|存在しない| F[Events.emplace_back Removed]
    E --> G[Receiver コールバック呼び出し]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（ファイルシステム監視のみ）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| inotify_init1失敗 | カーネルリソース不足 | エラー返却 |
| inotify_add_watch失敗 | パーミッション不足等 | エラー返却 |
| epoll_wait失敗 | システムエラー | StopWork()でウォッチャー停止 |
| FSEventStream作成失敗 | リソース不足 | エラー返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント駆動） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（リアルタイム配信）

## セキュリティ考慮事項

- ファイル名情報のみを含み、ファイル内容は含まれない
- ローカルファイルシステムイベントのみを監視
- 監視対象ディレクトリへのアクセス権限が必要

## 備考

- プラットフォーム依存の実装（Linux: inotify, macOS: FSEvents）
- サブディレクトリの再帰的監視は非サポート
- 重複イベントが発生する可能性がある（FSEventsのcoalescing動作）
- 初期スキャン時にIsInitial=trueでイベントが配信される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: APIインターフェースを理解する

DirectoryWatcher APIの公開インターフェースを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | Event構造体定義（行66-100） |
| 1-2 | DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | EventKind enum定義（行67-91） |
| 1-3 | DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | create()静的メソッド（行106-111） |

**読解のコツ**: EventKindのModifiedは「Content of a file was modified」とコメントされている。

#### Step 2: Linux実装を理解する

inotifyを使用したLinux実装を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | InotifyPollingLoop実装（行171-273） |
| 2-2 | DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | Modified判定条件（行252-254） |
| 2-3 | DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | inotify_add_watch設定（行333-340） |

**主要処理フロー**:
1. **行328-340**: inotify_add_watchでIN_CREATE | IN_DELETE | IN_MODIFY | IN_MOVED_FROM | IN_MOVED_TO等を監視
2. **行252-254**: `IN_CREATE | IN_MOVED_TO | IN_MODIFY`でModifiedイベント発行
3. **行253**: `Queue.push_back(DirectoryWatcher::Event::EventKind::Modified, Event->name)`

#### Step 3: macOS実装を理解する

FSEventsを使用したmacOS実装を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | eventStreamCallback実装（行105-158） |
| 3-2 | DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | ModifyingFileEvents定義（行101-103） |
| 3-3 | DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | Modified判定条件（行138-146） |

**主要処理フロー**:
1. **行101-103**: ModifyingFileEventsはkFSEventStreamEventFlagItemCreated | ItemRenamed | ItemModified
2. **行138-146**: ModifyingFileEventsの場合、ファイル存在確認してModified/Removed判定
3. **行143-144**: `Events.emplace_back(EventKind::Modified, llvm::sys::path::filename(Path))`

### プログラム呼び出し階層図

```
[Linux]
DirectoryWatcher::create()
    │
    └─ DirectoryWatcherLinux コンストラクタ
           │
           ├─ InotifyPollingThread
           │      └─ InotifyPollingLoop()
           │             ├─ epoll_wait()
           │             └─ inotifyイベント処理
           │                    └─ Queue.push_back(Modified, ...)
           │
           └─ EventsReceivingThread
                  └─ EventReceivingLoop()
                         └─ Receiver(Event, IsInitial)

[macOS]
DirectoryWatcher::create()
    │
    └─ DirectoryWatcherMac コンストラクタ
           │
           └─ dispatch_queue
                  └─ eventStreamCallback()
                         └─ Receiver(Events, IsInitial)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

[Linux]
inotify_event ─────────▶ イベントマスク判定 ───▶ DirectoryWatcher::Event
    │                           │                        │
    │ mask                      ▼                        ▼
    │ name              IN_CREATE/MODIFY?        Kind = Modified
    │                           │                Filename = name
    └───────────────────────────▶ Queue.push_back()
                                         │
                                         ▼
                                   Receiver コールバック

[macOS]
FSEventStreamEventFlags ──▶ フラグ判定 ───▶ DirectoryWatcher::Event
    │                           │                        │
    │ Flags                     ▼                        ▼
    │ Path              ModifyingFlags?          Kind = Modified
    │                   ファイル存在確認         Filename = basename(Path)
    └───────────────────────────▶ Events.emplace_back()
                                         │
                                         ▼
                                   Receiver コールバック
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| DirectoryWatcher.h | `clang/include/clang/DirectoryWatcher/DirectoryWatcher.h` | ヘッダ | 公開API定義 |
| DirectoryWatcher-linux.cpp | `clang/lib/DirectoryWatcher/linux/DirectoryWatcher-linux.cpp` | ソース | Linux実装（inotify） |
| DirectoryWatcher-mac.cpp | `clang/lib/DirectoryWatcher/mac/DirectoryWatcher-mac.cpp` | ソース | macOS実装（FSEvents） |
| DirectoryScanner.h | `clang/lib/DirectoryWatcher/DirectoryScanner.h` | ヘッダ | 初期スキャン用ユーティリティ |
