# アーキテクチャ設計書

## 概要

本ドキュメントは、Next.js フレームワークのアプリケーションアーキテクチャを記述する設計書である。Next.js は React ベースのフルスタック Web フレームワークであり、pnpm モノレポとして構成されている。TypeScript（Node.js）による JavaScript 層と、Rust（SWC / Turbopack）によるネイティブコンパイラ層の二層構造を持ち、開発サーバー・プロダクションサーバー・ビルドシステムの三つの主要サブシステムから成る。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| Node.js ランタイム | プロセス実行 | サーバーサイドレンダリング、ビルド実行、開発サーバー |
| Webpack | JavaScript API | モジュールバンドリング（従来バンドラー） |
| Turbopack (Rust) | N-API バインディング | 高速モジュールバンドリング（次世代バンドラー） |
| SWC (Rust) | N-API バインディング | TypeScript/JSX コンパイル、コード変換 |
| Rspack | JavaScript API | モジュールバンドリング（代替バンドラー、実験的） |
| React (canary) | npm パッケージ | UI レンダリング（Server Components / Client Components） |
| OpenTelemetry | npm パッケージ | 分散トレーシング |
| Vercel プラットフォーム | デプロイ API | エッジランタイム、サーバーレスデプロイ |

## レイヤー構成

### アーキテクチャスタイル

Next.js は**モジュラーモノリス + プラグインアーキテクチャ**を採用している。明確なクリーンアーキテクチャや DDD のレイヤー分離ではなく、機能ドメインごとにモジュールを分割し、サーバー・クライアント・ビルドの三つの主要ランタイムコンテキストに沿ってコードを組織化している。Route Module パターンによりルーティングとレンダリングの関心事が分離されている。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| CLI | コマンドラインインターフェース、プロセス起動 | `src/cli/next-dev.ts`, `src/cli/next-build.ts`, `src/cli/next-start.ts` |
| Router Server | HTTP リクエスト受付、ルーティング、ミドルウェア実行 | `src/server/lib/router-server.ts`, `src/server/lib/router-utils/` |
| Server Runtime | リクエスト処理、レンダリング実行、キャッシュ管理 | `src/server/base-server.ts`, `src/server/next-server.ts`, `src/server/dev/next-dev-server.ts` |
| Route Modules | ルート種別ごとのレンダリングロジック | `src/server/route-modules/app-page/`, `src/server/route-modules/app-route/`, `src/server/route-modules/pages/`, `src/server/route-modules/pages-api/` |
| App Render | React Server Components レンダリング | `src/server/app-render/app-render.tsx`, `src/server/app-render/create-component-tree.tsx` |
| Client Runtime | ブラウザ側ルーティング、ハイドレーション | `src/client/app-router.tsx`, `src/client/components/`, `src/client/router.ts` |
| Build System | コンパイル、バンドリング、最適化 | `src/build/index.ts`, `src/build/webpack-config.ts`, `src/build/compiler.ts` |
| Shared | 共有ユーティリティ、定数、ルーター共通ロジック | `src/shared/lib/constants.ts`, `src/shared/lib/router/` |
| Native (Rust) | SWC 変換、Turbopack バンドラー | `crates/next-core/`, `crates/next-custom-transforms/`, `turbopack/` |

### レイヤー間の依存関係ルール

- **CLI** は Server Runtime および Build System を起動するが、直接レンダリングロジックには関与しない
- **Router Server** は Server Runtime に依存し、リクエストを適切な Route Module にディスパッチする
- **Server Runtime** (`base-server.ts`) は抽象基底クラスであり、`next-server.ts`（プロダクション）と `next-dev-server.ts`（開発）がこれを継承する
- **Route Modules** は Server Runtime から呼び出されるが、互いに直接依存しない
- **Client Runtime** は Shared レイヤーのみに依存し、Server Runtime を直接参照しない
- **Build System** は Server/Client 双方のコードを解析するが、ランタイムからは独立している
- **Shared** は他のどのレイヤーからも参照可能だが、Shared 自身は他レイヤーに依存しない
- **Native (Rust)** は N-API を通じて Build System および Server Runtime から呼び出される

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Server | HTTP サーバー、SSR/RSC レンダリング、API 処理 | `packages/next/src/server/` |
| Client | クライアントサイドルーティング、ハイドレーション、UI コンポーネント | `packages/next/src/client/` |
| Build | コンパイル、バンドリング、静的解析、最適化 | `packages/next/src/build/` |
| CLI | コマンドラインツール、プロセス管理 | `packages/next/src/cli/` |
| Shared | 共通ユーティリティ、ルーター定義、定数 | `packages/next/src/shared/` |
| Telemetry | 使用状況の匿名収集 | `packages/next/src/telemetry/` |
| Trace | パフォーマンストレーシング | `packages/next/src/trace/` |
| Native SWC | Rust によるコード変換 | `crates/next-custom-transforms/`, `crates/next-core/` |
| Turbopack | Rust による次世代バンドラー | `turbopack/` |
| Font | Web フォント最適化 | `packages/font/` |
| Third Parties | サードパーティスクリプト統合 | `packages/third-parties/` |
| ESLint Plugin | Next.js 固有の lint ルール | `packages/eslint-plugin-next/` |
| create-next-app | プロジェクトスキャフォールディング | `packages/create-next-app/` |

### パッケージ構造

```
next.js/
├── packages/
│   ├── next/                          # メインフレームワーク（npmパッケージ "next"）
│   │   └── src/
│   │       ├── cli/                   # CLIエントリーポイント
│   │       ├── server/                # サーバーランタイム
│   │       │   ├── app-render/        # App Router RSCレンダリング
│   │       │   ├── dev/               # 開発サーバー（HMR等）
│   │       │   ├── lib/               # サーバーユーティリティ
│   │       │   │   ├── router-server.ts      # ルーターサーバー
│   │       │   │   ├── router-utils/         # ルーティングユーティリティ
│   │       │   │   ├── incremental-cache/    # ISRキャッシュ
│   │       │   │   ├── cache-handlers/       # キャッシュハンドラー
│   │       │   │   └── trace/                # トレーシング
│   │       │   ├── route-modules/     # ルートモジュール（app-page, app-route, pages, pages-api）
│   │       │   ├── route-matchers/    # ルートマッチング
│   │       │   ├── route-matcher-managers/  # ルートマッチャー管理
│   │       │   ├── normalizers/       # パス正規化
│   │       │   ├── web/               # Edge Runtime / Web API対応
│   │       │   ├── response-cache/    # レスポンスキャッシュ
│   │       │   ├── use-cache/         # "use cache" ディレクティブ
│   │       │   ├── after/             # after() API
│   │       │   ├── async-storage/     # AsyncLocalStorage管理
│   │       │   ├── instrumentation/   # インストルメンテーションフック
│   │       │   └── mcp/              # MCP (Model Context Protocol) サーバー
│   │       ├── client/                # クライアントランタイム
│   │       │   ├── components/        # App Routerコンポーネント
│   │       │   │   ├── app-router.tsx         # App Router本体
│   │       │   │   ├── router-reducer/        # Routerステート管理
│   │       │   │   ├── segment-cache/         # セグメントキャッシュ
│   │       │   │   └── navigation.ts          # ナビゲーションAPI
│   │       │   ├── app-dir/           # App Directory用クライアント
│   │       │   ├── dev/               # 開発時オーバーレイ
│   │       │   └── legacy/            # Pages Router互換
│   │       ├── build/                 # ビルドシステム
│   │       │   ├── webpack/           # Webpack設定・プラグイン
│   │       │   ├── webpack-build/     # Webpackビルド実行
│   │       │   ├── turbopack-build/   # Turbopackビルド実行
│   │       │   ├── swc/               # SWCバインディング
│   │       │   ├── templates/         # コード生成テンプレート
│   │       │   └── analysis/          # 静的解析
│   │       ├── shared/                # サーバー/クライアント共通
│   │       │   └── lib/
│   │       │       ├── constants.ts   # 共通定数
│   │       │       ├── router/        # ルーター共通ロジック
│   │       │       └── i18n/          # 国際化
│   │       ├── lib/                   # 汎用ユーティリティ
│   │       ├── telemetry/             # テレメトリ
│   │       ├── trace/                 # トレーシング
│   │       └── api/                   # パブリックAPI型定義
│   ├── create-next-app/               # CLIスキャフォールディングツール
│   ├── next-swc/                      # SWCネイティブバインディング
│   ├── font/                          # next/font 実装
│   ├── third-parties/                 # サードパーティスクリプト
│   ├── eslint-plugin-next/            # ESLintプラグイン
│   ├── eslint-config-next/            # ESLint設定
│   ├── next-mdx/                      # MDXサポート
│   ├── next-bundle-analyzer/          # バンドルサイズ分析
│   ├── next-codemod/                  # コードマイグレーション
│   └── next-env/                      # 環境変数ローダー
├── crates/                            # Rustクレート（Next.js固有）
│   ├── next-core/                     # Turbopack統合コア
│   ├── next-custom-transforms/        # SWCカスタム変換
│   ├── next-api/                      # Rust側APIバインディング
│   ├── next-build/                    # Rust側ビルドロジック
│   ├── next-napi-bindings/            # N-APIバインディング
│   └── wasm/                          # WebAssemblyビルド
├── turbopack/                         # Turbopackバンドラー（git subtree）
│   └── crates/
│       ├── turbopack/                 # バンドラーコア
│       ├── turbopack-core/            # コアインターフェース
│       ├── turbopack-ecmascript/      # ECMAScript処理
│       ├── turbopack-css/             # CSS処理
│       ├── turbo-tasks/               # 増分計算フレームワーク
│       └── ...                        # その他多数のクレート
├── test/                              # テストスイート
├── examples/                          # サンプルアプリケーション
└── scripts/                           # ビルド・メンテナンススクリプト
```

### コンポーネント依存関係

**サーバー継承階層：**
- `BaseServer`（`base-server.ts`）: 抽象基底クラス。ルーティング、キャッシュ、レンダリングの共通ロジックを提供
  - `NextNodeServer`（`next-server.ts`）: Node.js プロダクションサーバー。ファイルシステムアクセス、Incremental Cache を実装
    - `NextDevServer`（`dev/next-dev-server.ts`）: 開発サーバー。HMR、On-Demand コンパイル、Dev Overlay を追加

**ルートモジュール体系：**
- `RouteModule`（`route-modules/route-module.ts`）: 抽象基底クラス
  - `AppPageRouteModule`: App Router ページレンダリング
  - `AppRouteRouteModule`: App Router API Route（Route Handlers）
  - `PagesRouteModule`: Pages Router ページレンダリング
  - `PagesAPIRouteModule`: Pages Router API Route

**バンドラー抽象化：**
- Webpack / Turbopack / Rspack を `Bundler` 型で抽象化（`src/lib/bundler.ts`）
- ビルド設定は `next.config.js`（`NextConfig`型）で一元管理

## ミドルウェア構成

### データベース

Next.js フレームワーク自体はデータベースを内蔵しない。アプリケーション開発者が任意のデータベースを選択する設計となっている。

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| ファイルシステム | Node.js `fs` | - | ビルドキャッシュ、静的ファイル配信、ISRキャッシュのデフォルトストレージ |

### キャッシュ

| ミドルウェア | 実装箇所 | 用途 | TTL |
| --- | --- | --- | --- |
| Incremental Cache | `src/server/lib/incremental-cache/` | ISR (Incremental Static Regeneration) のページキャッシュ | `revalidate` 設定値に依存 |
| Response Cache | `src/server/response-cache/` | レンダリング結果のインメモリキャッシュ | リクエストライフサイクル |
| File System Cache | `src/server/lib/incremental-cache/file-system-cache.ts` | ISR キャッシュのファイルシステムバックエンド | `revalidate` 設定値に依存 |
| Memory Cache | `src/server/lib/incremental-cache/memory-cache.external.ts` | ISR キャッシュのインメモリバックエンド | LRU + TTL |
| Cache Handlers | `src/server/lib/cache-handlers/` | カスタムキャッシュハンドラー（プラグイン可能） | カスタム |
| use cache | `src/server/use-cache/` | `"use cache"` ディレクティブによるデータキャッシュ | `cacheLife` 設定値に依存 |
| Segment Cache | `src/client/components/segment-cache/` | クライアント側ルートセグメントキャッシュ | ナビゲーションライフサイクル |
| LRU Cache | `src/server/lib/lru-cache.ts` | 汎用 LRU キャッシュ | キー数ベース |

### メッセージキュー

Next.js フレームワーク自体はメッセージキューを内蔵しない。HMR（Hot Module Replacement）は WebSocket を通じたリアルタイム通知で実現される。

| ミドルウェア | 実装箇所 | 用途 |
| --- | --- | --- |
| WebSocket (HMR) | `src/server/dev/hot-middleware.ts` | 開発時 Hot Module Replacement 通知 |

### 検索エンジン

該当なし。Next.js フレームワーク自体は検索エンジンを内蔵しない。

### その他ミドルウェア

| ミドルウェア | 実装箇所 | 用途 |
| --- | --- | --- |
| SWC (Rust) | `crates/next-custom-transforms/`, `packages/next/src/build/swc/` | TypeScript / JSX コンパイル、コード最適化 |
| Turbopack (Rust) | `turbopack/`, `crates/next-core/` | 次世代モジュールバンドラー（増分計算ベース） |
| Webpack | `src/build/webpack-config.ts`, `src/build/webpack/` | 従来モジュールバンドラー |
| styled-jsx | 内蔵 | CSS-in-JS スコープスタイリング |
| OpenTelemetry | `src/server/lib/trace/tracer.ts` | 分散トレーシング（オプショナル） |
| Image Optimizer | `src/server/image-optimizer.ts` | 画像の自動最適化・リサイズ |

## データフロー

### リクエスト処理の流れ

**プロダクションサーバー（`next start`）：**

1. `next-start.ts` が `start-server.ts` を通じて HTTP サーバーを起動
2. `router-server.ts` が全リクエストを受信し、ファイルシステムチェック（`setupFsCheck`）を実行
3. `resolve-routes.ts` でルーティング解決（リダイレクト、リライト、ミドルウェア実行を含む）
4. ミドルウェアが定義されている場合、Edge Runtime で実行
5. `base-server.ts` の `run()` メソッドでリクエストコンテキストを構築
6. `RouteMatcherManager` が URL に対応する `RouteModule` を特定
7. 該当する Route Module（AppPage / AppRoute / Pages / PagesAPI）がレンダリングを実行
8. App Router の場合、`app-render.tsx` が React Server Components をストリーミングレンダリング
9. `ResponseCache` / `IncrementalCache` が ISR キャッシュを確認・更新
10. レスポンスをクライアントに送信

**開発サーバー（`next dev`）：**

1. `next-dev.ts` が子プロセスとして `start-server.ts` を fork
2. `router-server.ts` が `DevBundlerService` を初期化
3. リクエスト受信時、`NextDevServer` が On-Demand コンパイルを実行
4. HotReloader（Turbopack または Webpack）がモジュール変更を監視
5. 変更検出時、WebSocket 経由でクライアントに HMR 更新を通知

### 非同期処理の流れ

1. **ISR (Incremental Static Regeneration)**: バックグラウンドで静的ページを再生成。`IncrementalCache` が stale-while-revalidate パターンで古いキャッシュを配信しつつ再生成をトリガー
2. **after() API**: `src/server/after/` でリクエスト完了後の非同期タスクを実行（テレメトリ送信、ログ記録等）
3. **Turbopack 増分計算**: `turbo-tasks` フレームワークにより、ファイル変更時に影響を受けるモジュールのみを再計算
4. **Server Actions**: クライアントからサーバー関数を RPC 的に呼び出し、`action-handler.ts` で処理

### データ永続化の流れ

1. **ビルド成果物**: `next build` が `.next/` ディレクトリにマニフェスト（`build-manifest.json`, `prerender-manifest.json`, `routes-manifest.json` 等）とバンドルを出力
2. **ISR キャッシュ**: `FileSystemCache` が `.next/cache/` にページの HTML / RSC Payload / JSON を保存。`revalidate` 期間経過後にバックグラウンド再生成
3. **テレメトリ**: `src/telemetry/storage.ts` がローカルファイルシステムに匿名使用統計を保存

## 横断的関心事

### 認証・認可

Next.js フレームワーク自体は認証・認可の仕組みを内蔵しないが、以下の拡張ポイントを提供する。

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| Middleware | `src/server/web/` (Edge Runtime) | 全リクエストに対する認証ガード |
| Route Handlers | `src/server/route-modules/app-route/` | API エンドポイントでの認証 |
| Server Actions | `src/server/app-render/action-handler.ts` | サーバー関数呼び出し時の認可チェック |
| CSRF Protection | `src/server/app-render/csrf-protection.ts` | Server Actions への CSRF 攻撃防止 |
| Preview Mode / Draft Mode | `src/server/async-storage/draft-mode-provider.ts` | CMS プレビュー用の認証済みアクセス |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | `src/build/output/log.ts`（`Log.info`, `Log.warn`, `Log.error`） | 標準出力 / 標準エラー出力 |
| リクエストログ | `src/server/dev/log-requests.ts` | 標準出力（開発時） |
| トレーシング | `src/server/lib/trace/tracer.ts`（OpenTelemetry 統合） | 外部トレースコレクタ |
| テレメトリ | `src/telemetry/storage.ts` | ローカルファイルシステム |
| Server Action ログ | `src/server/dev/server-action-logger.ts` | 標準出力（開発時） |

### エラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| 404 Not Found | `UNDERSCORE_NOT_FOUND_ROUTE` による専用ページ | カスタム `not-found.tsx` または デフォルト 404 |
| 500 Internal Server Error | `create-error-handler.tsx` でキャッチ | カスタム `error.tsx` / `global-error.tsx` またはデフォルト 500 |
| ビルドエラー | `WrappedBuildError` でラップ | ビルド失敗メッセージ |
| デコードエラー | `DecodeError` クラス | 400 Bad Request |
| ミドルウェアエラー | Edge Runtime サンドボックスでキャッチ | 500 |
| React レンダリングエラー | Error Boundary (`error-boundary.tsx`) | フォールバック UI |
| インストルメンテーションエラー | `onRequestError` フック | カスタムエラーレポーティング |

### トランザクション管理

Next.js フレームワーク自体はトランザクション管理を内蔵しない。`AsyncLocalStorage` を用いたリクエストコンテキスト管理が、トランザクションスコープに相当する機能を提供する。

| 範囲 | 管理方式 | 用途 |
| --- | --- | --- |
| リクエストスコープ | `AsyncLocalStorage`（`src/server/async-storage/`） | リクエストごとのストア（cookies, headers 等）の隔離 |
| WorkUnit スコープ | `work-unit-async-storage` | レンダリング作業単位の管理 |
| Action スコープ | `action-async-storage` | Server Action 実行コンテキスト |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| テンプレートメソッドパターン | Server 継承階層 | `BaseServer` が共通フローを定義し、`NextNodeServer` / `NextDevServer` がフックメソッドをオーバーライド |
| ストラテジーパターン | バンドラー選択 | `Bundler` 型による Webpack / Turbopack / Rspack の切り替え |
| プロバイダーパターン | Route Matcher | `RouteMatcherProvider` がルート定義を供給し、`RouteMatcherManager` が管理 |
| ファクトリーパターン | Route Module 生成 | ルート種別に応じた `RouteModule` サブクラスのインスタンス化 |
| ノーマライザーチェーン | URL 正規化 | `src/server/normalizers/` で複数のノーマライザーをチェーンして URL を変換 |
| 関心の分離 | サーバー/クライアント境界 | React Server Components による自動的なサーバー/クライアントコード分離 |
| 増分計算 | Turbopack | `turbo-tasks` フレームワークによる依存関係追跡と最小限の再計算 |
| AsyncLocalStorage | リクエストスコープ | `src/server/async-storage/` で明示的なコンテキスト受け渡し不要なリクエストスコープ管理 |

### コーディング規約

- **言語**: TypeScript（サーバー/クライアント/ビルド）、Rust（SWC/Turbopack）
- **パッケージマネージャー**: pnpm 9.6.0
- **ビルドオーケストレーション**: Turborepo（`turbo.json`）
- **リンター**: ESLint 9、Prettier、ast-grep、alex（言語チェック）
- **テスト**: Jest（単体・統合）、Playwright（E2E）
- **Node.js バージョン**: >= 20.9.0
- **React バージョン**: canary チャネル（Server Components サポート）
- **ファイル拡張子**: `.ts` / `.tsx`（TypeScript）、`.external.ts`（外部バンドル対象）
- **マニフェスト管理**: ビルド成果物は JSON マニフェスト（`build-manifest.json`, `routes-manifest.json` 等）で管理
- **コンパイル出力**: `packages/next/src/` のミラー構造で `packages/next/dist/` に出力

## 備考

- Next.js は App Router（推奨）と Pages Router（レガシー）の二つのルーティングシステムを並行サポートしている。App Router は React Server Components、Streaming SSR、Server Actions などの新機能を利用可能
- Turbopack は Webpack の後継バンドラーとして開発中であり、`next dev --turbopack` で有効化される。プロダクションビルドでも利用可能
- `"use cache"` ディレクティブは実験的機能として `src/server/use-cache/` に実装されている
- MCP (Model Context Protocol) サーバーが `src/server/mcp/` に実装されており、AI ツールとの統合を提供する
- Edge Runtime（`src/server/web/`）により、Vercel Edge Functions やその他のエッジ環境でのサーバーサイド実行をサポート
