# Next.js コードリーディングガイドライン

## はじめに

このガイドラインは、Next.jsのコードベースを効率的に理解するための手引きです。
TypeScript/JavaScript およびRustに精通していないエンジニアでも、段階的に学習できるよう構成されています。

Next.jsはVercelが開発するReactベースのフルスタックWebフレームワークです。本リポジトリ（`next.js-canary`）はNext.jsフレームワーク本体のモノレポであり、npmパッケージ `next`（v16.2.0-canary.24）として公開されるコアパッケージを中心に、関連ツール群とRustで実装されたバンドラ（Turbopack）を含んでいます。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、TypeScript/JavaScript およびRustの基本構文と概念を解説します。

本プロジェクトは主に以下の2言語で構成されています。

| 言語 | 用途 | 主要ディレクトリ |
|------|------|-----------------|
| TypeScript/JavaScript | フレームワーク本体（サーバー、クライアント、ビルド、CLI） | `packages/next/src/` |
| Rust | バンドラ（Turbopack）、SWCトランスフォーム | `crates/`, `turbopack/` |

### 1.1 プログラム構造

**TypeScript側：** Next.jsのTypeScriptコードは、ES Modulesベースのモジュールシステムを採用しています。各ファイルは単一の関心事を持つモジュールとして構成されます。

```typescript
// ファイル: packages/next/src/cli/next-start.ts:1-8
#!/usr/bin/env node

// 環境初期化が最初に来る
if (!process.env.NEXT_PRIVATE_START_TIME) {
  process.env.NEXT_PRIVATE_START_TIME = Date.now().toString()
}

import '../server/lib/cpu-profile'
import { saveCpuProfile } from '../server/lib/cpu-profile'
import { startServer } from '../server/lib/start-server'
```

**Rust側：** Rustコードは `crates/` 以下にCargo workspaceとして構成されています。各クレートは `Cargo.toml` で依存関係を管理し、`src/lib.rs` をエントリーポイントとします。

```toml
# ファイル: crates/next-core/Cargo.toml:1-7
[package]
name = "next-core"
version = "0.1.0"
description = "TBD"
license = "MIT"
edition = "2024"
```

### 1.2 データ型と変数

TypeScriptでは、型エイリアス（`type`）、インターフェース（`interface`）、列挙型（`enum`）、const assertions（`as const`）が多用されます。

```typescript
// ファイル: packages/next/src/server/route-kind.ts:1-25
export const enum RouteKind {
  PAGES = 'PAGES',
  PAGES_API = 'PAGES_API',
  APP_PAGE = 'APP_PAGE',
  APP_ROUTE = 'APP_ROUTE',
  IMAGE = 'IMAGE',
}
```

型定義では `type` と `interface` を使い分けます。`interface` は拡張可能なオブジェクト型に、`type` はユニオン型やユーティリティ型に使われます。

```typescript
// ファイル: packages/next/src/server/route-definitions/route-definition.ts:1-18
import type { RouteKind } from '../route-kind'

export interface RouteDefinition<K extends RouteKind = RouteKind> {
  readonly kind: K
  readonly bundlePath: string
  readonly filename: string
  readonly page: string
  readonly pathname: string
}
```

### 1.3 制御構造

TypeScriptでは標準的な `if/else`、`switch`、`for...of`、`while` に加え、以下のパターンが頻出します。

- **Optional chaining (`?.`)** と **Nullish coalescing (`??`)**: null/undefinedの安全なアクセス
- **非同期制御**: `async/await`、`Promise`ベースの処理

```typescript
// ファイル: packages/next/src/cli/next-dev.ts:74-77
const CHILD_EXIT_TIMEOUT_MS = parseInt(
  process.env.NEXT_EXIT_TIMEOUT_MS ?? '100',
  10
)
```

### 1.4 関数/メソッド定義

関数定義はアロー関数式と `function` 宣言の両方が使われます。非同期関数には `async` キーワードが付与されます。

```typescript
// ファイル: packages/next/src/cli/next-start.ts:42-46
const nextStart = async (options: NextStartOptions, directory?: string) => {
  const dir = getProjectDir(directory)
  const hostname = options.hostname
  const inspect = options.inspect
  const port = options.port
```

型パラメータ付きのジェネリック関数も頻出します。

```typescript
// ファイル: packages/next/src/server/route-modules/route-module.ts:75-80
export interface RouteModuleOptions<
  D extends RouteDefinition = RouteDefinition,
  U = unknown,
> {
  readonly definition: Readonly<D>
  readonly userland: Readonly<U>
```

### 1.5 モジュール/インポート

ES Modules (`import/export`) を使用します。以下のパターンが特徴的です。

**型のみのインポート（`import type`）:** ランタイムには影響しない型情報のインポートに使用されます。

```typescript
// ファイル: packages/next/src/server/next-server.ts:5-8
import type { CacheFs } from '../shared/lib/utils'
import {
  DecodeError,
  PageNotFoundError,
  MiddlewareNotFoundError,
} from '../shared/lib/utils'
```

**相対パスインポート:** プロジェクト内のモジュールは相対パスで参照されます。

**compiled モジュール:** 外部依存は `next/dist/compiled/` 以下にプリコンパイルされたものを使う場合があります。

```typescript
// ファイル: packages/next/src/server/lib/start-server.ts:19-21
import Watchpack from 'next/dist/compiled/watchpack'
import * as Log from '../../build/output/log'
import setupDebug from 'next/dist/compiled/debug'
```

---

## 2. プロジェクト固有の概念

> このセクションでは、当プロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### App Router と Pages Router

Next.jsには2つのルーティングシステムが共存しています。

| 概念 | 説明 | ルートKind |
|------|------|-----------|
| **Pages Router** | `pages/` ディレクトリベースのルーティング（従来方式） | `PAGES`, `PAGES_API` |
| **App Router** | `app/` ディレクトリベースのルーティング（React Server Components対応） | `APP_PAGE`, `APP_ROUTE` |

これは `RouteKind` 列挙型（`packages/next/src/server/route-kind.ts`）で定義され、サーバー内部のルーティング分岐に使われます。

#### React Server Components (RSC)

App Routerでは、コンポーネントがデフォルトでサーバー側でレンダリングされます。クライアント側で動作させるには `'use client'` ディレクティブを明示します。

```typescript
// ファイル: packages/next/src/client/link.tsx:1
'use client'
```

RSC関連のヘッダー定数は `app-router-headers.ts` に集約されています。

```typescript
// ファイル: packages/next/src/client/components/app-router-headers.ts で定義
RSC_HEADER, NEXT_RSC_UNION_QUERY, NEXT_ROUTER_PREFETCH_HEADER,
NEXT_ROUTER_SEGMENT_PREFETCH_HEADER, NEXT_URL, NEXT_ROUTER_STATE_TREE_HEADER
```

#### AsyncLocalStorage パターン

サーバーサイドでリクエストスコープのデータ（cookies、headers等）を管理するために、Node.jsの `AsyncLocalStorage` を活用した「ストア」パターンが使われています。

```typescript
// ファイル: packages/next/src/server/app-render/work-async-storage.external.ts
// WorkStore: ビルド時・リクエスト処理時の共通ストア

// ファイル: packages/next/src/server/app-render/work-unit-async-storage.external.ts
// RequestStore, PrerenderStoreModernRuntime: リクエスト固有のストア
```

### 2.2 プロジェクト独自のパターン

#### サーバー継承階層

サーバーは抽象クラスの継承で構成されています。

```
BaseServer (base-server.ts)
  |-- NextNodeServer (next-server.ts)      # 本番サーバー
       |-- DevServer (dev/next-dev-server.ts) # 開発サーバー
```

`BaseServer` が共通のリクエスト処理ロジックを持ち、`NextNodeServer` がNode.js環境向けの実装を、`DevServer` がHMRやオンデマンドコンパイル機能を追加します。

#### Route Module パターン

ルートハンドリングは「Route Module」パターンで抽象化されています。

```
packages/next/src/server/route-modules/
  route-module.ts          # 基底クラス RouteModule
  app-page/                # App Router のページ
  app-route/               # App Router のAPIルート
  pages/                   # Pages Router のページ
  pages-api/               # Pages Router のAPIルート
```

各Route Moduleは `RouteDefinition`（ルート定義）と `userland`（ユーザーが書いたコード）を受け取り、統一的なインターフェースでリクエストを処理します。

#### Normalizer パターン

URLパスの正規化は `Normalizer` インターフェースを実装した多数のクラスで行われます。

```typescript
// ファイル: packages/next/src/server/normalizers/normalizer.ts:1-3
export interface Normalizer {
  normalize(pathname: string): string
}
```

`PathnameNormalizer` は `match` メソッドを追加で持ち、特定のパスパターンに一致するかどうかの判定も行います。

```typescript
// ファイル: packages/next/src/server/normalizers/request/pathname-normalizer.ts:1-6
export interface PathnameNormalizer extends Normalizer {
  match(pathname: string): boolean
  normalize(pathname: string, matched?: boolean): string
}
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `kebab-case.ts` | 標準的なソースファイル | `base-server.ts`, `route-kind.ts` |
| `*.test.ts` | テストファイル（同ディレクトリ配置） | `config.test.ts`, `csrf-protection.test.ts` |
| `*.external.ts` | 外部向け（バンドル境界を越える）モジュール | `work-async-storage.external.ts` |
| `*.shared-runtime.ts` | クライアント/サーバー共有ランタイムモジュール | `app-router-context.shared-runtime.ts` |
| `*.compiled.ts` | プリコンパイル済みモジュールへの参照 | `module.compiled.ts` |
| `*.d.ts` | TypeScript型定義ファイル | `ReactDOMServerPages.d.ts` |
| `next-*.ts` (CLI) | CLIサブコマンドのエントリーポイント | `next-dev.ts`, `next-build.ts`, `next-start.ts` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `UPPER_SNAKE_CASE` | 定数（マニフェスト名、フェーズ名等） | `PAGES_MANIFEST`, `PHASE_PRODUCTION_BUILD` |
| `Base*` | 抽象基底クラス | `BaseServer`, `BaseNextRequest` |
| `*Provider` | ルートマッチャーなどのデータ提供クラス | `AppPageRouteMatcherProvider` |
| `*Normalizer` | パス正規化処理クラス | `LocaleRouteNormalizer`, `PathnameNormalizer` |
| `*Store` | AsyncLocalStorageベースのリクエストスコープストア | `WorkStore`, `RequestStore` |
| `*Handler` | リクエスト処理関数 | `WorkerRequestHandler`, `WorkerUpgradeHandler` |
| `*Adapter` | 異なるインターフェース間の変換 | `HeadersAdapter`, `RequestCookiesAdapter` |
| `is*` / `has*` | Boolean返却のユーティリティ関数 | `isBot()`, `isDynamicRoute()`, `hasNextSupport` |
| `get*` | 値取得のユーティリティ関数 | `getHostname()`, `getRequestMeta()` |
| `create*` | ファクトリ関数 | `createRequestStoreForRender()`, `createWorkStore()` |
| `load*` | ファイルやモジュールの読み込み | `loadComponents()`, `loadEnvConfig()` |
| `Dev*` | 開発環境専用のクラス | `DevServer`, `DevRouteMatcherManager` |

### 3.3 プログラム分類一覧

| 分類 | 説明 | ファイル命名パターン |
|------|------|---------------------|
| CLI コマンド | `next dev`, `next build`, `next start` 等 | `packages/next/src/cli/next-*.ts` |
| サーバーランタイム | リクエスト処理、レンダリング | `packages/next/src/server/**/*.ts` |
| クライアントランタイム | ブラウザ側のルーティング、ハイドレーション | `packages/next/src/client/**/*.ts` |
| ビルドツール | Webpack/Turbopack設定、最適化 | `packages/next/src/build/**/*.ts` |
| 共有コード | サーバー/クライアント両方で使用 | `packages/next/src/shared/**/*.ts` |
| Rustクレート | SWCトランスフォーム、Turbopack | `crates/*/src/**/*.rs` |
| テスト | E2E、統合、ユニットテスト | `test/{e2e,integration,production,development,unit}/**` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
next.js-canary/
├── packages/                    # npmパッケージ群（pnpm monorepo）
│   ├── next/                    # メインパッケージ（next npm）
│   │   ├── src/                 # ソースコード
│   │   │   ├── api/             # next/headers, next/cookies 等の公開API
│   │   │   ├── bin/             # CLI バイナリエントリーポイント
│   │   │   ├── build/           # ビルドツール（webpack設定、最適化等）
│   │   │   ├── cli/             # CLIサブコマンド（dev, build, start等）
│   │   │   ├── client/          # クライアントサイドランタイム
│   │   │   ├── compiled/        # プリコンパイル済み外部依存
│   │   │   ├── export/          # 静的エクスポート機能
│   │   │   ├── lib/             # 汎用ユーティリティ
│   │   │   ├── server/          # サーバーランタイム（最重要）
│   │   │   ├── shared/          # サーバー/クライアント共有コード
│   │   │   ├── telemetry/       # テレメトリ収集
│   │   │   └── trace/           # トレーシング
│   │   └── dist/                # コンパイル済み出力（src/の鏡像）
│   ├── create-next-app/         # create-next-app CLI
│   ├── next-swc/                # ネイティブRustバインディング
│   ├── eslint-plugin-next/      # Next.js用ESLintルール
│   ├── font/                    # next/font 実装
│   └── third-parties/           # サードパーティスクリプト統合
├── crates/                      # Rust クレート（Next.js SWCバインディング）
│   ├── next-core/               # Next.js Turbopack統合コア
│   ├── next-custom-transforms/  # SWCカスタムトランスフォーム
│   ├── next-api/                # Turbopack API
│   ├── next-build/              # Turbopackビルド処理
│   └── next-napi-bindings/      # Node.js N-APIバインディング
├── turbopack/                   # Turbopackバンドラ（git subtree）
│   └── crates/                  # Turbopack関連Rustクレート群
├── test/                        # テストスイート
│   ├── e2e/                     # E2Eテスト
│   ├── development/             # 開発モードテスト
│   ├── production/              # 本番ビルドテスト
│   ├── integration/             # 統合テスト
│   ├── unit/                    # ユニットテスト
│   └── lib/                     # テストユーティリティ
├── examples/                    # サンプルNext.jsアプリケーション
├── docs/                        # ドキュメント
└── scripts/                     # ビルド・メンテナンススクリプト
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `packages/next/src/server/` | サーバーランタイム（最も変更が多い） | `base-server.ts`, `next-server.ts`, `app-render/app-render.tsx` |
| `packages/next/src/client/` | クライアントサイドランタイム | `link.tsx`, `app-router.tsx`, `components/navigation.ts` |
| `packages/next/src/build/` | ビルドツール | `index.ts`, `webpack-config.ts`, `compiler.ts` |
| `packages/next/src/cli/` | CLIエントリーポイント | `next-dev.ts`, `next-build.ts`, `next-start.ts` |
| `packages/next/src/shared/` | 共有ユーティリティ | `lib/constants.ts`, `lib/router/`, `lib/page-path/` |
| `packages/next/src/lib/` | 汎用ヘルパー | `file-exists.ts`, `find-pages-dir.ts`, `worker.ts` |
| `crates/next-custom-transforms/` | SWCカスタムトランスフォーム | `src/transforms/`, `src/lib.rs` |
| `test/e2e/` | E2Eテスト | `app-dir/`, `404-page-router/` 等 |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Next.jsは **レイヤードアーキテクチャ** と **プラグインアーキテクチャ** を組み合わせた構成です。CLIからの入力に応じて、ビルドパイプラインまたはサーバーランタイムが起動し、Route Moduleパターンにより各種ルートを統一的に処理します。

```
┌─────────────────────────────────────────────────────────┐
│                      CLI Layer                          │
│   next-dev.ts │ next-build.ts │ next-start.ts           │
├─────────────────────────────────────────────────────────┤
│                   Server Layer                          │
│  ┌──────────────────────────────────────────────┐       │
│  │ Router Server (router-server.ts)             │       │
│  │   ├── Route Resolution                       │       │
│  │   ├── Middleware Execution                   │       │
│  │   └── Request Dispatch                       │       │
│  ├──────────────────────────────────────────────┤       │
│  │ Base Server → Next Node Server → Dev Server  │       │
│  │   ├── Route Matcher Manager                  │       │
│  │   ├── Route Modules (App/Pages)              │       │
│  │   └── Response Cache / ISR                   │       │
│  └──────────────────────────────────────────────┘       │
├─────────────────────────────────────────────────────────┤
│                  Rendering Layer                        │
│  App Render (RSC)  │  Pages Render (SSR/SSG)            │
├─────────────────────────────────────────────────────────┤
│                    Build Layer                          │
│  Webpack Config │ Turbopack │ SWC Transforms            │
├─────────────────────────────────────────────────────────┤
│                 Shared / Utilities                      │
│  Constants │ Router Utils │ Page Path Utils              │
└─────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| CLI | コマンドライン引数の解析、プロセス管理 | `src/cli/next-dev.ts`, `src/cli/next-start.ts` |
| Router Server | HTTPリクエストのルーティング、ミドルウェア実行 | `src/server/lib/router-server.ts` |
| Base Server | リクエスト処理の共通ロジック（キャッシュ、マッチング） | `src/server/base-server.ts` |
| Route Modules | ルート種別ごとの処理ロジック | `src/server/route-modules/app-page/`, `pages/` |
| App Render | React Server Components のレンダリング | `src/server/app-render/app-render.tsx` |
| Build | ビルドパイプライン、バンドラ設定 | `src/build/index.ts`, `src/build/webpack-config.ts` |
| Client | ブラウザ側ルーティング、ハイドレーション | `src/client/app-router.tsx`, `src/client/link.tsx` |
| Shared | サーバー/クライアント共有の定数・ユーティリティ | `src/shared/lib/constants.ts`, `src/shared/lib/router/` |

### 5.3 データフロー

**開発時のリクエスト処理フロー:**

```
ブラウザ → HTTP Request
  → start-server.ts (HTTPサーバー起動)
    → router-server.ts (ルーティング)
      → resolve-routes (ルート解決)
        → ミドルウェア実行（あれば）
        → DevServer.handleRequest()
          → Route Module.handle()
            → app-render.tsx (App Router) or render.tsx (Pages Router)
              → React レンダリング
  → HTTP Response → ブラウザ
```

**ビルド時のフロー:**

```
next build コマンド
  → cli/next-build.ts
    → build/index.ts
      → ページ探索 (findPagesDir)
      → Webpack/Turbopack コンパイル
        → SWC トランスフォーム (Rust crates)
      → 静的ページ生成 (SSG/ISR)
      → マニフェストファイル出力
  → .next/ ディレクトリに出力
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

Next.jsには3つの主要エントリーポイントがあります。

| コマンド | エントリーポイント | 遷移先 |
|---------|-------------------|--------|
| `next dev` | `src/cli/next-dev.ts` | `src/server/dev/next-dev-server.ts` |
| `next build` | `src/cli/next-build.ts` | `src/build/index.ts` |
| `next start` | `src/cli/next-start.ts` | `src/server/next-server.ts` |

**開発サーバー起動の流れ:**

```typescript
// ファイル: packages/next/src/cli/next-dev.ts:42-58
export type NextDevOptions = {
  disableSourceMaps: boolean
  inspect?: DebugAddress | true
  turbo?: boolean
  turbopack?: boolean
  webpack?: boolean
  port: number
  hostname?: string
  experimentalHttps?: boolean
  // ...
}
```

CLIオプションを解析し、子プロセスとして開発サーバーを起動します。子プロセスはIPC経由で `distDir` などの情報を親プロセスに伝達します。

### 6.2 ビジネスロジック

Next.jsの「ビジネスロジック」はサーバーサイドのリクエスト処理とレンダリングです。

**サーバー階層:**

```typescript
// ファイル: packages/next/src/server/base-server.ts:1-50
// BaseServer: 全サーバーの基底クラス
// - ルートマッチング
// - レスポンスキャッシュ
// - ミドルウェア実行
// - 国際化(i18n)処理

// ファイル: packages/next/src/server/next-server.ts:68
import BaseServer from './base-server'
// NextNodeServer extends BaseServer
// - Node.js固有の実装（ファイルシステム、HTTP処理）
// - IncrementalCache (ISR)
// - MiddlewareManifest読み込み

// ファイル: packages/next/src/server/dev/next-dev-server.ts:37
import Server, { WrappedBuildError } from '../next-server'
// DevServer extends NextNodeServer
// - HMR (Hot Module Replacement)
// - オンデマンドコンパイル
// - エラーオーバーレイ
```

**App Router レンダリング:**

```typescript
// ファイル: packages/next/src/server/app-render/app-render.tsx:1-10
// RSC (React Server Components) のレンダリングエンジン
// - FlightRouterState の構築
// - Fizz ストリーミングレンダリング
// - プリレンダリング (SSG/ISR)
// - 動的レンダリング
```

### 6.3 データアクセス

Next.jsはデータベースに直接アクセスするのではなく、以下のデータ層を持ちます。

**マニフェストファイル:** ビルド時に生成される JSON ファイルで、ランタイムのルート解決やアセット管理に使用されます。

```typescript
// ファイル: packages/next/src/shared/lib/constants.ts:7-11
export const COMPILER_NAMES = {
  client: 'client',
  server: 'server',
  edgeServer: 'edge-server',
} as const
```

主要なマニフェスト定数（`packages/next/src/shared/lib/constants.ts` で定義）:

| 定数 | 説明 |
|------|------|
| `PAGES_MANIFEST` | Pages Router のページ一覧 |
| `APP_PATHS_MANIFEST` | App Router のパス一覧 |
| `BUILD_MANIFEST` | ビルドアセット情報 |
| `PRERENDER_MANIFEST` | プリレンダリング/ISR設定 |
| `MIDDLEWARE_MANIFEST` | ミドルウェア設定 |
| `ROUTES_MANIFEST` | リライト/リダイレクト/ヘッダー設定 |

**IncrementalCache (ISR):**

ISR（Incremental Static Regeneration）の実装は `src/server/lib/incremental-cache/` にあり、静的ページのキャッシュと再生成を管理します。

### 6.4 ユーティリティ/共通機能

**`src/shared/lib/`:** サーバー/クライアント双方で使用されるユーティリティ。

```typescript
// ファイル: packages/next/src/shared/lib/constants.ts:70-76
export const PHASE_EXPORT = 'phase-export'
export const PHASE_ANALYZE = 'phase-analyze'
export const PHASE_PRODUCTION_BUILD = 'phase-production-build'
export const PHASE_PRODUCTION_SERVER = 'phase-production-server'
export const PHASE_DEVELOPMENT_SERVER = 'phase-development-server'
export const PHASE_TEST = 'phase-test'
```

**`src/lib/`:** ビルドツールやCLIで使用される汎用ヘルパー。ファイルシステム操作、依存関係チェック、TypeScript設定などを含みます。

**`src/server/normalizers/`:** URLパスの正規化ロジック群。ロケール除去、ベースパス除去、トレイリングスラッシュ処理など。

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| クラス継承 | サーバーの機能拡張 | 高 | `base-server.ts` → `next-server.ts` → `next-dev-server.ts` |
| インターフェース + ジェネリクス | 型安全なRoute Module | 高 | `route-module.ts`, `route-definition.ts` |
| AsyncLocalStorage | リクエストスコープのデータ管理 | 高 | `work-async-storage.external.ts` |
| Normalizer チェーン | URL正規化パイプライン | 中 | `src/server/normalizers/` |
| Manifest 定数 | ビルド成果物の参照 | 高 | `src/shared/lib/constants.ts` |
| Provider パターン | ルートマッチャーへのデータ供給 | 中 | `src/server/route-matcher-providers/` |
| External / Compiled モジュール | バンドル境界の明示 | 中 | `*.external.ts`, `src/compiled/` |

### 各パターンの詳細

#### パターン1: サーバー継承階層

**目的:** 本番サーバーと開発サーバーで共通のリクエスト処理ロジックを共有しつつ、開発環境特有の機能（HMR、オンデマンドコンパイル）を追加する。

**実装例:**
```typescript
// ファイル: packages/next/src/server/next-server.ts:68
import BaseServer from './base-server'
// class NextNodeServer extends BaseServer { ... }

// ファイル: packages/next/src/server/dev/next-dev-server.ts:37
import Server, { WrappedBuildError } from '../next-server'
// class DevServer extends Server { ... }
```

**解説:** `BaseServer` はルートマッチング、レスポンスキャッシュ、ミドルウェア処理などの共通ロジックを定義します。`NextNodeServer` はNode.js環境固有のファイルシステムアクセスやHTTP処理を追加し、`DevServer` はさらにHMRやエラーオーバーレイを追加します。

#### パターン2: Route Module パターン

**目的:** App Router/Pages Router の各ルート種別を統一的なインターフェースで処理する。

**実装例:**
```typescript
// ファイル: packages/next/src/server/route-modules/route-module.ts:75-80
export interface RouteModuleOptions<
  D extends RouteDefinition = RouteDefinition,
  U = unknown,
> {
  readonly definition: Readonly<D>
  readonly userland: Readonly<U>
}
```

**解説:** `definition` はルートのメタデータ（kind, pathname, bundlePath）を、`userland` はユーザーが書いたエクスポート（`default export`, `getServerSideProps` 等）を保持します。これにより、ルートの種類に関わらず統一的なインターフェースでリクエスト処理が可能になります。

#### パターン3: External モジュールパターン

**目的:** Webpack/Turbopackのバンドル境界を明示し、サーバーとクライアントで異なるバンドルに含まれるモジュールを管理する。

**実装例:**
```
work-async-storage.external.ts    # バンドル境界の外部モジュール
work-async-storage-instance.ts    # 実際のインスタンス
```

**解説:** `.external.ts` サフィックスが付いたファイルは、バンドラがバンドル境界の外側にあるモジュールとして扱うことを示します。これにより、サーバーサイドのAsyncLocalStorageインスタンスがクライアントバンドルに含まれることを防ぎます。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（CLI → Server → Router）
2. 処理の流れを追跡（関数の呼び出し関係を追う）
3. データの変換を確認（リクエスト → レスポンス）
4. 最終的な出力を確認（HTML/JSON/Stream）

### 8.2 フロー追跡の実例

#### 例1: `next start` による本番サーバー起動

**概要:** `next start` コマンドがHTTPサーバーを起動し、リクエストを受け付けるまでの流れ。

**処理フロー:**
```
next-start.ts → startServer() → router-server.ts → NextNodeServer
```

**詳細な追跡:**

1. **CLIエントリーポイント** (`packages/next/src/cli/next-start.ts:42`)
   ```typescript
   const nextStart = async (options: NextStartOptions, directory?: string) => {
     const dir = getProjectDir(directory)
     const hostname = options.hostname
     const port = options.port
   ```
   コマンドライン引数を解析し、ポートやホスト名を取得します。

2. **サーバー起動** (`packages/next/src/server/lib/start-server.ts:1-6`)
   ```typescript
   import './cpu-profile'
   import { getNetworkHost } from '../../lib/get-network-host'
   import '../next'
   import '../require-hook'
   ```
   CPU プロファイリングの初期化、require hookの設定を行った後、HTTPサーバーを作成します。

3. **ルーターサーバー初期化** (`packages/next/src/server/lib/router-server.ts:64`)
   ```typescript
   const debug = setupDebug('next:router-server:main')
   ```
   `router-server.ts` の `initialize()` 関数で設定を読み込み、ファイルシステムチェック、ルート解決ロジックを準備します。

4. **リクエスト処理** 各リクエストは router-server.ts を経由して、ルート解決後に適切な Route Module にディスパッチされます。

#### 例2: App Router ページリクエストの処理

**概要:** ブラウザから `/dashboard` にアクセスした際の処理フロー。

**処理フロー:**
```
HTTP Request → Router Server → Route Resolution → App Page Route Module → app-render.tsx → RSC Render → HTML Stream
```

**詳細な追跡:**

1. **ルーティング解決** (`packages/next/src/server/lib/router-server.ts`)
   リクエストURLからミドルウェアの実行、リライト/リダイレクトのチェックを行います。

2. **Route Module マッチング** (`packages/next/src/server/base-server.ts`)
   `RouteMatcherManager` がリクエストパスに一致する Route Module を検索します。`RouteKind.APP_PAGE` に一致する場合、App Page Route Module が選択されます。

3. **RSC レンダリング** (`packages/next/src/server/app-render/app-render.tsx`)
   React Server Components をレンダリングし、Flight データとHTMLをストリーミングで出力します。

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか（`src/cli/next-*.ts`）
- [ ] サーバー階層を把握したか（BaseServer → NextNodeServer → DevServer）
- [ ] ルートマッチングの仕組みを理解したか（RouteKind, RouteDefinition, RouteModule）
- [ ] データの変換ポイントを確認したか（Request → Render → Response）
- [ ] エラーハンドリングを確認したか（error-boundary, create-error-handler）
- [ ] キャッシュの動作を確認したか（ResponseCache, IncrementalCache）

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `readme.md` - プロジェクト概要
2. `CLAUDE.md` - 開発ガイド（ディレクトリ構造、ビルド方法、テスト方法が網羅的）
3. `contributing.md` - コントリビューションガイド
4. `docs/code-to-docs/アーキテクチャ設計書/` - アーキテクチャ全体像

#### 特定機能を理解したい場合
1. `docs/code-to-docs/機能一覧/` - 機能一覧から対象機能を特定
2. `docs/code-to-docs/機能設計書/` - 対象機能の設計詳細
3. 該当ソースコード（`packages/next/src/server/` 以下が最も多い）

#### 改修作業を行う場合
1. `CLAUDE.md` の「Fast Local Development」セクション - 開発環境のセットアップ
2. 対象機能の設計書
3. `test/` 以下の関連テストケース
4. `docs/code-to-docs/単体テストケース一覧/` - テスト方針の確認

#### サーバーサイドの変更を行う場合
1. `packages/next/src/server/base-server.ts` - サーバーの基底クラスを理解
2. `packages/next/src/server/route-modules/` - Route Module パターンを理解
3. 対象ルート種別のRoute Moduleを修正

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `readme.md` | プロジェクト概要 | 初回参画時 |
| `CLAUDE.md` | 開発ガイド（ビルド、テスト、ディレクトリ構造） | 開発作業開始時 |
| `contributing.md` | コントリビューション手順 | PR作成前 |
| `UPGRADING.md` | バージョンアップガイド | メジャーバージョン更新時 |
| `errors/` | エラーメッセージとその説明 | エラー調査時 |
| `examples/` | 機能別のサンプルアプリケーション | 機能の使い方確認時 |
| `docs/code-to-docs/` | 自動生成された各種設計書 | 設計レビュー時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `*.external.ts` と通常の `.ts` ファイルの違いは何ですか？
A: `.external.ts` サフィックスは、Webpack/Turbopackのバンドル境界を越えるモジュールを示します。これにより、サーバーサイドでのみ利用されるモジュール（AsyncLocalStorage等）がクライアントバンドルに含まれることを防ぎます。参照: `packages/next/src/server/app-render/work-async-storage.external.ts`

#### Q: `*.shared-runtime.ts` ファイルとは何ですか？
A: サーバーとクライアントの両方のランタイムで共有されるモジュールです。React Contextの型定義やフック用のコンテキストなどが含まれます。参照: `packages/next/src/shared/lib/app-router-context.shared-runtime.ts`

#### Q: `compiled/` ディレクトリの役割は何ですか？
A: 外部依存パッケージをプリコンパイルし、node_modulesに依存しない形で含めるためのディレクトリです。これにより、Next.jsのインストールサイズが削減され、依存関係の競合が回避されます。

#### Q: Pages Router と App Router のコードはどこで分岐しますか？
A: `RouteKind`（`src/server/route-kind.ts`）による分岐が基本です。`PAGES`/`PAGES_API` は Pages Router、`APP_PAGE`/`APP_ROUTE` は App Router として処理されます。レンダリングは App Router が `app-render/app-render.tsx`、Pages Router が `render.tsx` を使用します。

#### Q: テストの実行方法がわかりません。
A: `CLAUDE.md` に詳細が記載されています。基本は以下のコマンドです:
- 開発モードテスト: `pnpm test-dev-turbo test/path/to/test.ts`
- 本番モードテスト: `pnpm test-start-turbo test/path/to/test.ts`
- ユニットテスト: `pnpm test-unit`
- 高速ローカルテスト: `NEXT_SKIP_ISOLATE=1 NEXT_TEST_MODE=dev pnpm testonly test/path/to/test.ts`

### デバッグのヒント

- **デバッグログ有効化:** `DEBUG=next:*` 環境変数を設定すると、Next.js内部のデバッグログが出力されます。
- **テレメトリ無効化:** `NEXT_TELEMETRY_DISABLED=1` でテスト時のテレメトリ送信を無効化できます。
- **Node.jsインスペクター:** `next dev --inspect` でChrome DevToolsでのデバッグが可能です。
- **ソースコード変更後:** `pnpm --filter=next build` でリビルドが必要です。Turbopack (Rust) コードの変更後は `pnpm swc-build-native` が必要です。
- **ウォッチモード:** `pnpm --filter=next dev` でファイル変更を監視し自動リビルドできます。

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| App Router | `app/` ディレクトリベースのルーティングシステム（React Server Components対応） |
| Pages Router | `pages/` ディレクトリベースの従来のルーティングシステム |
| RSC (React Server Components) | サーバーサイドでレンダリングされるReactコンポーネント |
| ISR (Incremental Static Regeneration) | 静的ページをリクエスト時に段階的に再生成する仕組み |
| SSG (Static Site Generation) | ビルド時に静的HTMLを生成する仕組み |
| SSR (Server-Side Rendering) | リクエスト時にサーバーでHTMLを生成する仕組み |
| SWC | Rustベースの超高速JavaScriptコンパイラ |
| Turbopack | Rustベースの次世代バンドラ（Webpack後継） |
| Flight | RSCのワイヤーフォーマット（サーバーからクライアントへのデータ転送形式） |
| Fizz | Reactのストリーミングサーバーレンダラー |
| Route Module | ルート種別ごとのリクエスト処理を抽象化するモジュール |
| Middleware | リクエスト処理前に実行されるEdgeランタイム上の処理 |
| Manifest | ビルド時に生成されるメタデータJSONファイル |
| N-API | Node.jsのネイティブアドオンAPI（Rustバインディングで使用） |
| Monorepo | 複数パッケージを単一リポジトリで管理する構成 |
| pnpm | 高速なNode.jsパッケージマネージャー |
| Turborepo | モノレポのビルドオーケストレーションツール |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `packages/next/src/server/base-server.ts` | サーバー基底クラス | ルートマッチング、キャッシュ、ミドルウェア |
| `packages/next/src/server/next-server.ts` | 本番サーバー | Node.js固有実装、ISR |
| `packages/next/src/server/dev/next-dev-server.ts` | 開発サーバー | HMR、オンデマンドコンパイル |
| `packages/next/src/server/lib/router-server.ts` | ルーターサーバー | リクエストルーティング |
| `packages/next/src/server/app-render/app-render.tsx` | App Routerレンダラー | RSCレンダリング |
| `packages/next/src/server/route-kind.ts` | ルート種別定義 | PAGES, APP_PAGE等のenum |
| `packages/next/src/server/route-modules/route-module.ts` | Route Module基底 | RouteModuleOptions |
| `packages/next/src/build/index.ts` | ビルドエントリーポイント | ビルドパイプライン |
| `packages/next/src/cli/next-dev.ts` | `next dev` CLI | 開発サーバー起動 |
| `packages/next/src/cli/next-build.ts` | `next build` CLI | ビルド実行 |
| `packages/next/src/cli/next-start.ts` | `next start` CLI | 本番サーバー起動 |
| `packages/next/src/shared/lib/constants.ts` | 共有定数 | マニフェスト名、フェーズ定数 |
| `packages/next/src/client/link.tsx` | Link コンポーネント | クライアントサイドナビゲーション |
| `packages/next/src/client/components/navigation.ts` | ナビゲーションフック | useSearchParams, usePathname等 |
| `crates/next-custom-transforms/` | SWCカスタムトランスフォーム | JSX変換、最適化 |
| `turbopack/` | Turbopackバンドラ | Rustベースのバンドラ |

### C. 参考資料

- [Next.js 公式ドキュメント](https://nextjs.org/docs)
- [TypeScript 公式ドキュメント](https://www.typescriptlang.org/docs/)
- [React Server Components RFC](https://github.com/reactjs/rfcs/blob/main/text/0188-server-components.md)
- [SWC 公式サイト](https://swc.rs/)
- [Turbopack ドキュメント](https://turbo.build/pack)
- [pnpm ドキュメント](https://pnpm.io/)
- [Turborepo ドキュメント](https://turbo.build/repo)
- [Rust プログラミング言語](https://doc.rust-lang.org/book/)
