# セキュリティ設計書

## 概要

本ドキュメントは、Next.jsフレームワーク（`packages/next`）のセキュリティ設計を記述する。Next.jsはWebアプリケーションフレームワークであり、アプリケーション開発者がセキュアなWebアプリケーションを構築するための基盤機能を提供している。本書では、フレームワーク自体が内蔵するセキュリティ機構（CSRF対策、暗号化、入出力エスケープ、画像最適化におけるSSRF防御など）を対象とする。なお、Next.jsはフレームワークであるため、認証・認可の具体的な実装はアプリケーション開発者に委ねられており、フレームワーク側では直接的な認証・認可機構を持たない。

## 認証設計

### 認証方式

Next.jsフレームワーク自体には、ユーザー認証の仕組みは組み込まれていない。認証はアプリケーション開発者がMiddleware（`middleware.ts`）やAPI Routes、Server Actionsなどを通じて独自に実装する設計となっている。

フレームワークが提供する認証関連の支援機能は以下の通り:

- **Middleware**: `packages/next/src/server/web/spec-extension/response.ts` の `NextResponse` を通じて、リクエストの前段で認証チェックを挟むことが可能
- **Draft Mode（旧Preview Mode）**: プレビュー用のセッショントークンをCookieベースで管理する仕組み（`packages/next/src/server/api-utils/node/try-get-preview-data.ts`）
- **`experimental.authInterrupts`**: 認証割り込みに関する実験的機能フラグ

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| Draft Mode Cookie: `httpOnly` | `true` | `packages/next/src/server/api-utils/node/api-resolver.ts` |
| Draft Mode Cookie: `sameSite` | 本番: `none` / 開発: `lax` | 環境に応じて切り替え |
| Draft Mode Cookie: `secure` | 本番: `true` / 開発: `false` | HTTPS環境でのみCookieを送信 |
| Draft Mode Cookie: `path` | `/` | サイト全体に適用 |
| セッション有効期限 | アプリケーション側で設定 | `maxAge`オプションで指定可能 |
| セッション固定化対策 | フレームワーク側に専用機構なし | アプリケーション開発者が実装 |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| （フレームワーク未定義） | - | Next.jsはロールベースの認可機構を持たない。アプリケーション開発者がMiddlewareやServer Components内で実装する |

### アクセス制御

Next.jsフレームワークは、以下のレベルでアクセス制御の基盤を提供している:

- **Middleware（`middleware.ts`）**: リクエスト単位でのアクセス制御をアプリケーション開発者が実装可能。`NextResponse.redirect()` や `NextResponse.rewrite()` でルーティングを制御
- **Server Actions Origin検証**: Server Actionsの実行時に`origin`ヘッダーと`host`/`x-forwarded-host`ヘッダーを比較し、クロスサイトからの不正な呼び出しを防止（`packages/next/src/server/app-render/action-handler.ts`）
- **開発モードにおけるクロスサイトリクエストブロック**: 開発サーバーの内部エンドポイント（`/_next/*`、`/__nextjs*`）への不正なクロスサイトリクエストを検出しブロック（`packages/next/src/server/lib/router-utils/block-cross-site.ts`）
- **画像最適化のリモートパターン制御**: `remotePatterns` 設定により、許可されたドメインからの画像のみを最適化処理（`packages/next/src/server/image-optimizer.ts`）

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | フレームワーク自体はHTTPS終端を行わない。デプロイ環境（リバースプロキシ、CDN等）で対応。`secure`フラグ付きCookieにより本番環境ではHTTPSが前提 |
| HSTS | フレームワーク内にHSTSヘッダーの自動設定機構はない。`next.config.js` の `headers` 設定、またはMiddlewareでアプリケーション開発者が設定可能 |
| Server Actions キャッシュ防止 | Server Actions応答時に `Cache-Control: no-cache, no-store, max-age=0, must-revalidate` を自動設定（`packages/next/src/server/app-render/action-handler.ts`） |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| Server Actions Bound Args | AES-256-GCM（Web Crypto API `crypto.subtle`）。ランダムIV（16バイト）を使用。鍵は環境変数 `NEXT_SERVER_ACTIONS_ENCRYPTION_KEY` または自動生成（`packages/next/src/server/app-render/encryption-utils.ts`） |
| Preview Data / Draft Mode | AES-256-GCM（Node.js `crypto`モジュール）。PBKDF2（100,000イテレーション、SHA-512）で鍵導出。ランダムIV（16バイト）、ランダムSalt（64バイト）使用（`packages/next/src/server/crypto-utils.ts`） |
| Subresource Integrity（SRI） | SHA-256 / SHA-384 / SHA-512（設定可能）。ビルド時にアセットのハッシュ値を算出し、マニフェストに記録（`packages/next/src/build/webpack/plugins/subresource-integrity-plugin.ts`） |

### 機密情報管理

- **Server Actions暗号鍵**: 環境変数 `NEXT_SERVER_ACTIONS_ENCRYPTION_KEY` で外部から注入可能。未設定の場合は `crypto.subtle.generateKey` で自動生成し、`.rscinfo` ファイルにキャッシュ（有効期限14日間）。ビルド間で鍵は維持される（`packages/next/src/server/app-render/encryption-utils-server.ts`）
- **React Taint API**: `experimental.taint` フラグにより、機密データがクライアントに漏洩するのを防ぐReactの実験的機能を有効化可能（`packages/next/src/server/config-shared.ts`）
- **`server-only` パッケージ**: サーバー専用モジュール（暗号化処理等）がクライアントバンドルに含まれないことをビルド時に保証

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | HTMLエスケープ: `&`, `>`, `<`, `\u2028`, `\u2029` をUnicodeエスケープシーケンスに変換（`packages/next/src/server/htmlescape.ts`）。JavaScript URLサニタイズ: `javascript:` プロトコルを正規表現で検出しブロック（`packages/next/src/client/lib/javascript-url.ts`）。CSP nonce対応: `Content-Security-Policy` ヘッダーからnonce値を抽出し `<script>` タグに付与（`packages/next/src/server/app-render/get-script-nonce-from-header.tsx`）。Trusted Types対応: `nextjs` ポリシーによるDOM操作の制御（`packages/next/src/client/trusted-types.ts`）。nonce値のHTMLエスケープ文字検証による注入防止 |
| SQLインジェクション | フレームワーク自体にデータベースアクセス層はない。アプリケーション開発者が使用するORM/クエリビルダーにて対応 |
| CSRF | Server Actionsにおける `origin` ヘッダーと `host`/`x-forwarded-host` ヘッダーの比較検証。`allowedOrigins` 設定によるワイルドカードパターンマッチング（単一レベル `*` および多レベル `**`）。大文字小文字を区別しないRFC 1035準拠の比較（`packages/next/src/server/app-render/action-handler.ts`、`packages/next/src/server/app-render/csrf-protection.test.ts`）。開発サーバーの `allowedDevOrigins` 設定による開発時のクロスオリジン制御（`packages/next/src/server/lib/router-utils/block-cross-site.ts`） |
| SSRF（画像最適化経由） | 画像最適化エンドポイントでリモートURLのDNS解決結果を検査し、プライベートIPアドレスへのアクセスを拒否（`packages/next/src/server/image-optimizer.ts`、`packages/next/src/server/is-private-ip.ts`）。`remotePatterns` / `localPatterns` による許可リスト方式のURL制御 |
| ヘッダーインジェクション | Server Actions処理時に禁止ヘッダー（`connection`, `expect`, `content-length`, `set-cookie`）をフィルタリング（`packages/next/src/server/lib/server-ipc/utils.ts`） |
| リクエストサイズ制限 | Server Actionsの `bodySizeLimit` 設定、API Routesの `bodyParser.sizeLimit` 設定によるリクエストボディサイズの制限 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| リクエストログ | `logging.incomingRequests` 設定による受信リクエストのロギング。正規表現パターンによるフィルタリング対応（`packages/next/src/server/dev/log-requests.ts`） | フレームワーク未規定（実行環境依存） |
| フェッチログ | `logging.fetches` 設定によるサーバーサイドfetchのロギング。HMRリフレッシュ時のログ出力制御対応 | フレームワーク未規定（実行環境依存） |
| Server Functionsログ | `logging.serverFunctions` 設定によるServer Function実行のロギング（デフォルト: 有効） | フレームワーク未規定（実行環境依存） |
| CSRF検出ログ | Server Actionsにおけるオリジン不一致時の `console.error` 出力。クロスサイトリクエストブロック時の `warnOnce` 出力 | フレームワーク未規定（実行環境依存） |

## 備考

- **X-Powered-By ヘッダー**: デフォルトで `X-Powered-By: Next.js` ヘッダーが付与される（`poweredByHeader: true`）。セキュリティ上の観点から、`next.config.js` で `poweredByHeader: false` に設定して無効化することが推奨される（`packages/next/src/server/send-payload.ts`）
- **crossOrigin設定**: `next.config.js` の `crossOrigin` オプション（`'anonymous'` | `'use-credentials'`）により、`<script>` タグの `crossorigin` 属性を制御可能
- **ETag生成**: `generateEtags` 設定によりレスポンスのETag自動生成が可能（デフォルト: 有効）。キャッシュ制御とコンテンツ整合性検証に活用
- **RSCリクエストヘッダー検証**: `experimental.validateRSCRequestHeaders` 設定により、RSCリクエストのヘッダーとキャッシュバスティングパラメータの整合性を検証可能
- **SVG画像の取り扱い**: `dangerouslyAllowSVG` 設定はデフォルトで無効。SVGはスクリプト実行が可能であるため、有効化時は `contentSecurityPolicy` の設定と併用が推奨される
- **本フレームワークの位置付け**: Next.jsはWebアプリケーションフレームワークであり、認証・認可・セッション管理・データベースセキュリティなどの多くの側面はアプリケーション開発者の責任範囲となる。フレームワークはこれらの実装を支援する基盤（Middleware、Server Components、Server Actions等）を提供する
