# バッチ設計書 12-code-freeze

## 概要

本ドキュメントは、Next.jsプロジェクトにおけるコードフリーズ管理バッチ `code-freeze` の設計仕様を定義する。

### 本バッチの処理概要

GitHubのcanaryブランチのブランチ保護ルールを更新し、コードフリーズの有効化・無効化を切り替えるバッチである。

**業務上の目的・背景**：Next.jsのリリースプロセスにおいて、安定版リリース前にcanaryブランチへのマージを一時的に制限する必要がある。コードフリーズ期間中は、リリースに影響を与える変更がcanaryブランチに入ることを防止する。本バッチは、GitHub Branch Protection APIを使用して、ブランチ保護ルールに「Potentially publish release」チェックを追加・削除することでコードフリーズを制御する。

**バッチの実行タイミング**：安定版リリースの準備期間に手動で実行される。コードフリーズ開始時に `--type enable` で実行し、リリース完了後に `--type disable` で解除する。

**主要な処理内容**：
1. 環境変数 `CODE_FREEZE_TOKEN` からGitHub認証トークンを取得する
2. CLIオプション `--type` で enable（有効化）または disable（無効化）を指定する
3. GitHub Branch Protection APIから現在のcanaryブランチ保護ルールを取得する
4. コードフリーズルール（`Potentially publish release` チェック, app_id: 15368）の存在を確認する
5. enable時：ルールが未追加であれば `required_status_checks.contexts` に追加し、APIで更新する
6. disable時：ルールが存在すれば `required_status_checks.contexts` から削除し、APIで更新する

**前後の処理との関連**：本バッチはリリースプロセスの一環として使用される。start-release（No.5）の前にenableし、publish-release（No.6）完了後にdisableする運用が想定される。

**影響範囲**：GitHubリポジトリ `vercel/next.js` のcanaryブランチのブランチ保護ルールに影響する。コードフリーズ有効時は、指定のステータスチェックが通過しない限りcanaryブランチへのマージがブロックされる。

## バッチ種別

データ連携（GitHub API連携）

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時 |
| 実行時刻 | 指定なし |
| 実行曜日 | 指定なし |
| 実行日 | 指定なし |
| トリガー | 手動（リリース準備時） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| CODE_FREEZE_TOKEN | 環境変数 `CODE_FREEZE_TOKEN` にGitHub認証トークンが設定されていること |
| ブランチ保護設定済み | canaryブランチにブランチ保護ルールが既に設定されていること |
| ネットワーク接続 | GitHub APIにアクセス可能であること |

### 実行可否判定

`CODE_FREEZE_TOKEN` 環境変数が未設定の場合、即座にエラーで終了する。`--type` オプションが `enable` または `disable` 以外の場合もエラーとなる。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| --type | string | Yes | なし | `enable` または `disable` を指定。コードフリーズの有効化/無効化を制御する |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| GitHub Branch Protection API | JSON | canaryブランチの現在の保護ルールを取得 |
| 環境変数 CODE_FREEZE_TOKEN | string | GitHub API認証用Bearer トークン |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| GitHub Branch Protection API | JSON | 更新されたブランチ保護ルール |
| 標準出力 | テキスト | 処理結果メッセージ |

### 出力ファイル仕様

ファイル出力はなし。GitHub APIへの更新リクエストと標準出力のみ。

## 処理フロー

### 処理シーケンス

```
1. 環境変数チェック
   └─ CODE_FREEZE_TOKENの存在確認。未設定時はエラー終了
2. CLIオプション解析
   └─ --typeオプションの値をenable/disableで検証
3. 現在のブランチ保護ルール取得
   └─ GitHub API GET /repos/vercel/next.js/branches/canary/protection
4. コードフリーズルールの存在確認
   └─ required_status_checks.contextsに"Potentially publish release"があるか
5. ルールの更新（enable時）
   └─ ルールが未追加の場合、contextsに追加してPUTで更新
6. ルールの更新（disable時）
   └─ ルールが存在する場合、contextsからフィルタ除去してPUTで更新
7. 結果出力
   └─ "Enabled code freeze" / "Disabled code freeze" / "Already enabled" / "Already disabled"
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{CODE_FREEZE_TOKEN設定済み?}
    B -->|No| C[エラー終了]
    B -->|Yes| D[--typeオプション解析]
    D --> E{type = enable or disable?}
    E -->|その他| F[エラー終了]
    E -->|有効値| G[現在のブランチ保護ルール取得]
    G --> H{コードフリーズルール存在?}
    H -->|enable & 未追加| I[ルールを追加してAPI更新]
    H -->|enable & 追加済み| J[Already enabled出力]
    H -->|disable & 存在| K[ルールを削除してAPI更新]
    H -->|disable & 非存在| L[Already disabled出力]
    I --> M[Enabled code freeze出力]
    K --> N[Disabled code freeze出力]
    J --> O[バッチ終了]
    L --> O
    M --> O
    N --> O
```

## データベース操作仕様

### 操作別データベース影響一覧

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| なし | - | - | データベース操作は行わない（GitHub APIのみ） |

### テーブル別操作詳細

本バッチはデータベース操作を行わない。GitHub APIを介してブランチ保護ルールを更新する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 環境変数エラー | CODE_FREEZE_TOKENが未設定 | 環境変数を設定して再実行 |
| - | パラメータエラー | --typeがenable/disable以外 | 正しいオプションを指定 |
| HTTP 4xx/5xx | APIエラー | GitHub APIがエラーレスポンスを返した | トークンの権限確認、API状態確認後に再実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（自動リトライ未実装） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

### 障害時対応

APIエラーの場合、レスポンスステータスコードとエラーメッセージがコンソールに出力される。トークンの有効性とスコープ（admin:repo権限必要）を確認し、再実行する。冪等性があるため、再実行で副作用は発生しない。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | なし（GitHub API単一リクエスト） |
| コミットタイミング | API PUT リクエスト送信時 |
| ロールバック条件 | APIリクエスト失敗時は変更なし |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | APIリクエスト2回（GET + PUT） |
| 目標処理時間 | 10秒以内 |
| メモリ使用量上限 | 128MB |

## 排他制御

同時実行は推奨しない。同時にenable/disableが実行されるとレースコンディションが発生する可能性がある。運用上は1人のリリース担当者が順次実行する。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| ルール情報 | ルール取得後 | 現在のブランチ保護ルールのJSON |
| 結果ログ | 処理完了時 | "Enabled code freeze" / "Disabled code freeze" / "Already enabled" / "Already disabled" |
| エラーログ | エラー発生時 | エラー内容とステータスコード |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| APIエラー | 1件 | コンソール |
| 処理時間 | 30秒 | コンソール |

## 備考

- 実行スクリプト: `scripts/code-freeze.js`
- コードフリーズルールのcontext名: `Potentially publish release`（app_id: 15368）
- GitHub API バージョン: `2022-11-28`
- ブランチ保護の更新にはリポジトリのadmin権限を持つトークンが必要
