# バッチ設計書 26-check-error-codes

## 概要

本ドキュメントは、Next.jsパッケージにおけるエラーコードの整合性検証バッチ「check-error-codes」の設計を記述する。`.errors`ディレクトリ内の新規エラーファイルを検出し、`errors.json`に統合した後、新しいエラーコードが存在する場合はエラー終了する。

### 本バッチの処理概要

check-error-codesバッチは、`packages/next/.errors`ディレクトリに新しいエラーメッセージファイルが存在するかを検査し、存在する場合はそれらを`errors.json`に統合する。統合後、新しいエラーコードが追加された場合はexit 1で終了し、開発者にerrors.jsonの変更をコミットするよう促す。

**業務上の目的・背景**：Next.jsはビルド時にSWCプラグインによりエラーメッセージを数値コードに変換する。新しいエラーメッセージが追加された場合、対応するエラーコードを`errors.json`に登録する必要がある。このバッチにより、未登録のエラーコードを検出し、適切にerrors.jsonが更新されていることを保証する。

**バッチの実行タイミング**：ビルドプロセスの一部として`taskr build`タスク内で`compile`の後に実行される。CI/CDパイプラインでも実行される。

**主要な処理内容**：
1. `.errors`ディレクトリの存在チェック（存在しなければ正常終了）
2. 既存の`errors.json`を読み込み
3. `.errors`ディレクトリ内の各ファイルを処理し、新しいエラーメッセージを検出
4. 重複チェック（既存メッセージと同一であれば追加しない）
5. 新しいメッセージに次番のエラーコードを割り当て
6. 更新された`errors.json`を書き出し
7. `.errors`ディレクトリを削除
8. 新しいエラーが追加された場合はexit 1で終了

**前後の処理との関連**：update-error-codes（No.27）バッチの一部として呼び出される。ビルド（compile）タスクがSWCプラグインにより`.errors`ディレクトリにエラーファイルを生成した後に実行される。merge-errors-json（No.28）はGitマージ時のerrors.jsonのコンフリクト解消に使用される。

**影響範囲**：`packages/next/errors.json`および`packages/next/.errors`ディレクトリ。ビルドプロセスの成否に影響する。

## バッチ種別

コード品質検証 / データ統合

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（ビルド時） |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | ビルドプロセス（taskr build）、手動実行（`pnpm check-error-codes`） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js >= 20.9.0 | package.jsonのenginesで指定 |
| packages/next/errors.json | エラーコード定義ファイル（存在しない場合は空オブジェクトから開始） |

### 実行可否判定

`.errors`ディレクトリが存在しない場合はexit 0で即座に正常終了する。ディレクトリが存在する場合のみ処理を実行する。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| なし | - | - | - | コマンドライン引数は不要 |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| packages/next/errors.json | JSON | 既存のエラーコード定義（キー: コード番号、値: エラーメッセージ） |
| packages/next/.errors/*.json | JSON | SWCプラグインにより生成された新しいエラーファイル（各ファイルにerrorMessageフィールド） |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| packages/next/errors.json | JSON | 更新されたエラーコード定義 |
| 終了コード | 数値 | 0: 新規エラーなし、1: 新規エラーあり（errors.jsonを更新済み） |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | errors.json |
| 出力先 | packages/next/ |
| 文字コード | UTF-8 |
| 区切り文字 | 該当なし（JSON形式、2スペースインデント） |

## 処理フロー

### 処理シーケンス

```
1. .errorsディレクトリ存在チェック
   └─ 存在しなければexit 0で正常終了
2. errors.json読み込み
   └─ 存在しない場合は空オブジェクト{}で初期化
3. 次番コード番号の算出
   └─ 既存の最大コード番号 + 1
4. .errorsディレクトリ内ファイルの走査
   └─ 各JSONファイルのerrorMessageを取得
5. 重複チェック
   └─ 既存のエラーメッセージと照合
6. 新規エラーメッセージの追加
   └─ 重複しないメッセージに次番コードを割り当て
7. errors.jsonの書き出し
   └─ JSON.stringify(existingErrors, null, 2) + os.EOL
8. .errorsディレクトリの削除
   └─ fs.rm(errorsDir, { recursive: true, force: true })
9. exit 1で終了
   └─ 新しいエラーが追加されたことを示すために常にexit 1
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{.errorsディレクトリ存在?}
    B -->|なし| C[exit 0]
    B -->|あり| D[errors.json読み込み]
    D --> E[次番コード番号算出]
    E --> F[.errors内ファイル走査]
    F --> G{各ファイルのメッセージが既存?}
    G -->|既存| H[スキップ]
    G -->|新規| I[次番コードで追加]
    H --> J[errors.json書き出し]
    I --> J
    J --> K[.errorsディレクトリ削除]
    K --> L[exit 1]
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作は行わない。ファイルシステム上のJSONファイルを操作する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 新規エラー検出 | .errorsディレクトリに新しいエラーファイルが存在する | errors.jsonの変更をコミットする |
| 例外 | JSONパースエラー | .errorsディレクトリ内のファイルが不正なJSON | エラーファイルの内容を確認・修正 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | CI環境ではリトライなし（exit 1で失敗） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

CI環境でexit 1が発生した場合は、ローカルで`pnpm build`または`pnpm update-error-codes`を実行し、生成されたerrors.jsonの変更をコミットする。taskr経由で実行された場合は、CI環境ではエラーメッセージを表示して終了し、ローカル環境ではcompileタスクを再実行する。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 全体（errors.json書き込みと.errorsディレクトリ削除は一連の処理） |
| コミットタイミング | 全ファイル処理完了後 |
| ロールバック条件 | 該当なし（errors.jsonは上書き） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | .errors内のファイル数（通常は少数） |
| 目標処理時間 | 1秒以内 |
| メモリ使用量上限 | 明示的な上限なし |

## 排他制御

ビルドプロセス中に実行されるため、同時に複数のビルドが実行されないことを前提とする。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| エラーログ | CI環境での失敗時 | "check_error_codes FAILED: There are new errors..." メッセージ |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| CI実行結果 | 失敗時 | GitHub PR ステータスチェック |

## 備考

- スクリプトパス: `packages/next/check-error-codes.js`
- package.jsonでの呼び出し: `"check-error-codes": "node packages/next/check-error-codes.js"`
- .errorsディレクトリはSWCプラグインによってコンパイル時に生成される
- exit 1は「エラー」ではなく「新しいエラーコードが追加されたのでコミットが必要」を示す
- taskr内のcheck_error_codesタスクでは、CI環境では失敗を報告し、ローカル環境ではcompileタスクを再実行する
