# バッチ設計書 29-sync-react

## 概要

本ドキュメントは、Next.jsモノレポにおけるReactおよび関連パッケージの同期バッチ「sync-react」の設計を記述する。Reactのcanaryバージョンをnpmレジストリから取得し、モノレポ全体のpackage.json、ピア依存関係、およびベンダーファイルを更新する。オプションでGitHub Pull Requestの自動作成も行う。

### 本バッチの処理概要

sync-reactバッチは、ReactおよびReact DOM、scheduler等の依存パッケージを最新のcanary（またはexperimental）バージョンに同期するためのスクリプトである。npmレジストリから最新バージョンを取得し、モノレポのpackage.json（devDependencies、pnpm overrides）を更新し、関連ファイルのピア依存関係バージョン文字列を置換する。必要に応じてpnpm installとncc-compiledを実行してベンダーファイルを再生成し、変更をGitHub Pull Requestとして自動作成する。

**業務上の目的・背景**：Next.jsはReactの最新canaryバージョンに追従して開発されている。Reactチームが新しいcanaryバージョンをリリースした際に、Next.js側の依存関係を迅速に更新する必要がある。手動での更新は複数ファイルにまたがり煩雑であるため、この自動化バッチにより一貫性のある更新とPR作成を実現する。

**バッチの実行タイミング**：GitHub Actionsワークフローによる定期実行、または開発者による手動実行。

**主要な処理内容**：
1. npmレジストリから最新のReact canaryバージョンを取得
2. experimentalチャネルのReactバージョンを更新（package.json devDependencies/pnpm overrides）
3. canaryチャネルのReactバージョンを更新
4. schedulerバージョンをReact DOMの依存関係から取得して更新
5. ピア依存関係バージョン参照ファイルを更新（run-tests.js、create-next-app templates、test/lib等）
6. ライブラリマニフェスト（packages/next、packages/third-parties）のpeerDependenciesを更新
7. アプリケーションマニフェスト（examples、test用）のdependenciesを更新
8. pnpm installを実行してロックファイルを更新
9. ncc-compiledを実行してベンダーReactファイルを再生成
10. GitHub APIを使用してPull Requestを作成（--create-pullオプション時）

**前後の処理との関連**：automated-update-workflow（No.31）の具体的な実装の一つ。ビルドバッチ（next build / No.1）の前提となるReactバージョンに影響する。

**影響範囲**：モノレポ全体のpackage.json、pnpm-lock.yaml、packages/next/src/compiled/配下のベンダーReactファイル、テスト設定ファイル、create-next-appテンプレート。

## バッチ種別

依存関係更新 / データ連携

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（Reactの新バージョンリリース時） |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | GitHub Actions / 手動実行（`pnpm sync-react`） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js >= 20.9.0 | package.jsonのenginesで指定 |
| pnpm 9.6.0 | パッケージマネージャのバージョン |
| npm CLIアクセス | npmレジストリへのアクセスが必要 |
| GITHUB_TOKEN | --create-pullオプション使用時に必要（環境変数） |
| Gitリポジトリ | --create-pullオプション使用時にGitリポジトリが必要 |

### 実行可否判定

npmレジストリにアクセス可能であれば実行可能。--create-pullオプション使用時はGITHUB_TOKENと--actorの指定が必須。既に同名のブランチが存在する場合はスキップする。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| --version | string | No | null（最新canaryを自動取得） | 同期するReactバージョン |
| --create-pull | boolean | No | false | GitHub PRを作成するか |
| --actor | string | No | - | PR作成時のGitHubユーザー名（--create-pull時必須） |
| --commit | boolean | No | false | 中間ステップごとにGitコミットを作成するか |
| --no-install | boolean | No | false（install=true） | 依存関係インストールをスキップするか |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| npmレジストリ（react） | API | Reactの利用可能バージョン一覧 |
| npmレジストリ（react-dom） | API | React DOMの依存関係情報（scheduler version） |
| GitHub API（commits） | API | Reactリポジトリのコミットログ（変更履歴生成用） |
| package.json | JSON | モノレポルートのパッケージ定義 |
| 各種マニフェストファイル | JSON | ライブラリ・アプリケーションのpackage.json |
| ピア依存関係参照ファイル | TypeScript/JavaScript | nextjsReactPeerVersionを含むファイル |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| package.json | JSON | 更新されたdevDependencies/pnpm overrides |
| pnpm-lock.yaml | YAML | 更新されたロックファイル |
| packages/next/src/compiled/ | JavaScript | 再生成されたベンダーReactファイル |
| ピア依存関係参照ファイル | TypeScript/JavaScript | 更新されたバージョン文字列 |
| 各種マニフェスト | JSON | 更新されたpeerDependencies/dependencies |
| GitHub Pull Request | - | --create-pullオプション時に作成 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | 複数ファイル（package.json, ピア依存関係ファイル等） |
| 出力先 | モノレポ各所 |
| 文字コード | UTF-8 |
| 区切り文字 | 該当なし（JSON形式、2スペースインデント + 末尾改行） |

## 処理フロー

### 処理シーケンス

```
1. バージョン決定
   └─ --versionが指定されていなければnpmから最新canaryバージョンを取得
2. バージョン情報の抽出
   └─ semver-releaseLabel-sha-dateStringのフォーマットからメタ情報を抽出
3. ブランチ存在チェック（--create-pull時）
   └─ update/react/{version}ブランチが既に存在する場合はスキップ
4. experimentalチャネルのReact更新
   └─ devDependenciesとpnpm overridesのバージョン文字列を置換
5. canaryチャネルのReact更新
   └─ 同様にバージョン文字列を置換
6. ピア依存関係ファイルの更新
   └─ nextjsReactPeerVersionの文字列を更新
7. アプリケーションマニフェストの更新
   └─ examples, testのpackage.jsonのreact/react-domバージョンを更新
8. ライブラリマニフェストの更新
   └─ packages/next, packages/third-partiesのpeerDependenciesを更新
9. pnpm install（--no-installでなければ）
   └─ --no-frozen-lockfileでロックファイルを更新
10. ncc-compiled（--no-installでなければ）
    └─ ベンダーReactファイルを再生成
11. 変更履歴の取得
    └─ GitHub APIからbaseとnewのsha間のコミットログを取得
12. Pull Request作成（--create-pull時）
    └─ ブランチ作成、プッシュ、PR作成、レビュアー/ラベル設定
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B{--version指定あり?}
    B -->|なし| C[npmから最新canary取得]
    B -->|あり| D[指定バージョンを使用]
    C --> E[バージョン情報抽出]
    D --> E
    E --> F{--create-pull?}
    F -->|はい| G{ブランチ既存?}
    G -->|はい| H[スキップ]
    G -->|なし| I[experimental更新]
    F -->|いいえ| I
    I --> J[canary更新]
    J --> K[ピア依存関係ファイル更新]
    K --> L[マニフェスト更新]
    L --> M{--no-install?}
    M -->|はい| N[インストールスキップ]
    M -->|いいえ| O[pnpm install]
    O --> P[ncc-compiled]
    P --> Q[変更履歴取得]
    N --> Q
    Q --> R{--create-pull?}
    R -->|はい| S[ブランチ作成・プッシュ・PR作成]
    R -->|いいえ| T[結果出力]
    S --> T
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作は行わない。外部API（npm registry、GitHub API）を使用する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | バージョンフォーマットエラー | 指定バージョンが期待フォーマットに合致しない | 正しいcanaryバージョンを指定 |
| 1 | アクター未指定 | --create-pullで--actorが未指定 | --actorオプションを追加 |
| 1 | トークン未設定 | --create-pullでGITHUB_TOKEN未設定 | GITHUB_TOKEN環境変数を設定 |
| 1 | npmアクセスエラー | npmレジストリへのアクセス失敗 | ネットワーク接続を確認 |
| 1 | インストールエラー | pnpm installの失敗 | 依存関係の競合を手動解決 |
| 1 | ncc-compiledエラー | ベンダーファイル再生成の失敗 | エラーメッセージに従い対処 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

失敗した場合は、エラーメッセージを確認し、手動で該当ステップを修正する。--no-installオプションで依存関係インストールをスキップして実行し、問題箇所を特定することも可能。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 全体（複数ファイルの一括更新） |
| コミットタイミング | --commitオプション時は各ステップ完了時、それ以外はPR作成時に一括 |
| ロールバック条件 | git resetで変更を取り消し可能 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 更新対象ファイル約10〜20件 |
| 目標処理時間 | 数分（pnpm installとncc-compiledに依存） |
| メモリ使用量上限 | 明示的な上限なし |

## 排他制御

同一ブランチ名（`update/react/{version}`）が存在する場合はスキップすることで、重複実行を防止する。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| バージョン情報ログ | 処理開始時 | "Updating react@{channel} to {version}..." |
| 更新不要ログ | バージョン同一時 | "Already up to date." |
| インストールログ | install実行中 | "Installing dependencies..." |
| ビルドログ | ncc-compiled実行中 | "Building vendored React files..." |
| PR作成ログ | PR作成時 | "Created pull request {url}" |
| 変更履歴ログ | 処理完了時 | Reactリポジトリのコミットログ |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| GitHub Actions実行結果 | 失敗時 | GitHub通知 |

## 備考

- スクリプトパス: `scripts/sync-react.js`
- package.jsonでの呼び出し: `"sync-react": "node ./scripts/sync-react.js"`
- デフォルトのlatestチャネルは`canary`
- activePagesRouterReactは`^19.0.0`に固定（nullに設定するとPages RouterもApp Routerと同バージョンに同期）
- PRラベル: `type: react-sync`
- PRレビュアー: `eps1lon`
- Reactバージョンフォーマット: `{semver}-{releaseLabel}-{sha}-{dateString}`
- schedulerバージョンはreact-domの依存関係から自動取得される
- pnpm installは`--no-frozen-lockfile`で実行される（ロックファイル更新が前提のため）
- 更新対象ファイル群:
  - ピア依存関係参照: run-tests.js, packages/create-next-app/templates/index.ts, test/lib/next-modes/base.ts
  - ライブラリマニフェスト: packages/third-parties/package.json, packages/next/package.json
  - アプリマニフェスト: examples/reproduction-template/package.json, test/.stats-app/package.json, テスト用package.json
