# バッチ設計書 30-update-google-fonts

## 概要

本ドキュメントは、Next.jsプロジェクトにおけるGoogle Fontsメタデータ更新バッチ「update-google-fonts」の設計を記述する。Google Fonts APIから最新のフォントメタデータを取得し、next/fontパッケージの型定義ファイルとフォントデータJSONを再生成する。

### 本バッチの処理概要

update-google-fontsバッチは、Google Fonts APIのメタデータエンドポイントからフォントファミリー情報を取得し、各フォントのウェイト、スタイル、サブセット、バリアブルフォント軸情報を解析して、TypeScript型定義ファイル（index.ts）とフォントデータJSON（font-data.json）を自動生成する。

**業務上の目的・背景**：`next/font/google`モジュールは、Google Fontsを型安全に利用するためのAPIを提供している。Google Fontsに新しいフォントが追加された場合や既存フォントの情報が変更された場合に、next/fontパッケージの型定義とデータを最新の状態に更新する必要がある。手動での更新は非常に煩雑であり（数千のフォントファミリーが対象）、このバッチにより自動化する。

**バッチの実行タイミング**：定期的な手動実行、またはGoogle Fontsの更新に合わせた実行。

**主要な処理内容**：
1. Google Fonts APIからフォントメタデータを取得
2. 各フォントファミリーについてウェイト、スタイル、サブセット、バリアブルフォント軸を解析
3. 特定サブセット（日本語、韓国語、中国語等）を除外
4. TypeScript型定義（各フォントの関数シグネチャ）を生成
5. フォントデータJSON（weights、styles、axes、subsets）を生成
6. packages/font/src/google/にindex.tsとfont-data.jsonを書き出し

**前後の処理との関連**：生成されたファイルは、packages/fontパッケージのビルドに使用される。automated-update-workflow（No.31）から呼び出される可能性がある。

**影響範囲**：`packages/font/src/google/index.ts`（型定義）、`packages/font/src/google/font-data.json`（フォントデータ）。next/font/googleモジュールのユーザー向けAPIに影響する。

## バッチ種別

データ連携 / コード生成

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（Google Fonts更新時） |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動実行（`pnpm update-google-fonts`） |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js >= 20.9.0 | package.jsonのenginesで指定 |
| インターネットアクセス | Google Fonts APIへのアクセスが必要 |
| node-fetchパッケージ | devDependenciesに含まれていること |

### 実行可否判定

Google Fonts API（https://fonts.google.com/metadata/fonts）にアクセス可能であれば実行可能。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| なし | - | - | - | コマンドライン引数は不要 |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| https://fonts.google.com/metadata/fonts | JSON API | Google Fontsのフォントメタデータ（familyMetadataList） |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| packages/font/src/google/index.ts | TypeScript | フォント関数の型定義 |
| packages/font/src/google/font-data.json | JSON | フォントデータ（weights, styles, axes, subsets） |

### 出力ファイル仕様

#### index.ts

| 項目 | 内容 |
|-----|------|
| ファイル名 | index.ts |
| 出力先 | packages/font/src/google/ |
| 文字コード | UTF-8 |
| 内容 | 各フォントファミリーの型付き関数宣言（export declare function） |

#### font-data.json

| 項目 | 内容 |
|-----|------|
| ファイル名 | font-data.json |
| 出力先 | packages/font/src/google/ |
| 文字コード | UTF-8 |
| 内容 | フォントファミリー名をキー、weights/styles/axes/subsetsをバリューとするJSONオブジェクト（2スペースインデント） |

## 処理フロー

### 処理シーケンス

```
1. Google Fonts APIからメタデータ取得
   └─ https://fonts.google.com/metadata/fonts にfetchリクエスト
   └─ familyMetadataListを取得
2. 各フォントファミリーの解析
   └─ サブセットフィルタリング（日本語、韓国語、中国語等を除外）
   └─ バリアント情報からウェイトとスタイルを抽出
   └─ バリアブルフォント判定（axes.length > 0）
   └─ バリアブルフォントの場合は"variable"ウェイトを追加
   └─ 非weightの軸をoptionalAxesとして抽出
3. TypeScript型定義の生成
   └─ 各フォントの関数シグネチャを文字列として構築
   └─ フォント名の先頭が数字の場合はアンダースコアプレフィックスを付与
   └─ スペースをアンダースコアに変換
   └─ weight, style, display, variable, preload, fallback, adjustFontFallback, subsets, axesのオプション定義
4. フォントデータJSONの生成
   └─ weights, styles, axes, subsetsを収集
5. ファイル書き出し（Promise.allで並列）
   └─ index.tsを書き出し
   └─ font-data.jsonを書き出し
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[Google Fonts API呼び出し]
    B --> C[familyMetadataList取得]
    C --> D[各フォントファミリーを走査]
    D --> E[サブセットフィルタリング]
    E --> F[ウェイト・スタイル抽出]
    F --> G{バリアブルフォント?}
    G -->|はい| H[variableウェイト追加 + 軸情報抽出]
    G -->|いいえ| I[固定ウェイトのみ]
    H --> J[型定義文字列生成]
    I --> J
    J --> K[fontDataオブジェクト構築]
    K --> L{全フォント処理完了?}
    L -->|いいえ| D
    L -->|はい| M[Promise.all: index.ts + font-data.json書き出し]
    M --> N[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作は行わない。外部API（Google Fonts）を使用する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 非0終了 | APIアクセスエラー | Google Fonts APIへのアクセス失敗 | ネットワーク接続を確認して再実行 |
| 非0終了 | JSONパースエラー | APIレスポンスが不正 | APIのレスポンス形式を確認 |
| 非0終了 | ファイル書き込みエラー | 出力先ディレクトリの権限不足等 | ファイルシステムの権限を確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

APIアクセスエラーの場合は、ネットワーク環境を確認して再実行する。APIのレスポンス形式が変更された場合は、スクリプトの更新が必要。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 2ファイル（index.ts, font-data.json）の同時書き込み |
| コミットタイミング | Promise.allによる並列書き込み完了時 |
| ロールバック条件 | git resetで変更を取り消し可能 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | Google Fontsの全フォントファミリー（約1,500ファミリー以上） |
| 目標処理時間 | 数秒〜数十秒（API応答時間に依存） |
| メモリ使用量上限 | 明示的な上限なし |

## 排他制御

同時実行は推奨されない（同じファイルへの並列書き込みが発生するため）。通常は単発実行される。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 該当なし | - | 標準的なログ出力は実装されていない |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 該当なし | - | - |

## 備考

- スクリプトパス: `scripts/update-google-fonts.js`
- package.jsonでの呼び出し: `"update-google-fonts": "node ./scripts/update-google-fonts.js"`
- 除外サブセット: menu, japanese, korean, chinese-simplified, chinese-hongkong, chinese-traditional
- フォント名の変換ルール: 先頭が数字の場合はアンダースコアプレフィックス（例: `_123Font`）、スペースはアンダースコアに変換（例: `Open_Sans`）
- バリアブルフォント判定: axes配列の長さが0より大きい場合
- バリアブルフォントのオプションパラメータはweight?（オプショナル）となる
- 型定義はNextFont / NextFontWithVariable型を返す
- 自動生成コメント: "This is an autogenerated file by scripts/update-google-fonts.js"
